<?php

namespace Website\Model;

class Generator
{
	const CONTROLLER_SECTION_DOCUMENT_ACTIONS = 'DOCUMENT ROUTED ACTIONS';
	const CONTROLLER_SECTION_FORM_ACTIONS = 'FORM HANDLERS';

	public function createClassDefinition($name, $url = false, $localized = false, $icon = null)
	{
		$classDefinition = new \Pimcore\Model\Object\ClassDefinition();
		$classDefinition->setName($name);
		$classDefinition->setCreationDate(time());
		$classDefinition->setModificationDate(time());
		$classDefinition->setUserOwner(2);
		$classDefinition->setUserModification(2);
		//$classDefinition->setUseTraits(false);
		$classDefinition->setAllowInherit(false);
		$classDefinition->setAllowVariants(false);
		$classDefinition->setShowVariants(false);
		if ($icon) {
			$classDefinition->setIcon((string)$icon);
		}
		$classDefinition->setPreviewUrl('');
		$classDefinition->setPropertyVisibility([
			'grid' => [
				'id' => true,
				'path' => false,
				'published' => true,
				'modificationDate' => false,
				'creationDate' => true
			],
			'search' => [
				'id' => true,
				'path' => true,
				'published' => true,
				'modificationDate' => true,
				'creationDate' => true
			]
		]);

		$fieldDefinitions = [];
		$nameInput = new \Pimcore\Model\Object\ClassDefinition\Data\Input();
		$nameInput->setWidth(($localized) ? 550 : 600);
		$nameInput->setName('name');
		$nameInput->setTitle('Name');
		$nameInput->setMandatory(false);
		$nameInput->setVisibleGridView(true);
		$nameInput->setVisibleSearch(true);
		$urlInput = null;
		if ($url) {
			$urlInput = new \Pimcore\Model\Object\ClassDefinition\Data\CalculatedValue();
			$urlInput->setCalculatorClass('\Website\Model\ValueCalculator');
			$urlInput->setWidth(($localized) ? 550 : 600);
			$urlInput->setName('url');
			$urlInput->setTitle('Url');
			$urlInput->setMandatory(false);
			$urlInput->setNoteditable(true);
			$urlInput->setVisibleGridView(true);
			$urlInput->setVisibleSearch(false);
		}
		if ($localized) {
			$lf = new \Pimcore\Model\Object\ClassDefinition\Data\Localizedfields();
			$lf->setName('localizedfields');
			$lf->setWidth(700);
			$lf->addChild($nameInput);
			if ($urlInput) {
				$lf->addChild($urlInput);
			}
			$fieldDefinitions['localizedfields'] = $lf;
		} else {
			$fieldDefinitions['name'] = $nameInput;
			if ($urlInput) {
				$fieldDefinitions['url'] = $urlInput;
			}
		}
		$classDefinition->setFieldDefinitions($fieldDefinitions);

		$panel = new \Pimcore\Model\Object\ClassDefinition\Layout\Panel();
		$panel->setLabelWidth(100);
		$panel->setName($name);
		$panel->setChilds(array_values($fieldDefinitions));
		$layout = new \Pimcore\Model\Object\ClassDefinition\Layout\Panel();
		$layout->setLabelWidth(100);
		$layout->setName('pimcore_root');
		$layout->addChild($panel);

		$classDefinition->setLayoutDefinitions($layout);

		try {
			$classDefinition->save();
			return $classDefinition->getId();
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: %s\n", $e->getMessage()));
			exit;
		}
	}

	public function deleteClassDefinition($name)
	{
		$classDefinition = \Pimcore\Model\Object\ClassDefinition::getByName($name);
		if (!$classDefinition) {
			fputs(STDOUT, sprintf("Error: class [%s] not found\n", $name));
			exit;
		}

		try {
			$classDefinition->delete();
			return true;
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: %s\n", $e->getMessage()));
			exit;
		}
	}

	public function createClassMapping($classNameOrId, $mapList = true)
	{
		$className = $this->getClassName($classNameOrId);
		$configPath = \Pimcore\Config::locateConfigFile("classmap.php");
		$classMap = include($configPath);

		$parent = sprintf('Object\\%s', $className);
		$parentList = sprintf('Object\\%s\\Listing', $className);
		$child = sprintf('Website\\Model\\%s', $className);
		$childList = sprintf('Website\\Model\\%s\\Listing', $className);
		if (!isset($classMap[$parent])) {
			$classMap[$parent] = $child;
		}
		$this->createMappedClass($className);
		if ($mapList) {
			if (!isset($classMap[$parentList])) {
				$classMap[$parentList] = $childList;
			}
			$this->createMappedClassList($className);
		}

		return $this->writePhpFile($configPath, $classMap, true);
	}

	public function deleteClassMapping($classNameOrId)
	{
		$className = $this->getClassName($classNameOrId);
		$configPath = \Pimcore\Config::locateConfigFile("classmap.php");
		$classMap = include($configPath);

		$parent = sprintf('Object\\%s', $className);
		$parentList = sprintf('Object\\%s\\Listing', $className);
		if (isset($classMap[$parent])) {
			unset($classMap[$parent]);
			$this->deleteMappedClass($className);
		}
		if (isset($classMap[$parentList])) {
			unset($classMap[$parentList]);
			$this->deleteMappedClassList($className);
		}

		return $this->writePhpFile($configPath, $classMap, true);
	}

	public function createMappedClass($className)
	{
		$template = '<?php

namespace Website\Model;

class %s extends \Pimcore\Model\Object\%s 
{

	/*								Helpers										*/

	/*								Mapped										*/

	/* 								For elastic search							*/

}
';

		$modelPath = sprintf('%s/models/%s.php', PIMCORE_WEBSITE_PATH, $className);
		try {
			if (!file_exists($modelPath)) {
				$this->writePhpFile($modelPath, vsprintf($template, [
					$className,
					$className
				]));
			}
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: file [%s] not writable\n", $modelPath));
			exit;
		}

		return true;
	}

	public function createMappedClassList($className)
	{
		$template = '<?php

namespace Website\Model\%s;

class Listing extends \Pimcore\Model\Object\%s\Listing
{

}
';

		$modelPath = sprintf('%s/models/%s/Listing.php', PIMCORE_WEBSITE_PATH, $className);
		try {
			if (!file_exists($modelPath)) {
				$this->writePhpFile($modelPath, vsprintf($template, [
					$className,
					$className
				]));
			}
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: file [%s] not writable\n", $modelPath));
			exit;
		}

		return true;
	}

	public function deleteMappedClass($className)
	{
		$modelPath = PIMCORE_WEBSITE_PATH.'/models/'.$className.'.php';
		try {
			if (file_exists($modelPath)) {
				unlink($modelPath);
			}
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: could not delete file [%s]\n", $modelPath));
			exit;
		}

		return true;
	}

	public function deleteMappedClassList($className)
	{
		$modelsDir = PIMCORE_WEBSITE_PATH.'/models';
		$modelPath = sprintf('%s/%s/Listing.php', $modelsDir, $className);
		try {
			if (file_exists($modelPath)) {
				unlink($modelPath);
			}
			if (file_exists($modelsDir.'/'.$className)) {
				rmdir($modelsDir.'/'.$className);
			}
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: could not delete file [%s] or directory [%s]\n", $modelPath, $modelsDir.'/'.$className));
			exit;
		}

		return true;
	}

	public function createAction($name, $controller, $form = null, $view = true, $forceExistenceError = true)
	{
		$tokens = $this->loadControllerTokens($controller);
		$actionExists = $this->checkIfTokenExists($tokens, lcfirst($name).'Action', $forceExistenceError);

		if (!$actionExists) {
			$sectionIndex = $this->findControllerSectionTokenIndex($tokens, self::CONTROLLER_SECTION_DOCUMENT_ACTIONS);
			$actionContent = $this->createControllerAction($name, $form);
			array_splice($tokens, $sectionIndex, 0, $actionContent);
		}

		$scriptPath = vsprintf('%s/views/scripts/%s/%s.php', [
			PIMCORE_WEBSITE_PATH,
			$this->normalizeScriptFileName($controller),
			$this->normalizeScriptFileName($name)
		]);
		if ($view && !file_exists($scriptPath)) {
			$scriptContents = ($view === true)
				? "\n\n<?\$this->layout()->setLayout('standard');?>\n"
				: file_get_contents(PIMCORE_DOCUMENT_ROOT . $view);
			try {
				$this->writePhpFile($scriptPath, $scriptContents);
			} catch (Exception $e) {
				fputs(STDOUT, sprintf("Error: view script [%s] can not be saved\n"));
				exit;
			}
		}

		if (!$actionExists) {
			return $this->saveControllerTokens($controller, $tokens);
		} else {
			return true;
		}
	}

	public function createFormHandler($name, $controller)
	{
		$tokens = $this->loadControllerTokens($controller);
		$handlerName = 'handle' . $name . 'Form';
		$this->checkIfTokenExists($tokens, $handlerName);

		$sectionIndex = $this->findControllerSectionTokenIndex($tokens, self::CONTROLLER_SECTION_FORM_ACTIONS);
		$actionContent = $this->createControllerFormHandler($name);

		array_splice($tokens, $sectionIndex, 0, $actionContent);

		$this->saveControllerTokens($controller, $tokens);

		return $handlerName;
	}

	public function createControllerAction($name, $form = null)
	{
		$template = 'public function %sAction()
	{
		$this->enableLayout();%s
	}

	';
		$formHandlerTemplate = '
		$form = new \Website\Form\%sForm(); 

		if ($this->getRequest()->isPost()) {
			if ($this->handle%sForm($form)) {
				$this->gotoUrl($this->view->docUrl(\'/\'.$this->defaultLanguage));
			}
		}

		$this->view->%sForm = $form;';

		return vsprintf($template, [
			lcfirst($name),
			($form) ? vsprintf($formHandlerTemplate, [$form, $form, lcfirst($form)]) : ''
		]);
	}

	public function createControllerFormHandler($name)
	{
		$template = 'private function handle%sForm(\Zend_Form &$form)
	{
		if ($form->isValid($this->getRequest()->getPost())) {
			$data = $form->getValues();

			return true;
		} else {
			$this->addErrorMsg($this->translate(\'msg_form_invalid\'));
		}
		return false;
	}

	';

		return vsprintf($template, [$name]);
	}

	public function createDocumentPage($title = null, $controller = null, $action = null, $parent = null, $view = null, $master = null)
	{
		# check parent or load the first under "home"
		$parentId = null;
		if (!$parent) {
			$parent = current(\Pimcore\Model\Document::getById(1)->getChilds());
			if ($parent) {
				$parentId = $parent->getId();
			}
		} else {
			$tmp = null;
			if (intval($parent)) {
				$tmp = \Pimcore\Model\Document::getById(intval($parent));
			} else {
				$parent = $this->normalizePathParam($parent, false);
				$tmp = \Pimcore\Model\Document::getByPath($parent);
			}
			if ($tmp) {
				$parentId = $tmp->getId();
			}
		}
		if (!$parentId) {
			fputs(STDOUT, sprintf("Error: can not load parent document [%s]\n", $parent));
			exit;
		}

		# check content document master
		$masterId = null;
		if ($master) {
			$tmp = null;
			if (intval($master)) {
				$tmp = \Pimcore\Model\Document\Page::getById(intval($master));
			} else {
				$master = $this->normalizePathParam($master, false);
				$tmp = \Pimcore\Model\Document\Page::getByPath($master);
			}
			if ($tmp) {
				$masterId = $tmp->getId();
			}
			if (!$tmp) {
				fputs(STDOUT, sprintf("Error: could not load master document [%s]\n", $master));
				exit;
			}
		}

		# check controller
		$controllerPath = PIMCORE_WEBSITE_PATH.'/controllers/'.$controller.'Controller.php';
		if (!file_exists($controllerPath)) {
			fputs(STDOUT, sprintf("Error: controller [%s] does not exists\n", $controllerPath));
			exit;
		}

		# check or try to create action
		if (!$action) {
			$action = 'TemplateSimple';
		}
		if (!$view) {
			$view = true;
		}
		$this->createAction($action, $controller, null, $view, false);

		# create document
		$document = new \Pimcore\Model\Document\Page();
		$document->setIndex(0);
		$document->setPublished(true);
		$document->setController($this->normalizeScriptFileName($controller));
		$document->setAction($this->normalizeScriptFileName($action));
		$document->setParentId($parentId);
		$document->setTitle($title);
		$document->setKey(\Website\Tool\Utils::webalize($this->normalizeScriptFileName($title)));
		$document->setProperty('navigation_name', 'text', $title, false, false);
		$document->setProperty('navigation_title', 'text', $title, false, false);
		if ($masterId) {
			$document->setContentMasterDocumentId($masterId);
		}
		try {
			$document->save();
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: could not save document [%s]\n", $e->getMessage()));
			exit;
		}

		return $document;
	}

	public function createDocumentEmail($subject, $parent = null)
	{

		# check parent or load the first under "/notifikace"
		$parentId = null;
		if (!$parent) {
			$notificationsFolderPath = '/notifikace';
			$notificationsFolder = \Pimcore\Model\Document\Folder::getByPath($notificationsFolderPath);
			if ($notificationsFolder) {
				$parentId = $notificationsFolder->getId();
			}
		} else {
			$tmp = null;
			if (intval($parent)) {
				$tmp = \Pimcore\Model\Document::getById(intval($parent));
			} else {
				$parent = $this->normalizePathParam($parent, false);
				$tmp = \Pimcore\Model\Document::getByPath($parent);
			}
			if ($tmp) {
				$parentId = $tmp->getId();
			}
		}
		if (!$parentId) {
			fputs(STDOUT, sprintf("Error: can not load document parent [%s]\n", $parent));
			exit;
		}

		$webTitle = \Pimcore\Model\Document\Page::getById(1)->getProperty('titlePostfix');
		$email = new \Pimcore\Model\Document\Email();
		$email->setParentId($parentId);
		$email->setKey(\Website\Tool\Utils::webalize($this->normalizeScriptFileName($subject)));
		$email->setPublished(true);
		$email->setController('snippet');
		$email->setAction('simple-mail');
		$email->setFrom('noreply@vm3.portadesign.cz');
		$email->setSubject(sprintf('%s - %s', $webTitle, $subject));
		$email->setIndex(0);
		try {
			$email->save();
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: could not create email [%s], reason [%s]\n", $subject, $e->getMessage()));
			exit;
		}
		$rte = \Pimcore\Model\Document\Tag::factory('wysiwyg', 'email_body', $email->getId());

		$rte->text = sprintf('<p>Dobrý den,</p><p>Text macro example: %%Text(textVariableName);</p><p>Object macro example: %%Object(objectVariableName, {"method":"getMethodName"});</p><p>%s</p>', $webTitle);
		$email->setElement('email_body', $rte);
		$email->save();

		return $email;
	}

	public function createArea($name, $icon)
	{
		# create folder
		$areaFolderPath = sprintf('%s/views/areas/%s', PIMCORE_WEBSITE_PATH, $name);
		if (file_exists($areaFolderPath)) {
			fputs(STDOUT, sprintf("Error: area [%s] already exists\n", $name));
			exit;
		}
		\Pimcore\File::mkdir($areaFolderPath);

		# create area config
		$config = $this->loadXmlConfig(null);
		$config->id = $name;
		$config->name = 'area_'.$name;
		$config->description = 'area_'.$name.'_description';
		$config->icon = ($icon) ? $icon : '/pimcore/static/img/icon/html.png';
		$config->version = '1.0';
		$this->writeXmlConfig($config, $areaFolderPath . "/area.xml");

		# enable area in extensions.php
		$extensionsPath = \Pimcore\Config::locateConfigFile("extensions.php");
		$exConfig = include($extensionsPath);
		if (!isset($exConfig['brick'][$name])) {
			$exConfig['brick'][$name] = 1;
		}
		$this->writePhpFile($extensionsPath, $exConfig, true);

		# create action.php and view.php files
		$actionTemplate = '<?php

namespace Pimcore\Model\Document\Tag\Area;

class %s extends AbstractArea
{

	public function getBrickHtmlTagOpen($brick)
	{
		return \'\';
	}

	public function getBrickHtmlTagClose($brick)
	{
		return \'\';
	}

}
';
		try {
			$this->writePhpFile($areaFolderPath.'/action.php', sprintf($actionTemplate, $name));
			$this->writePhpFile($areaFolderPath.'/view.php', "\n");
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: could not create action [%s] files\n", $name));
			exit;
		}

		return true;
	}

	public function deleteArea($name)
	{
		$areaFolderPath = sprintf('%s/views/areas/%s', PIMCORE_WEBSITE_PATH, $name);

		# remove files
		try {
			foreach (scandir($areaFolderPath) as $fileOrFolder) {
				if (!in_array($fileOrFolder, ['.', '..'])) {
					unlink($areaFolderPath.'/'.$fileOrFolder);
				}
			}
			rmdir($areaFolderPath);
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: could not remove action [%s] files\n", $name));
			exit;
		}

		# disable area in extensions.php
		$extensionsPath = \Pimcore\Config::locateConfigFile("extensions.php");
		$exConfig = include($extensionsPath);
		if (isset($exConfig['brick'][$name])) {
			unset($exConfig['brick'][$name]);
		}
		return $this->writePhpFile($extensionsPath, $exConfig, true);
	}

	public function normalizeScriptFileName($action)
	{
		return strtolower(preg_replace('/([A-Z]+)/', "-$1", lcfirst($action)));
	}

	public function normalizePathParam($path, $startWithSlash = true)
	{
		if (stristr($path, 'Program Files') && stristr($path, '/Git/')) {
			$pos = stripos($path, '/Git/');
			$path = substr($path, $pos + 4);
		}
		if ($startWithSlash && $path[0] != '/') {
			$path = '/'.$path;
		}
		return $path;
	}

	private function getClassName($classNameOrId)
	{
		$class = null;
		$className = null;
		if (is_int($classNameOrId)) {
			$class = \Pimcore\Model\Object\ClassDefinition::getById($classNameOrId);
		} else {
			$class = \Pimcore\Model\Object\ClassDefinition::getByName($classNameOrId);
		}

		if ($class) {
			$className = $class->getName();
		}

		if (!$className) {
			fputs(STDOUT, sprintf("Error: class [%s] does not exist\n", $classNameOrId));
			exit;
		}

		return $className;
	}

	private function loadXmlConfig($pathOrXmlString = null)
	{
		if ($pathOrXmlString === null) {
			$pathOrXmlString = <<<EOT
<?xml version="1.0"?>
<zend-config xmlns:zf="http://framework.zend.com/xml/zend-config-xml/1.0/">
</zend-config>
EOT;
		}
		$config = null;
		try {
			$config = new \Zend_Config_Xml($pathOrXmlString, null, ["allowModifications" => true]);
		} catch (Exception $e) {
		}
		if (!$config instanceof \Zend_Config) {
			fputs(STDOUT, sprintf("Error: could not load [%s]\n", $pathOrXmlString));
			exit;
		}

		return $config;
	}

	private function writeXmlConfig($config, $path)
	{
		try {
			$writer = new \Zend_Config_Writer_Xml([
				"config" => $config,
				"filename" => $path
			]);
			$writer->write();
			// replace spaces with tabs
			file_put_contents($path, str_replace("	", "\t", file_get_contents($path)));
			file_put_contents($path, str_replace("  ", "\t", file_get_contents($path)));
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: could not write [%s]\n", $path));
			exit;
		}

		return true;
	}

	private function writePhpFile($path, $data, $toPhpDataFileFormat = false)
	{
		if ($toPhpDataFileFormat) {
			$data = to_php_data_file_format($data);
		}
		// replace spaces with tabs
		$data = str_replace("	", "\t", $data);
		return \Pimcore\File::putPhpFile($path, $data);
	}

	private function loadControllerTokens($controllerName)
	{
		$path = sprintf('%s/controllers/%sController.php', PIMCORE_WEBSITE_PATH, $controllerName);

		$fileContents = file_get_contents($path);

		if (!$fileContents) {
			fputs(STDOUT, sprintf("Error: controller [%s] can not be loaded\n", $path));
			exit;
		}

		return token_get_all($fileContents);
	}

	private function saveControllerTokens($controllerName, $tokens)
	{
		$path = sprintf('%s/controllers/%sController.php', PIMCORE_WEBSITE_PATH, $controllerName);

		$fileContents = '';
		foreach ($tokens as $token) {
			if (is_string($token)) {
				$fileContents .= $token;
			} else {
				list(, $text) = $token;
				$fileContents .= $text;
			}
		}

		try {
			$this->writePhpFile($path, $fileContents);
		} catch (Exception $e) {
			fputs(STDOUT, sprintf("Error: controller [%s] could not be saved\n", $path));
			exit;
		}

		return true;
	}

	private function checkIfTokenExists($tokens, $tokenPart, $forceExit = true)
	{
		foreach ($tokens as $token) {
			$tokenString = null;
			if (is_string($token)) {
				$tokenString .= $token;
			} else {
				list(, $tokenString) = $token;
			}
			if (stristr($tokenString, $tokenPart)) {
				if ($forceExit) {
					fputs(STDOUT, sprintf("Error: token [%s] already exists\n", $tokenPart));
					exit;
				}
				return true;
			}
		}

		return false;
	}

	private function findControllerSectionTokenIndex($tokens, $section)
	{
		$sectionFound = false;
		foreach ($tokens as $key => $token) {
			$text = null;
			if (is_string($token)) {
				$text = $token;
			} else {
				list(, $text) = $token;
			}
			$text = trim($text);

			if (strstr($text, $section) !== false) {
				$sectionFound = true;
			} elseif ($sectionFound && !empty($text)) {
				return $key;
			}
		}

		fputs(STDOUT, sprintf("Error: section [%s] not found\n", $section));
		exit;
	}
}
