<?php

namespace Website\Model;

require_once(PIMCORE_WEBSITE_PATH . '/models/GoPay/country_code.php');
require_once(PIMCORE_WEBSITE_PATH . '/models/GoPay/gopay_config.php');
require_once(PIMCORE_WEBSITE_PATH . '/models/GoPay/gopay_helper.php');
require_once(PIMCORE_WEBSITE_PATH . '/models/GoPay/gopay_http.php');
require_once(PIMCORE_WEBSITE_PATH . '/models/GoPay/gopay_soap.php');
require_once(PIMCORE_WEBSITE_PATH . '/models/GoPay/payment_methods.php');

class GoPay
{
	const LOG_TYPE_LENGTH = 30;

	protected $merchant;
	protected $secret;
	protected $language;
	protected $notificationsUrl;
	protected $callbackUrl;
	protected $successUrl;
	protected $failUrl;
	protected $productName;
	protected $goPayUrl;

	public function __construct($language)
	{
		$this->language = $language;
		$this->merchant = 8906967157;
		$this->secret = 'KxzMLHWAPrxSmNukD8KSUY4e';
		$this->productName = 'Nákup z www.kabea.cz';

		$baseUrl = \Zend_Controller_Front::getInstance()->getRequest()->getScheme() . '://' . \Zend_Controller_Front::getInstance()->getRequest()->getHttpHost() . \Zend_Controller_Front::getInstance()->getBaseUrl();

		$this->notificationsUrl = $baseUrl . \Website\Tool\Utils::url('e-commerce-gopay-notifications');
		$this->callbackUrl = $baseUrl . \Website\Tool\Utils::url('e-commerce-gopay-callback');
		$this->successUrl = $baseUrl . \Website\Tool\Utils::url('e-commerce-gopay-success');
		$this->failUrl = $baseUrl . \Website\Tool\Utils::url('e-commerce-gopay-fail');

		GopayConfig::init(GopayConfig::TEST);
	}

	public function createOrder(Order $order)
	{
		if ($order->getPaid()) {
			$this->log('already_paid', get_object_vars($order));
			header('Location: ' . $this->successUrl . "?sessionState=" . GopayHelper::PAID);
			exit;
		}
		/*
		 * Pole kodu platebnich metod, ktere se zobrazi na brane.
		 * Hodnoty viz vzorovy e-shop, zalozka "Prehled aktivnich platebnich metod"
		 * Zde nastaveno zobrazovani platebnich metod GoPay penezenka, platebni karty VISA, MasterCard, ePlatby a SuperCASH
		 * Pro zobrazovani vsech platebnich metod ponechte pole prazdne
		 */
		$paymentChannels = ['cz_cs_c'];

		/*
		 * Platebni metoda, ktere je prvotne vybrana na brane.
		 * Zde nastaveno prvotni vybrani platebni metody platebni karty VISA, MasterCard
		 */
		$defaultPaymentChannel = "cz_cs_c";

		$p1 = null;
		$p2 = null;
		$p3 = null;
		$p4 = null;

		/*
		 * Vytvoreni platby na strane GoPay prostrednictvim API funkce
		 * Pokud vytvoreni probehne korektne, je navracen identifikator $paymentSessionId
		 * Pokud nastane chyba, je vyhozena vyjimka
		 */

		try {
			$data = [
				'targetGoId' => (float) $this->merchant,
				'productName' => $this->productName,
				'totalPriceInCents' => $this->getPrice($order),
				'currency' => $this->getCurrency($order),
				'orderNumber' => $order->getOrderId(),
				'successUrl' => $this->callbackUrl,
				'failUrl' => $this->callbackUrl,
				'paymentChannels' => $paymentChannels,
				'defaultPaymentChannel' => $defaultPaymentChannel,
				'secretKey' => $this->secret,
				'firstName' => null,
				'lastName' => null,
				'city' => null,
				'street' => null,
				'postalCode' => null,
				'countryCode' => null,
				'email' => $order->getFEmail(),
				'phoneNumber' => null,
				'p1' => $p1,
				'p2' => $p2,
				'p3' => $p3,
				'p4' => $p4,
				'lang' => $this->language
			];
			$this->log('create_soap_payment', $data);
			$paymentSessionId = GopaySoap::createPayment((float) $this->merchant, $this->productName, $this->getPrice($order), $this->getCurrency($order), $order->getOrderId(), $this->callbackUrl, $this->callbackUrl, $paymentChannels, $defaultPaymentChannel, $this->secret, null, null, null, null, null, null, $order->getFEmail(), null, $p1, $p2, $p3, $p4, $this->language
			);

			/*
			 * Platba na strane GoPay uspesne vytvorena
			 * Ulozeni paymentSessionId k objednavce. Slouzi pro komunikaci s GoPay
			 */
			$paymentSessionIds = (array)explode(',', (string)$order->getGopayPaymentSessionId());
			if (!in_array($paymentSessionId, $paymentSessionIds)) {
				$paymentSessionIds[] = $paymentSessionId;
			}
			$order->setGopayPaymentSessionId(implode(',', $paymentSessionIds));
			$order->setOmitMandatoryCheck(true);
			$order->save();
		} catch (\Exception $e) {
			/*
			 *  Osetreni chyby v pripade chybneho zalozeni platby
			 */
			$this->log('create_soap_payement_error:' . $e->getMessage());
			header('Location: ' . $this->failUrl . "?sessionState=" . GopayHelper::FAILED);
			exit;
		}

		$encryptedSignature = GopayHelper::encrypt(
						GopayHelper::hash(GopayHelper::concatPaymentSession((float) $this->merchant, (float) $paymentSessionId, $this->secret)), $this->secret
		);

		$redirectUrl = GopayConfig::fullIntegrationURL() . "?sessionInfo.targetGoId=" . $this->merchant . "&sessionInfo.paymentSessionId=" . $paymentSessionId . "&sessionInfo.encryptedSignature=" . $encryptedSignature;
		$this->log('created_soap_payment_redirecting_to', ['url' => $redirectUrl]);
		/*
		 * Presmerovani na platebni branu GoPay s predvybranou platebni metodou GoPay penezenka ($defaultPaymentChannel)
		 */
		header('Location: ' . $redirectUrl);
		exit;
	}

	public function handleNotification($params)
	{
		try {
			$order = $this->getOrderByGoPaySessionId($params['paymentSessionId']);

			$this->setLanguageFromOrder($order);

			$returnedPaymentSessionId = $params['paymentSessionId'];
			$returnedParentPaymentSessionId = $params['parentPaymentSessionId'];
			$returnedGoId = $params['targetGoId'];
			$returnedOrderNumber = $params['orderNumber'];
			$returnedEncryptedSignature = $params['encryptedSignature'];

			GopayHelper::checkPaymentIdentity(
					(float) $returnedGoId, (float) $returnedPaymentSessionId, (float) $returnedParentPaymentSessionId, $returnedOrderNumber, $returnedEncryptedSignature, (float) $this->merchant, $order->getOrderId(), $this->secret
			);

			/*
			 * Kontrola zaplacenosti objednavky na strane GoPay
			 */
			$result = GopaySoap::isPaymentDone(
							(float) $returnedPaymentSessionId, (float) $this->merchant, $order->getOrderId(), $this->getPrice($order), $this->getCurrency($order), $this->productName, $this->secret
			);

			switch ($result["sessionState"]) {
				case GopayHelper::PAID:
					/*
					 * Zpracovat pouze objednavku, ktera jeste nebyla zaplacena
					 */
					if (empty($returnedParentPaymentSessionId)) {
						// notifikace o bezne platbe
						if ($order->getGopayState() != GopayHelper::PAID) {
							$order->setGopayState(GopayHelper::PAID);
							$order->setPaid(true);
							$order->setOmitMandatoryCheck(true);
							$order->save();
						}
					} else {
						throw new \Exception('not_implemented_yet');
					}
					break;
				case GopayHelper::TIMEOUTED:
				case GopayHelper::REFUNDED:
				case GopayHelper::AUTHORIZED:
				case GopayHelper::CANCELED:
				default:
					//@TODO
					throw new \Exception('not_implemented_yet');
					break;
			}
		} catch (\Exception $e) {
			/*
			 * Nevalidni informace z http notifikaci - prevdepodobne pokus o podvodne zaslani notifikace
			 */
			$this->log('handle_notification_error:' . $e->getMessage(), $params);
			return true;
		}

		$this->log('handle_notification_success', $params);
	}

	public function handleCallback($params)
	{
		/*
		 * Kontrola validity parametru v redirectu, opatreni proti podvrzeni potvrzeni / zruseni platby
		 */
		try {
			/*
			 * Parametry obsazene v redirectu po potvrzeni / zruseni platby, predavane od GoPay e-shopu
			 */
			$returnedPaymentSessionId = $params['paymentSessionId'];
			$returnedGoId = $params['targetGoId'];
			$returnedOrderNumber = $params['orderNumber'];
			$returnedEncryptedSignature = $params['encryptedSignature'];

			/*
			 * Nacist data objednavky dle prichoziho paymentSessionId, zde z testovacich duvodu vse primo v testovaci tride Order
			 * Upravte dle ulozeni vasich objednavek
			 */
			$order = $this->getOrderByGoPaySessionId($params['paymentSessionId']);

			GopayHelper::checkPaymentIdentity(
					(float) $returnedGoId, (float) $returnedPaymentSessionId, null, $returnedOrderNumber, $returnedEncryptedSignature, (float) $this->merchant, $order->getOrderId(), $this->secret
			);

			/*
			 * Kontrola zaplacenosti objednavky na serveru GoPay
			 */
			$result = GopaySoap::isPaymentDone(
							(float) $returnedPaymentSessionId, (float) $this->merchant, $order->getOrderId(), $this->getPrice($order), $this->getCurrency($order), $this->productName, $this->secret
			);

			switch ($result["sessionState"]) {
				case GopayHelper::PAID:
					/*
					 * Zpracovat pouze objednavku, ktera jeste nebyla zaplacena
					 */
					if ($order->getGopayState() != GopayHelper::PAID) {

						/*
						 *  Zpracovani objednavky  ! UPRAVTE !
						 */
						$order->setGopayState(GopayHelper::PAID);
						$order->setPaid(true);
						$order->save();
					}

					/*
					 * Presmerovani na prezentaci uspesne platby
					 */
					$location = $this->successUrl;
					break;
				/* Platba ceka na zaplaceni */
				case GopayHelper::PAYMENT_METHOD_CHOSEN:
				/* Platba byla autorizovana, ceka se na dokonceni  */
				case GopayHelper::AUTHORIZED:
				//$order->autorizePayment();
				/* Platba byla vracena - refundovana  */
				case GopayHelper::REFUNDED:
					//$order->refundPayment();
					$location = $this->successUrl;
					break;
				/* Platba nebyla zaplacena */
				case GopayHelper::CREATED:
				/* Platba byla zrusena objednavajicim */
				case GopayHelper::CANCELED:
				//$order->cancelPayment();
				/* Platnost platby vyprsela  */
				case GopayHelper::TIMEOUTED:
				//$order->timeoutPayment();
				default:
					$location = $this->failUrl;
					break;
			}

			$this->log('handle_callback_success', $params);
			header('Location: ' . $location . "?sessionState=" . $result["sessionState"] . "&sessionSubState=" . $result["sessionSubState"]);
			exit;
		} catch (\Exception $e) {
			/*
			 * Nevalidni informace z redirectu
			 */
			$this->log('handle_callbaack_error:' . $e->getMessage(), $params);
			header('Location: ' . $this->failUrl . "?sessionState=" . GopayHelper::FAILED);
			exit;
		}
	}

	private function getOrderByGoPaySessionId($id)
	{
		$orderList = new Order\Listing();
		$orderList->setUnpublished(true);
		$orderList->setCondition(\Pimcore\Db::get()->quoteInto('gopayPaymentSessionId LIKE ?', '%'.$id.'%'));
		$orderList->load();
		$order = $orderList->current();

		if (!$order instanceof Order) {
			throw new \Exception('could_not_load_order');
		}

		return $order;
	}

	private function getCurrency(Order $order)
	{
		return 'CZK';
	}

	private function getPrice(Order $order)
	{
		//@TODO other currencies when eshop allows them
		$price = $order->getFinalPrice();
		return (int) (round($price) * 100);
	}

	private function setLanguageFromOrder(Order $order)
	{
		$this->language = substr($order->getCurrency(), 0, 2);
	}

	public function log($type, $data = [])
	{
		$parts = [];
		foreach ($data as $key => $value) {
			if (!is_array($value) && !is_object($value)) {
				$parts[] = $key . ':' . $value;
			}
		}
		\Pimcore\Log\Simple::log('gopay', str_pad($type, self::LOG_TYPE_LENGTH, ' ', STR_PAD_LEFT).' - '.  implode('|', $parts));
	}
}
