<?php

/**
 *
 * @author Martin Kuric <martin.kuric@portadesign.cz>
 */

namespace Website\Model;

class OrderManager
{
	const ORDER_ID_LENGTH = 8;
	const INVOICE_ID_LENGTH = 5;
	//statuses are hardcoded in Order class (status select values)
	const STATUS_NEW = 'new';
	const STATUS_COMPLETE = 'complete';
	const STATUS_CANCELED = 'canceled';
	const STATUS_RETURNED = 'returned';
	//payment types are hardcoded in pimcore admin in Order class (paymentType select values) and in EshopSettings class in the structured tables
	const PAYMENT_CASH = 'payment_cash';
	const PAYMENT_COD = 'payment_cod';
	const PAYMENT_BANK_TRANSFER = 'payment_bank_transfer';
	const PAYMENT_CREDIT_CARD = 'payment_credit_card';
	//shipping types are hardcoded in pimcore admin in Order class (shipingType select values) and in EshopSettings class in the structured tables
	const SHIPPING_PICK_UP = 'shipping_pick_up';
	const SHIPPING_CZECH_PPL = 'shipping_czech_ppl';
	const SHIPPING_CZECH_POST = 'shipping_czech_post';
	const SHIPPING_ULOZENKA = 'shipping_ulozenka';

	/**
	 * @var \Website\Model\Currency
	 */
	protected $currencyModel;
	/**
	 *
	 * @var \Pimcore\Translate\Website
	 */
	protected $translator;

	public function __construct($translator, \Website\Model\Currency $currencyModel)
	{
		$this->currencyModel = $currencyModel;
		$this->translator = $translator;
	}

	/**
	 * For store...
	 * @return boolean|Order
	 */
	public function createOrderByHand()
	{
		$eshopSettings = \Website\Tool\Utils::getEshopSettings();

		$order = new Order();
		$order->setO_creationDate(time());
		$order->setParentId(8);
		$orderId = $this->genNextOrderId();
		$order->setKey(\Pimcore\File::getValidFilename($orderId));
		$order->setOrderId($orderId);
		$order->setPublished(true);
		$order->setFromStore(true);
		$order->setStatus(self::STATUS_COMPLETE);
		$order->setPaymentType(self::PAYMENT_CASH);
		$order->setShippingType(self::SHIPPING_PICK_UP);

		$order->setShippingPrice(0);
		$order->setBaseShippingPrice(0);
		$order->setShippingDiscount(0);
		$order->setShippingCost(0);
		$order->setPaymentPrice(0);
		$order->setBasePaymentPrice(0);
		$order->setPaymentDiscount(0);
		$order->setPaymentCost(0);
		$order->setProductsPrice(0);
		$order->setProductsDiscount(0);
		$order->setFinalPrice(0);

		$order->setVatLow($eshopSettings->getVatLow());
		$order->setVatHigh($eshopSettings->getVatHigh());
		$order->setCurrency('cs_CZ');
		$order->setCountry('CZ');
		$order->setPaid(true);

		try {
			$order->save();
		} catch (\Exception $e) {
			\Pimcore\Log\Simple::log('exceptions', 'CREATE ORDER BY HAND - '.$e->getMessage() . "\n" . $e->getTraceAsString());
			return false;
		}

		return $order;
	}

	/**
	 * For administration...
	 * @param string $orderId Order id
	 * @param srting $items format fieldCollectionKey:productId:count|fieldCollectionKey2:productId2:count2...
	 * @param string $shippingType
	 * @param string $paymentType
	 * @param int $discountCodeId
	 * @param boolean $persist - if you turn it off, the inStock numbers on products won't be updated
	 * @return string response as an array ready for json action helper
	 */
	public function recalculateOrder($orderId, $items = null, $shippingType = null, $paymentType = null,
									$discountCodeId = null, $persist = true)
	{
		\Website\Tool\Utils::forceInheritanceAndFallbackValues();

		$order = Order::getById($orderId);
		if (!$order) {
			return ['status' => false];
			exit;
		}
		$response = ['status' => true, 'products' => []]; # empty products array in response means no changes (not no products)
		//@TODO currency model and translator are now set through the construstor of shop model, maybe create a new ones for this method ?
		$language = substr($this->translator->getLocale(), 0, 2);
		$country = $this->currencyModel->getCountry();

		//do the products
		$productsPrice = 0;
		$oldProducts = [];
		$newProducts = new \Pimcore\Model\Object\Fieldcollection();
		$newProductsIds = []; #for stock count update of removed products
		$tmp = $order->getProducts();
		$itemsTmp = [];
		if ($tmp) {
			foreach ($tmp as $fieldCollectionKey => $fieldCollection) {
				if ($fieldCollection->getProduct()) {
					if (!$items) {
						$itemsTmp[] = $fieldCollectionKey.':'.$fieldCollection->getProduct()->getId().':'.$fieldCollection->getCount();
					}
					$oldProducts[$fieldCollection->getProduct()->getId()] = $fieldCollection;
				}
			}
		}
		if (!$items) {
			$items = implode('|', $itemsTmp);
		}

		if (!empty($items)) {
			foreach (explode('|', $items) as $item) {
				list($fieldCollectionKey, $productId, $count) = explode(':', $item);
			//usage of list because of unpublished products (prefered way of hiding sold out products in e-shop)
			$productList = new Product\Listing();
				$productList->setUnpublished(true);
				$productList->setCondition(\Pimcore\Db::get()->quoteInto('o_id = ?', $productId));
				$productList->load();
				$product = $productList->current();
				if ($product) {
					list(, $productPrice) = $this->currencyModel->getProductPrices($product);
					$price = $productPrice * $count;
					$productCost = $product->{'getCost'.$country}();
					$name = $product->getVariantName($language);
					$vat = $product->getIsLowVat() ? $order->getVatLow() : $order->getVatHigh();
					$inStockDiff = 0;
				//update stock count based on old order products (handles [product update | product addition])
				if (isset($oldProducts[$productId])) {
					$inStockDiff =  $oldProducts[$productId]->getCount() - $count;
				} else {
					$inStockDiff = -1 * $count;
				}
					if ($inStockDiff != 0) {
						$product->setInStock(max([0, $product->getInStock() + $inStockDiff]));
						if ($persist) {
							try {
								$product->setOmitMandatoryCheck(true);
								$product->save();
							} catch (\Exception $e) {
								\Pimcore\Log\Simple::log('exceptions', 'ORDER UPDATE PRODCUCT STOCK FAILED - '.$e->getMessage() . "\n" . $e->getTraceAsString());
								continue;
							}
						}
					}
					$newProductsIds[] = $productId; #for removal handling
				$fieldCollection = new \Pimcore\Model\Object\Fieldcollection\Data\Orderproduct();
					$fieldCollection->setProduct($product);
					$fieldCollection->setName($name);
					$fieldCollection->setCount($count);
					$fieldCollection->setPrice($productPrice);
					$fieldCollection->setCost($productCost);
					$fieldCollection->setIsLowVat($product->getIsLowVat());
					$fieldCollection->setVat($vat);
					$newProducts->add($fieldCollection);
					$response['products'][$fieldCollectionKey] = [
					'name' => $fieldCollection->getName(), 'vat' => $fieldCollection->getVat(),
					'price' => $fieldCollection->getPrice(), 'cost' => $fieldCollection->getCost(),
					'count' => $fieldCollection->getCount(), 'isLowVat' => $fieldCollection->getIsLowVat()
				];
					$productsPrice += $price;
				} else { # only copy the existing collection
				$newProducts->add($tmp->get($fieldCollectionKey));
				}
			}
		}

		//update stock count based on old order products (handles [product removal])
		foreach ($oldProducts as $productId => $fieldCollection) {
			$product = $fieldCollection->getProduct();
			if (!in_array($productId, $newProductsIds)) {
				$product->setInStock($product->getInStock() + $fieldCollection->getCount());
				if ($persist) {
					try {
						$product->save();
					} catch (\Exception $e) {
						\Pimcore\Log\Simple::log('exceptions', 'ORDER UPDATE PRODCUCT STOCK FAILED SILENTLY - '.$e->getMessage() . "\n" . $e->getTraceAsString());
					}
				}
			}
		}
		$order->setProducts($newProducts);
		$order->setProductsPrice($productsPrice);

		//discount code, products discount
		$productsDiscount = 0;
		$discountCode = null;
		if (!$discountCodeId && $order->getDiscountCode()) {
			$discountCodeId = $order->getDiscountCode()->getId();
		}
		if ($discountCodeId) {
			$tmp = $this->applyDiscountCode((integer)$discountCodeId, $order->getProductsPrice(), true);
			if ($tmp !== false) {
				list($productsDiscount, $discountedProductsPrice, $discountCode) = $tmp;
			}
		}
		$order->setProductsDiscount($productsDiscount);
		//shipping price
		$applyDiscountOnShipping =($discountCode && $discountCode->getApplyOnShipping()) ? true : false;
		if ($shippingType) {
			$order->setShippingType($shippingType);
		}
		list($baseShippingPrice, $shippingPrice) = $this->getShippingPrices(
			$order->getCountry(),
			$order->getShippingType(),
			$order->getProductsPrice(),
			$productsDiscount,
			$applyDiscountOnShipping
		);
		$order->setShippingPrice($shippingPrice);
		$order->setBaseShippingPrice($baseShippingPrice);
		$order->setShippingDiscount($baseShippingPrice - $shippingPrice);
		$order->setShippingCost(\Website\Tool\Utils::getOurShippingPrice($order->getPaymentType(), $order->getShippingType(), $country));
		//payment price and payment type discount
		if ($paymentType) {
			$order->setPaymentType($paymentType);
		}
		list($basePaymentPrice, $paymentPrice) = $this->getPaymentPrices(
			$order->getCountry(),
			$order->getShippingType(),
			$order->getPaymentType(),
			$order->getProductsPrice(),
			$productsDiscount,
			$applyDiscountOnShipping
		);
		$order->setPaymentPrice($paymentPrice);
		$order->setBasePaymentPrice($basePaymentPrice);
		$order->setPaymentDiscount($basePaymentPrice - $paymentPrice);
		$order->setPaymentCost(0);//always 0 so far...
		//calculate the final price and global discount
		$finalPrice = max([0, ($order->getProductsPrice() - $productsDiscount) + $order->getShippingPrice() + $order->getPaymentPrice()]);
		$order->setFinalPrice($finalPrice);
		if ($finalPrice <= 0) {
			$order->setPaid(true);
		}

		$response['shippingPrice'] = $order->getShippingPrice();
		$response['baseShippingPrice'] = $order->getBaseShippingPrice();
		$response['shippingDiscount'] = $order->getShippingDiscount();
		$response['shippingCost'] = $order->getShippingCost();
		$response['paymentPrice'] = $order->getPaymentPrice();
		$response['basePaymentPrice'] = $order->getBasePaymentPrice();
		$response['paymentDiscount'] = $order->getPaymentDiscount();
		$response['paymentCost'] = $order->getPaymentCost();
		$response['productsPrice'] = $order->getProductsPrice();
		$response['productsDiscount'] = $order->getProductsDiscount();
		$response['finalPrice'] = $order->getFinalPrice();
		//update order
		if ($persist) {
			try {
				$order->save();
			} catch (\Exception $e) {
				\Pimcore\Log\Simple::log('exceptions', 'ORDER UPDATE FAILED - '.$e->getMessage() . "\n" . $e->getTraceAsString());
			}
		}

		\Website\Tool\Utils::restorInheritanceAndFallbackValues();

		return $response;
	}

	public function createOrder(\Website\Model\Cart $cart, \Pimcore\Model\Object\User $user)
	{
		$orderObjectId = $cart->getOrderId();
		$order = $cart->getOrder();
		$eshopSettings = \Website\Tool\Utils::getEshopSettings();
		$language = substr($this->translator->getLocale(), 0, 2);

		try {
			//set contact stuff
			$contact = $order->contactInfo;
			if (empty($contact['deliveryAddress'])) {
				$contact['dFirstName'] = $contact['fFirstName'];
				$contact['dLastName'] = $contact['fLastName'];
				$contact['dCompany'] = $contact['fCompany'];
				$contact['dStreet'] = $contact['fStreet'];
				$contact['dCity'] = $contact['fCity'];
				$contact['dZipCode'] = $contact['fZipCode'];
			}
			if ($orderObjectId) {
				$orderObject = Order::getById($orderObjectId);
				$orderObject->setValues($contact);
			} else {
				$orderObject = Order::create($contact);
				$orderId = $this->genNextOrderId();
				if (!$orderId) {
					return false;
				}
				$orderObject->setParentId(\Pimcore\Model\Object\Folder::getByPath('/eshop/objednavky')->getId());
				$orderObject->setKey(\Pimcore\File::getValidFilename($orderId));
				$orderObject->setOrderId($orderId);
				$orderObject->setO_creationDate(time());
				$orderObject->setPublished(true);
			}
			if (!$orderObject instanceof Order) {
				throw new \Exception('could_not_load_order_from_order_id');
			}

			//set object stuff
			$orderObject->setUser($user);
			$orderObject->setDiscountCode(null);
			$orderObject->setStatus(self::STATUS_NEW);

			//set order stuff
			$products = new \Pimcore\Model\Object\Fieldcollection();
			foreach ($cart->getProductObjects() as $productId => $product) {
				$newProduct = new \Pimcore\Model\Object\Fieldcollection\Data\Orderproduct();
				$newProduct->setProduct($product);
				$newProduct->setCount($order->products[$productId]->count);
				list(, $discountedPrice) = $this->currencyModel->getProductPrices($product);
				$newProduct->setPrice($discountedPrice);
				$newProduct->setCost($product->{'getCost' . $this->currencyModel->getCountry()}());
				$newProduct->setName($product->getVariantName($language));
				$isLowVat = $product->getIsLowVat();
				$newProduct->setIsLowVat($isLowVat);
				$newProduct->setVat(($isLowVat) ? $eshopSettings->getVatLow() : $eshopSettings->getVatHigh());
				$products->add($newProduct);
				if (!$orderObjectId) {
					$product->decreaseInStockSize($newProduct->getCount());
				}
			}
			$orderObject->setProducts($products);

			$orderObject->setProductsPrice($order->productsPrice);
			//add discount from discount code if any
			$productsDiscount = $cart->getProductsDiscount();
			if ($productsDiscount->discountValue) {
				$orderObject->setDiscountCode($productsDiscount->code);
				$productsDiscount->code->setUsed(true);
				$productsDiscount->code->save();
			}
			$orderObject->setShippingType($order->shippingType);
			$orderObject->setShippingPrice($order->shippingPrice);
			$orderObject->setBaseShippingPrice($order->baseShippingPrice);
			$orderObject->setShippingDiscount($order->baseShippingPrice - $order->shippingPrice);
			$orderObject->setShippingCost(\Website\Tool\Utils::getOurShippingPrice($order->paymentType, $order->shippingType, $this->currencyModel->getCountry()));
			$orderObject->setPaymentType($order->paymentType);
			$orderObject->setPaymentPrice($order->paymentPrice);
			$orderObject->setBasePaymentPrice($order->basePaymentPrice);
			$orderObject->setPaymentDiscount($order->basePaymentPrice - $order->paymentPrice);
			$orderObject->setPaymentCost(0);//always 0

			$discountedProductsPrice = ($productsDiscount->discountValue) ? $productsDiscount->discountedPrice : $order->productsPrice;
			$finalPrice = max([0, $discountedProductsPrice + $order->shippingPrice + $order->paymentPrice]);
			if ($finalPrice <= 0) {
				$orderObject->setPaid(true);
			}
			$orderObject->setFinalPrice($finalPrice);
			$orderObject->setProductsDiscount($order->productsPrice - $discountedProductsPrice);//TODO
			$orderObject->setCurrency($this->currencyModel->getCurrency()->getLocale());
			$orderObject->setCountry($order->country);
			$orderObject->setVatLow($eshopSettings->getVatLow());
			$orderObject->setVatHigh($eshopSettings->getVatHigh());
			$orderObject->setIsTaxPayer($eshopSettings->getSupplierIsTaxPayer());

			$orderObject->save();

			$this->sendConfirmationEmail($orderObject, true);

			if (!$orderObjectId) {
				//place for other code that should be called only when the order is really created
			}

			return $orderObject;
		} catch (\Exception $e) {
			\Pimcore\Log\Simple::log('order_exceptions', $e->getMessage() . "\n" . $e->getTraceAsString());
			return false;
		}
	}

	public function sendConfirmationEmail(Order $order, $notifyAdministrator = true)
	{
		$eshopSettings = \Website\Tool\Utils::getEshopSettings();
		$language = substr($this->translator->getLocale(), 0, 2);

		$notificationMail = [
			'id' => '',
			'products' => '',
			'orderInfo' => '',
			'invoiceAddress' => '',
			'deliveryAddress' => ''
		];
		foreach ($order->getProducts() as $orderProduct) {
			//notification mail stuff
			$notificationMail['products'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_product_name'), $orderProduct->getName());
			$notificationMail['products'] .= sprintf('%s: %d<br/>%s: %s<br/>',
				$this->translator->translate('label_product_count'),
				$orderProduct->getCount(),
				$this->translator->translate('label_price_per_one'),
				$this->currencyModel->localizeCurrency($orderProduct->getPrice(), true)
			);
		}

		$helper = new \Zend_View_Helper_ServerUrl();
		$notificationMail['order'] = $order;
		$notificationMail['deeplink'] = $helper->serverUrl().'/admin/login/deeplink?object_'.$order->getId().'_object';
		$notificationMail['orderInfo'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_payment_type'), $this->translator->translate('label_' . $order->getPaymentType(), $language));
		$notificationMail['orderInfo'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_shipping_type'), $this->translator->translate('label_' . $order->getShippingType(), $language));
		$notificationMail['orderInfo'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_products_price'), $this->currencyModel->localizeCurrency($order->getProductsPrice(), true));
		if ($order->getBasePaymentPrice()) {
			$notificationMail['orderInfo'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_payment_price'), $this->currencyModel->localizeCurrency($order->getBasePaymentPrice(), true));
		}
		if ($order->getBaseShippingPrice()) {
			$notificationMail['orderInfo'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_shipping_price'), $this->currencyModel->localizeCurrency($order->getBaseShippingPrice(), true));
		}
		$discountValue = $order->getProductsDiscount() + $order->getShippingDiscount() + $order->getPaymentDiscount();
		if ($discountValue) {
			$notificationMail['orderInfo'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_discount'), $this->currencyModel->localizeCurrency($discountValue, true));
			if ($order->getDiscountCode()) {
				$notificationMail['orderInfo'] .= sprintf(' (%s: %s)', $this->translator->translate('label_code'), $order->getDiscountCode()->getCode());
			}
			$notificationMail['orderInfo'] .= '<br/>';
		}
		$notificationMail['orderInfo'] .= sprintf('%s: <b>%s</b><br/>', $this->translator->translate('label_final_price'), $this->currencyModel->localizeCurrency($order->getFinalPrice(), true));
		if ($order->getMessage()) {
			$notificationMail['orderInfo'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_customer_message'), $order->getMessage());
		}

		$notificationMail['invoiceAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_first_name'), $order->getFFirstName());
		$notificationMail['invoiceAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_last_name'), $order->getFLastName());
		if ($order->getFCompany()) {
			$notificationMail['invoiceAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_company'), $order->getFCompany());
		}
		$notificationMail['invoiceAddress'] .= sprintf('%s: %s, %s, %s<br/>', $this->translator->translate('label_address'), $order->getFStreet(), $order->getFZipCode(), $order->getFCity());
		$countries = \Zend_locale::getTranslationList('Territory', $this->translator->getLocale(), 2);
		$notificationMail['invoiceAddress'] .= sprintf('%s: %s<br/>',
			$this->translator->translate('label_country'),
			isset($countries[$order->getCountry()])
				? $countries[$order->getCountry()]
				: $this->translator->translate('label_country_unknown')
		);
		$notificationMail['invoiceAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_email'), $order->getFEmail());
		$notificationMail['invoiceAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_phone'), $order->getFPhone());
		if ($order->getFIco()) {
			$notificationMail['invoiceAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_ic'), $order->getFIco());
		}
		if ($order->getFDic()) {
			$notificationMail['invoiceAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_dic'), $order->getFDic());
		}

		$notificationMail['deliveryAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_first_name'), $order->getDFirstName());
		$notificationMail['deliveryAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_last_name'), $order->getDLastName());
		if ($order->getDCompany()) {
			$notificationMail['deliveryAddress'] .= sprintf('%s: %s<br/>', $this->translator->translate('label_company'), $order->getDCompany());
		}
		$notificationMail['deliveryAddress'] .= sprintf('%s: %s, %s, %s<br/>', $this->translator->translate('label_address'), $order->getDStreet(), $order->getDZipCode(), $order->getDCity());

		$result = 1;
		$mailer = new \Website\Model\MailManager();
		if ($order->getFinalPrice() <= 0) {
			$result = $mailer->sendMail('/notifikace/potvrzeni-objednavky-zdarma', $language, $order->getFEmail(), null, $notificationMail);
		} elseif ($order->getPaymentType() == self::PAYMENT_CASH) {
			$result = $mailer->sendMail('/notifikace/potvrzeni-objednavky-platba-hotove', $language, $order->getFEmail(), null, $notificationMail);
		} elseif ($order->getPaymentType() == self::PAYMENT_COD) {
			$result =  $mailer->sendMail('/notifikace/potvrzeni-objednavky-platba-na-dobirku', $language, $order->getFEmail(), null, $notificationMail);
		} elseif ($order->getPaymentType() == self::PAYMENT_BANK_TRANSFER) {
			$notificationMail['bankAccountNumber'] = $eshopSettings->getSupplierAccountNumber();
			$result =  $mailer->sendMail('/notifikace/potvrzeni-objednavky-platba-prevodem', $language, $order->getFEmail(), null, $notificationMail);
		} elseif ($order->getPaymentType() == self::PAYMENT_CREDIT_CARD) {
			$result =  $mailer->sendMail('/notifikace/potvrzeni-objednavky-platba-kartou', $language, $order->getFEmail(), null, $notificationMail);
		}
		if ($notifyAdministrator) {
			$result *= $mailer->sendMail('/notifikace/pro-administratora-nova-objednavka', $language, null, null, $notificationMail);
		}

		return (bool)$result;
	}

	public function sendCompletionEmail(Order $order)
	{
		$mailManager = new \Website\Model\MailManager();
		$language = substr($this->translator->getLocale(), 0, 2);

		$result = true;
		if ($order->getShippingType() == self::SHIPPING_CZECH_POST) {
			$result = $mailManager->sendMail('/notifikace/vyrizeni-objednavky-ceska-posta', $language, $order->getFEmail(), null, [
				'order' => $order
			]);
		} elseif ($order->getShippingType() == self::SHIPPING_CZECH_PPL) {
			$mailManager->sendMail('/notifikace/vyrizeni-objednavky-ppl', $language, $order->getFEmail(), null, [
				'order' => $order
			]);
		} elseif ($order->getShippingType() == self::SHIPPING_ULOZENKA) {
			$result = $mailManager->sendMail('/notifikace/vyrizeni-objednavky-ulozenka', $language, $order->getFEmail(), null, [
				'order' => $order
			]);
		} elseif ($order->getShippingType() == self::SHIPPING_PICK_UP) {
			$result = $mailManager->sendMail('/notifikace/vyrizeni-objednavky-osobni-odber', $language, $order->getFEmail(), null, [
				'order' => $order
			]);
		}

		return $result;
	}

	private function genNextOrderId()
	{
		$yearStart = mktime(0, 0, 0, 1, 1, date("Y"));
		$yearEnd = mktime(0, 0, 0, 1, 1, (int)date("Y")+1) - 1;

		$list = new Order\Listing();
		$list->setUnpublished(true);
		$list->setCondition(sprintf('o_creationDate BETWEEN %d AND %d', $yearStart, $yearEnd));
		$list->setOrder('DESC');
		$list->setOrderKey('orderId');
		$list->setLimit(1);

		$tmpOrder = $list->current();
		if ($tmpOrder instanceof Order) {
			$tmpOrderId = $tmpOrder->getOrderId();
			if (!$tmpOrderId) {
				\Pimcore\Log\Simple::log('exceptions', 'Empty object error generating orderId - probably a cache inconsistency');
				return false;
			}
			$orderId = $tmpOrderId + 1;//NOTICE this is not safe without transactions, duplicate object key error might happen
		} else {
			$orderId = date('y', time()).str_pad(1, self::ORDER_ID_LENGTH, "0", STR_PAD_LEFT);
		}

		return $orderId;
	}

	/**
	 * @param int $orderId
	 * @return boolean|Order
	 */
	public function genInvoice($orderId)
	{
		$yearStart = mktime(0, 0, 0, 1, 1, date("Y"));
		$yearEnd = mktime(0, 0, 0, 1, 1, (int)date("Y")+1) - 1;

		$order = Order::getById($orderId);

		if (!$order || $order->getFromStore()) {
			return 0;
		}
		if (!in_array($order->getStatus(), [self::STATUS_NEW, self::STATUS_COMPLETE])) {
			return -1;
		}

		$list = new Order\Listing();
		$list->setUnpublished(true);
		$list->setCondition(sprintf("(invoiceId IS NOT NULL AND invoiceId <> 0) AND issuedOn BETWEEN %d AND %d", $yearStart, $yearEnd));
		$list->setOrder('DESC');
		$list->setOrderKey('orderId');
		$list->setLimit(1);

		$tmpOrder = $list->current();
		if ($tmpOrder instanceof Order) {
			$tmpInvoiceId = $tmpOrder->getInvoiceId();
			if (!$tmpInvoiceId) {
				\Pimcore\Log\Simple::log('exceptions', 'Empty object error generating InvoiceId - probably a cache inconsistency');
				return 0;
			}
			$invoiceId = $tmpInvoiceId + 1;//NOTICE this is not safe without transactions, duplicate object key error might happen
		} else {
			$year = date('Y', time());
			$offset = 1;
			$invoiceId = $year.str_pad($offset, self::INVOICE_ID_LENGTH, "0", STR_PAD_LEFT);
		}

		try {
			$eshopSettings = \Website\Tool\Utils::getEshopSettings();
			$order->setInvoiceId($invoiceId);
			$order->setIssuedOn(new \Zend_Date());
			$order->setDueTo(new \Zend_Date());
			$order->setIsTaxPayer($eshopSettings->getSupplierIsTaxPayer());
			$order->save();
		} catch (\Exception $e) {
			\Pimcore\Log\Simple::log('exceptions', 'GENERATING INVOICE - '.$e->getMessage() . "\n" . $e->getTraceAsString());
			return 0;
		}

		return $order;
	}

	/**
	 * @param int $orderId
	 * @return boolean|Order
	 */
	public function genCreditNote($orderId)
	{
		$yearStart = mktime(0, 0, 0, 1, 1, date("Y"));
		$yearEnd = mktime(0, 0, 0, 1, 1, (int)date("Y")+1) - 1;

		$order = Order::getById($orderId);

		if (!$order || $order->getFromStore()) {
			return 0;
		}
		if (!$order->getInvoiceId()) {
			return -1;
		}
		$list = new Order\Listing();
		$list->setUnpublished(true);
		$list->setCondition(sprintf("(creditNoteId IS NOT NULL AND creditNoteId <> 0) AND cnIssuedOn BETWEEN %d AND %d", $yearStart, $yearEnd));
		$list->setOrder('DESC');
		$list->setOrderKey('creditNoteId');
		$list->setLimit(1);

		$tmpOrder = $list->current();
		if ($tmpOrder instanceof Order) {
			$tmpCreditNoteId = $tmpOrder->getCreditNoteId();
			if (!$tmpCreditNoteId) {
				\Pimcore\Log\Simple::log('exceptions', 'Empty object error generating CreditNoteId - probably a cache inconsistency');
				return 0;
			}
			$creditNoteId = $tmpCreditNoteId + 1;//NOTICE this is not safe without transactions, duplicate object key error might happen
		} else {
			$year = date('Y', time());
			$offset = 1;
			$creditNoteId = $year.str_pad($offset, self::INVOICE_ID_LENGTH, "0", STR_PAD_LEFT);
		}

		try {
			$order->setCreditNoteId($creditNoteId);
			$order->setCnIssuedOn(new \Zend_Date());
			$order->setCnDueTo(new \Zend_Date());
			$order->save();
		} catch (\Exception $e) {
			\Pimcore\Log\Simple::log('exceptions', 'GENERATING CREDIT NOTE - '.$e->getMessage() . "\n" . $e->getTraceAsString());
			return 0;
		}

		return $order;
	}

	/**
	 * @TODO this is just copied from kabea.cz...
	 * @param int $since timestamp
	 * @param int $until timestamp
	 * @param boolean $exportCreditNotes insead of invoices
	 * @return array response ready for json action helper
	 */
	public function varioExport($since, $until, $exportCreditNotes)
	{
		$response = [];

		$path = str_replace('\\', '/', PIMCORE_TEMPORARY_DIRECTORY);

		$orderList = new Order\Listing();
		$orderList->setOrderKey(($exportCreditNotes) ? 'creditNoteId' : 'invoiceId');
		$orderList->setOrder('desc');
		$db = \Pimcore\Db::get();
		$condition = '(fromStore IS NULL OR fromStore = 0) AND invoiceId IS NOT NULL ';
		$issuedOnColumn = ($exportCreditNotes) ? 'cnIssuedOn' : 'issuedOn';
		if ($exportCreditNotes) {
			$condition .= ' AND creditNoteId IS NOT NULL ';
		} else {
			$condition .= " AND (status = 'vyrizena' OR status = 'nevyzvednuto' OR status= 'vraceno') ";
		}
		$condition .= $db->quoteInto(' AND '.$issuedOnColumn.' >= ? ', $since);
		$condition .= $db->quoteInto(' AND '.$issuedOnColumn.' <= ? ', $until);
		$orderList->setCondition($condition);
		$orderList->load();

		if ($exportCreditNotes) {
			$invoices = 'Credit note ID ; Variable symbol ; Full name ; Street ; Zip code ; City ; Company ; IC ; DIC ; Payment type ; Issue date ; Due date; Invoice ID' . "\r\n";
			$invoiceItems = 'Credit note ID ; Item name ; Price ; Count ; Total price' . "\r\n";
		} else {
			$invoices = 'Invoice ID ; Variable symbol ; Full name ; Street ; Zip code ; City ; Company ; IC ; DIC ; Payment type ; Issue date ; Due date ; Storno' . "\r\n";
			$invoiceItems = 'Invoice ID ; Item name ; Price ; Count ; Total price' . "\r\n";
		}
		if ($orderList->count()) {
			foreach ($orderList->getItems(0, 0) as $order) {
				$paymentType = $order->getPaymentType();
				if (!$exportCreditNotes) {
					$issueDate = ($order->getIssuedOn()) ?  date('j.n.Y', $order->getIssuedOn()->getTimestamp()) : '';
					$dueDate = ($order->getDueTo()) ? date('j.n.Y', $order->getDueTo()->getTimestamp()) : '';
				} else {
					$issueDate = ($order->getCnIssuedOn()) ?  date('j.n.Y', $order->getCnIssuedOn()->getTimestamp()) : '';
					$dueDate = ($order->getCnDueTo()) ? date('j.n.Y', $order->getCnDueTo()->getTimestamp()) : '';
				}
				$fullName = str_replace('"', '', strip_tags($order->getFFullName()));
				$invoice = [
					($exportCreditNotes) ? $order->getCreditNoteId() : $order->getInvoiceId(),
					$order->getOrderId(),
					iconv('utf-8', 'windows-1250', $fullName),
					iconv('utf-8', 'windows-1250', $order->getFStreet()),
					$order->getFZipCode(),
					iconv('utf-8', 'windows-1250', $order->getFCity()),
					iconv('utf-8', 'windows-1250', $order->getFCompany()),
					$order->getFIco(),
					$order->getFDic(),
					$paymentType,
					$issueDate,
					$dueDate,
					($exportCreditNotes) ? $order->getInvoiceId() : (($order->getStatus() == self::STATUS_RETURNED) ? 1 : 0)
				];
				$invoices .= implode(";", $invoice) . "\r\n";
				foreach ($order->getProducts() as $product) {
					$count = ($exportCreditNotes) ? (int)$product->getReturned() : (int)$product->getCount();
					if ($exportCreditNotes && !$count) {
						continue;
					}
					$name = str_replace('"', '', strip_tags($product->getName()));
					$invoiceItem = [
						($exportCreditNotes) ? $order->getCreditNoteId() : $order->getInvoiceId(),
						iconv('utf-8', 'windows-1250', $name),
						number_format($product->getPrice(), 4, ',', ''),
						$count,
						number_format($product->getPrice() * $count, 4, ',', '')
					];
					$invoiceItems .= implode(";", $invoiceItem) . "\r\n";
				}
				$shipping = $order->getShippingPrice() - $order->getShippingDiscount() + $order->getPaymentPrice() - $order->getPaymentDiscount();
				if ($shipping) {
					$invoiceItem = [
						($exportCreditNotes) ? $order->getCreditNoteId() : $order->getInvoiceId(),
						iconv('utf-8', 'windows-1250', 'Poštovné'),
						number_format($shipping, 4, ',', ''),
						1,
						number_format($shipping, 4, ',', '')
					];
					$invoiceItems .= implode(";", $invoiceItem) . "\r\n";
				}
				$discount = $order->getProductsDiscount() + $order->getShippingDiscount() + $order->getPaymentDiscount();
				if ($discount) {
					$invoiceItem = [
						($exportCreditNotes) ? $order->getCreditNoteId() : $order->getInvoiceId(),
						'Sleva',
						number_format(-1 * $discount, 4, ',', ''),
						1,
						number_format(-1 * $discount, 4, ',', '')
					];
					$invoiceItems .= implode(";", $invoiceItem) . "\r\n";
				}
			}

			//save the CSVs as files
			$idk = ($exportCreditNotes) ? 'credit-note' : 'invoice';
			$idk2 = ($exportCreditNotes) ? 'Dobropisy' : 'Faktury';
			$fp = fopen($path.'/vario-export-'.$idk.'s.csv', 'w+');
			fwrite($fp, $invoices);
			fclose($fp);

			$fp = fopen($path.'/vario-export-'.$idk.'-items.csv', 'w+');
			fwrite($fp, $invoiceItems);
			fclose($fp);

			//create zip
			$zip = new ZipArchive();
			if ($zip->open($path.'/vario-export-'.$idk.'s.zip', ZipArchive::CREATE) === true) {
				$zip->addFile($path.'/vario-export-'.$idk.'s.csv', $idk2.'.csv');
				$zip->addFile($path.'/vario-export-'.$idk.'-items.csv', $idk2.'-polozky.csv');
				$zip->close();
			}

			unlink($path.'/vario-export-'.$idk.'s.csv');
			unlink($path.'/vario-export-'.$idk.'-items.csv');
		}

		$response['success'] = true;
		$response['count'] = $orderList->count();
		$response['creditNotes'] = $exportCreditNotes;
		$response['errorCode'] = -1;
		$response['errors'] = "";

		return $response;
	}

	public function mockOrder()
	{
		$lang = substr($this->translator->getLocale(), 0, 2);

		$order = new Order();
		$products = new \Pimcore\Model\Object\Fieldcollection();
		$productsList = new Product\Listing();
		$productsList->setLimit(2);
		$productsList->load();
		foreach ($productsList as $product) {
			$newProduct = new \Pimcore\Model\Object\Fieldcollection\Data\Orderproduct();
			$newProduct->setProduct($product);
			$newProduct->setName($product->getName($lang));
			$newProduct->setCount(1);
			$newProduct->setPrice(123);
			$products->add($newProduct);
		}
		$order->setValues([
			'status' => self::STATUS_NEW,
			'user' => \Pimcore\Model\Object\User::getByEmail('john.doe@example.com')->current(),
			'orderId' => '1234567890',
			'paymentType' => self::PAYMENT_BANK_TRANSFER,
			'shippingType' => self::SHIPPING_CZECH_PPL,
			'productsPrice' => '123',
			'productsDiscount' => '0',
			'shippingPrice' => '123',
			'baseShippingPrice' => '123',
			'shippingDiscount' => '0',
			'shippingCost' => '0',
			'paymentPrice' => '0',
			'basePaymentPrice' => '0',
			'paymentDiscount' => '0',
			'paymentCost' => '0',
			'finalPrice' => '692',
			'currency' => 'cs_CZ',
			'fFirstName' => 'John',
			'fLastName' => 'Doe',
			'fStreet' => 'Street',
			'fCity' => 'City',
			'fZipCode' => '12345',
			'fEmail' => 'john.doe@example.com',
			'fPhone' => '0987654321',
			'dFirstName' => 'John',
			'dLastName' => 'Doe',
			'dStreet' => 'Street',
			'dCity' => 'City',
			'dZipCode' => '12345',
			'products' => $products
		]);

		return $order;
	}

	/**
	 * @param type $country
	 * @param type $type
	 * @param type $productsPrice
	 * @param type $discountValue
	 * @return array (basePrice, price)
	 */
	public function getShippingPrices($country, $type, $productsPrice, $discountValue, $applyDiscountOnShipping)
	{
		$tmp = $this->getShippingAndPaymentTypePrices($productsPrice, $discountValue, $applyDiscountOnShipping);
		if (isset($tmp[$country][$type])) {
			return [$tmp[$country][$type]['basePrice'], $tmp[$country][$type]['price']];
		} else {
			\Pimcore\Log\Simple::log('exceptions', 'COULD NOT GET SHIPPING PRICES');
			throw new \Exception('Could not load shipping prices.');
		}
	}

	/**
	 * @param type $country
	 * @param type $shippingType
	 * @param type $type
	 * @param type $productsPrice
	 * @param type $discountValue
	 * @return type
	 */
	public function getPaymentPrices($country, $shippingType, $type, $productsPrice, $discountValue, $applyDiscountOnShipping)
	{
		$tmp = $this->getShippingAndPaymentTypePrices($productsPrice, $discountValue, $applyDiscountOnShipping);
		if (isset($tmp[$country][$shippingType]['paymentPrices'][$type])) {
			return [
				$tmp[$country][$shippingType]['paymentPrices'][$type]['basePrice'],
				$tmp[$country][$shippingType]['paymentPrices'][$type]['price'],
			];
		} else {
			\Pimcore\Log\Simple::log('exceptions', 'COULD NOT GET PAYMNET PRICES');
			throw new \Exception('Could not load payment prices.');
		}
	}
	/**
	 * methods returns actual shipping and payment prices after applying discounts
	 * if there is an overflow from an absolute value discount code, it also applies on the shipping
	 *
	 * @param type $productsPrice
	 * @param type $productsDiscountValue
	 * @return array
	 * @throws \Exception
	 */
	public function getShippingAndPaymentTypePrices($productsPrice, $discountValue = 0, $applyOnShipping = true)
	{
		$country = $this->currencyModel->getCountry();

		$eshopSettings = \Website\Tool\Utils::getEshopSettings();
		$shippingOptions = $eshopSettings->{'getShippingOptions' . $country}()->data;
		$paymentOptions = $eshopSettings->{'getPaymentOptions' . $country}()->data;
		$shippingDiscountOptions = $eshopSettings->{'getShippingDiscountOptions' . $country}()->data;

		$currentShippingDiscountStep = current($this->getShippingDiscountSteps($productsPrice, $shippingDiscountOptions));

		//calculate discountCodeOverflow value by which a granted discount exceeds the product price, works only with absolute discounts...
		if ($applyOnShipping) {
			$discountCodeOverflow = max([0, $discountValue - $productsPrice]);
		} else {
			$discountCodeOverflow = 0;
		}

		$prices = [];

		foreach ($shippingOptions as $country => $shippingTypes) {
			//because of Pimcore structured table keys lowering
			$countryUpper = strtoupper($country);
			$prices[$countryUpper] = [];
			$countryValid = false;
			foreach ($shippingTypes as $shippingType => $shippingPrice) {
				$shippingPrice = $this->currencyModel->localizeCurrency($shippingPrice, null, true);
				if ($shippingPrice >= 0) {
					$baseShippingPrice = $shippingPrice;
					//substract oveflow from base
					$discountCodeOverflowRemaining = max([0, $discountCodeOverflow - $shippingPrice]);
					$shippingPrice = max([0, $shippingPrice - $discountCodeOverflow]);
					//apply shipping discount
					$discountedShippingPrice = $this->applyShippingDiscount($shippingPrice, $currentShippingDiscountStep);

					$prices[$countryUpper][$shippingType] = [
						'basePrice' => $baseShippingPrice,
						'price' => $discountedShippingPrice,
						'paymentPrices' => []
					];

					foreach ($paymentOptions[$shippingType] as $paymentType => $paymentPrice) {
						$paymentPrice = $this->currencyModel->localizeCurrency($paymentPrice, null, true);
						if ($paymentPrice >= 0) {
							$basePaymentPrice = $paymentPrice;
							$countryValid = true;
							//substract remaining overflow from base
							$paymentPrice = max([0, $paymentPrice - $discountCodeOverflowRemaining]);
							$paymentOptions[$shippingType][$paymentType] = $shippingPrice;
							//apply payment discount
							$prices[$countryUpper][$shippingType]['paymentPrices'][$paymentType] = [
								'basePrice' => $basePaymentPrice,
								'price' => $this->applyShippingDiscount($paymentPrice, $currentShippingDiscountStep, $shippingPrice - $discountedShippingPrice)
							];
						}
					}
				}
			}

			if (!$countryValid) {
				unset($prices[$countryUpper]);
			}
		}

		if (empty($prices)) {
			throw new \Exception('No valid shipping and payment options set.');
			return;
		}

		return $prices;
	}

	/**
	 *
	 * @param string $code
	 * @param double $productsPrice
	 * @param boolean $noCheck disabled code validation
	 * @return mixed false on not valid code or [discountValue, discountedPrice, discountCodeObject]
	 */
	public function applyDiscountCode($code, $productsPrice, $noCheck = false, $country = null)
	{
		$country = ($country) ? $country : $this->currencyModel->getCountry();

		/* @var $discountCode DiscountCode */
		if (is_int($code)) {
			$discountCode = DiscountCode::getById($code);
		} else {
			$discountCode = DiscountCode::getByCode($code)->current();
		}
		if ($discountCode && ($noCheck || !$discountCode->getUsed() || $discountCode->getReusable())) {
			$time = time();
			if (!$noCheck && ($discountCode->getActiveSince()->getTimestamp() > $time || $discountCode->getActiveUntil()->getTimestamp() < $time)) {
				return false;
			} else {
				$discount = $discountCode->{'getDiscount' . $country}();
				if ($discountCode->getIsPercentual()) {
					$discountValue = $productsPrice / 100 * $discount;
				} else {
					$discountValue = $discountCode->{'getDiscount' . $country}();
				}
				return [$discountValue, max([0, $productsPrice - $discountValue]), $discountCode];
			}
		} else {
			return false;
		}
	}

	/**
	 * calculates the current achieved (if any) shipping discount step and the next possible (if any)
	 *
	 * @param double $productsPrice
	 * @param array $steps shipping discount options
	 * @return array [0] is achieved step, [1] is next step
	 */
	public function getShippingDiscountSteps($productsPrice, $steps = null)
	{
		$country = $this->currencyModel->getCountry();

		// TODO caching
		$current = null;
		$next = null;

		if ($steps === null) {
			$steps = \Website\Tool\Utils::getEshopSettings()->{'getShippingDiscountOptions' . $country}()->data;
		}

		//localize step prices
		foreach ($steps as $key => $data) {
			$steps[$key]['price'] = $this->currencyModel->localizeCurrency($data['price'], null, true);
			if (!$data['ispercentual']) {
				$steps[$key]['discount'] = $this->currencyModel->localizeCurrency($data['discount'], null, true);
			}
		}

		//current
		foreach ($steps as $data) {
			if ($data['price'] > 0) {
				if ($productsPrice >= $data['price']) {
					$current = $data;
				}
			}
		}

		//next
		foreach ($steps as $data) {
			if ($data['price'] > 0) {
				if ($current === null) {
					$next = $data;
					break;
				}
				if ($current !== null) {
					if ($productsPrice < $data['price'] && $current['price'] < $data['price']) {
						$next = $data;
						break;
					}
				}
			}
		}
		$tmp = [$current, $next];

		return $tmp;
	}

	/**
	 * shipping discount is applied on shipping price and also on the payment type fee and can be absolute or percentual
	 *
	 * @param double $base base price
	 * @param mixed $step array or null
	 * @param double $alreadyApplied already applied discount (because of seperated payment type prices)
	 * @return double discounted price
	 */
	private function applyShippingDiscount($base, $step, $alreadyApplied = 0)
	{
		if (!$step) {
			return round($base, 6);
		}

		$value = $step['discount'];
		$isPercentual = $step['ispercentual'];

		if (!$isPercentual) {
			return round(max([0, $base + $alreadyApplied - $value]), 6);
		} else {
			$percents = min(100, $value);
			return round($base - ($base / 100 * $percents), 6);
		}
	}
}
