<?php

namespace Website\Model\Templator;

use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Sunra\PhpSimple\HtmlDomParser;

class PortaPimcore
{
	protected $inputDir = null;
	protected $outputDir = null;

	protected $formIndexes = [];
	protected $snippetIndexes = [];
	protected $savedSnippetPaths = [];

	/**
	 * @var Filesystem
	 */
	protected $fs = null;

	/**
	 * @var Parser
	 */
	protected $parser = null;

	public function __construct($inputDir, $outputDir)
	{
		$this->outputDir = $outputDir;
		$this->inputDir = $inputDir;
		$this->copyStatic = $copyStaticFiles;
		$this->fs = new Filesystem;
		$this->parser = new Parser($this->inputDir);
	}

	public function run($copyStaticFiles = false, $phpMin = false)
	{
		try {
			$this->checkDirectories();

			if ($copyStaticFiles) {
				$this->copyStaticFiles();
			}

			$contentStack = $this->generateLayout($phpMin);

			$this->generateContent($contentStack);
		} catch (\Exceptionn $e) {
			return $e->getMessage();
		}

		return 0;
	}

	private function generateLayout($phpMin)
	{
		list($htmlAttribs, $bodyAttribs) = $this->parser->getHtmlBodyAttribs(
			['lang' => ['<?=$this->language;?>']]
		);

		list($css, $js, $inlineScripts, $externalCssAndJs) = $this->parser->getCssJsIncludes();

		$this->parser->removeScriptTags();

		//extract the layout markup, try to extract header, footer and forms from it
		$layoutTop = '';
		$layoutBottom = '';
		$specific = null;
		$mainStruct = [
			'stack' => [
				['selector' => 'body', 'root' => true]
			]
		];
		$contentStack = null;
		$action = Parser::ACTION_COPY_UNTIL_SPECIFIC_OR_FIRST_DIFF;
		$layoutPart = 'layoutTop';
		try {
			do {
				$restartCycle = false;
				end($mainStruct['stack']);
				$mainStruct['stack'][key($mainStruct['stack'])]['skip'] = true;
				$mainStruct['stack'][key($mainStruct['stack'])]['break'] = false;
				$mainStruct['stack'][key($mainStruct['stack'])]['visited'] = (boolean)$specific;
				$mainStruct = $this->parser->DFS(0, $mainStruct['stack'], $action, count($mainStruct['stack']));
				# dead end reached (more differences); try to backtrack
				if (isset($mainStruct['deadEnd'])) {
					fputs(STDOUT, sprintf("Reached dead end at:\n%s\nTrying to backtrack one level up...\n\n",
						$this->parser->printStackTrace($mainStruct['stack'], ($this->parser->templates[0]['crawler']))
					));
					# introduce a difference on upper level
					$upperLevelNode = $this->parser->find(
						$this->parser->templates[0]['crawler'],
						array_slice($mainStruct['stack'], 0, -1)
					);
					$upperLevelNode->class = ($upperLevelNode->class)
						? $upperLevelNode->class . ' dummy'
						: 'dummy';
					# reset stuff
					$layoutTop = '';
					$layoutBottom = '';
					$specific = false;
					$mainStruct = ['stack' => [['selector' => 'body', 'root' => true]]];
					$contentStack = null;
					$action = Parser::ACTION_COPY_UNTIL_SPECIFIC_OR_FIRST_DIFF;
					$layoutPart = 'layoutTop';
					# remove already saved snippet files
					foreach ($this->savedSnippetPaths as $path) {
						unlink($path);
					}
					# reset counters
					$this->formIndexes = [];
					$this->snippetIndexes = [];
					$this->savedSnippetPaths = [];
					# restart cycle
					$restartCycle = true;
					continue;
				}
				${$layoutPart} .= $mainStruct['output'];
				$specific = (isset($mainStruct['specific']))
					? $mainStruct['specific']
					: false;
				switch ($specific) {
					case 'navigation': case 'mobile-nav':
						$headerName = $this->genSnippetName('navigation');
						${$layoutPart} .= str_repeat($this->parser->indent, count($mainStruct['stack'])).sprintf('<?= $this->render(\'snippets/%s.php\'); ?>', $headerName)."\n";
						$this->saveSnippet(0, $mainStruct['stack'], '/website/views/layouts/snippets/'.$headerName.'.php', __DIR__.'/templates/pimcore/porta/navigation.phtml', 'navigation');
						break;
					case 'header': case 'header-wrapper':
						$headerName = $this->genSnippetName('header');
						${$layoutPart} .= str_repeat($this->parser->indent, count($mainStruct['stack'])).sprintf('<?= $this->render(\'snippets/%s.php\'); ?>', $headerName)."\n";
						$this->saveSnippet(0, $mainStruct['stack'], '/website/views/layouts/snippets/'.$headerName.'.php', __DIR__.'/templates/pimcore/porta/header.phtml', 'header');
						break;
					case 'footer': case 'footer-wrapper':
						$footerName = $this->genSnippetName('footer');
						${$layoutPart} .= str_repeat($this->parser->indent, count($mainStruct['stack'])).sprintf('<?= $this->render(\'snippets/%s.php\'); ?>', $footerName)."\n";
						$this->saveSnippet(0, $mainStruct['stack'], '/website/views/layouts/snippets/'.$footerName.'.php', __DIR__.'/templates/pimcore/porta/footer.phtml', 'footer');
						break;
					case 'form':
						$formName = $this->genSnippetName('form', 'layout');
						${$layoutPart} .= str_repeat($this->parser->indent, count($mainStruct['stack'])).sprintf('<?=$this->render(\'forms/%s.php\');?>', $formName)."\n";
						$this->saveSnippet(0, $mainStruct['stack'], '/website/views/scripts/forms/'.$formName.'.php', __DIR__.'/templates/pimcore/porta/form.phtml');
						break;
				}
				//switch to bottom part of layout
				if (!$contentStack && !$specific) {
					$contentStack = $mainStruct['stack'];
					$layoutPart = 'layoutBottom';
					$specific = true;
					$action = Parser::ACTION_COPY_UNTIL_SPECIFIC_OR_END_AND_BREAK_ON_DIFF;
				}
			} while ($specific || $restartCycle);
		} catch (TemplatorException $e) {
			fputs(STDOUT, $e->getMessage());
			exit;
		}

		//concat includes for template placeholders
		list($cssIncludes, $jsIncludes) = $this->generateIncludes($css, $js, $phpMin);
		$externalIncludes = '';
		foreach ($externalCssAndJs as $include) {
			$externalIncludes .= sprintf("%s%s\n", str_repeat($this->parser->indent, 2), $include);
		}
		$inlineScriptsIncludes = '';
		foreach ($inlineScripts as $inlineScript) {
			$inlineScriptsIncludes .= sprintf("%s%s", str_repeat($this->parser->indent, 2), $inlineScript)."\n";
		}

		//replace placeholders in layout template and save the layout
		$template = file_get_contents(__DIR__.'/templates/pimcore/porta/layout.phtml');
		$output = vsprintf($template, [
			implode(' ', $htmlAttribs),
			$cssIncludes,
			$jsIncludes,
			$externalIncludes,
			((count($bodyAttribs)) ? ' '.implode(' ', $bodyAttribs) : ''),
			$layoutTop,
			str_repeat($this->parser->indent, count($contentStack)),
			$layoutBottom,
			$inlineScriptsIncludes
		]);
		$h = fopen($this->outputDir.'/website/views/layouts/standard.php', 'w');
		fwrite($h, $output);
		fclose($h);

		//save some layout snippets, which do not change => just copy the templates
		$snippets = ['breadcrumbs-render', 'breadcrumbs', 'flash-messages', 'meta'];
		foreach ($snippets as $snippet) {
			$snippetTemplate = file_get_contents(__DIR__.'/templates/pimcore/porta/'.$snippet.'.phtml');
			$h = fopen($this->outputDir.'/website/views/layouts/snippets/'.$snippet.'.php', 'w');
			fwrite($h, $snippetTemplate);
			fclose($h);
		}

		return $contentStack;
	}

	private function generateContent($contentStack)
	{
		foreach (array_keys($this->parser->templates) as $key) {
			$path = '/website/views/scripts/cms/'.$this->parser->templates[$key]['name'].'.php';
			$this->saveSnippet($key, $contentStack, $path, __DIR__.'/templates/pimcore/porta/content.phtml', 'content');
		}
	}

	private function saveSnippet($templateIndex, $snippetStack, $path, $templatePath = null, $deep = false)
	{
		//deep checks for forms inside the DOM subtree
		if ($deep) {
			$content = '';
			$stack = $snippetStack;
			$continue = false;
			$i = 0;
			do {
				end($stack);
				$stack[key($stack)]['skip'] = $i;
				$stack[key($stack)]['break'] = !$i;
				$stack[key($stack)]['visited'] = $i;
				$result = $this->parser->DFS($templateIndex, $stack, Parser::ACTION_COPY_AND_BREAK_ON_FORM, count($stack) - count($snippetStack));
				$content .= $result['output'];
				$continue = false;
				$i += 1;
				if (!isset($result['finished']) || !$result['finished']) {
					$stack = $result['stack'];
					$formName = $this->genSnippetName('form', (($deep == 'content') ? $this->parser->templates[$templateIndex]['name'] : $deep));
					$content .= str_repeat($this->parser->indent, count($stack) - count($snippetStack)).sprintf('<?=$this->render(\'forms/%s.php\');?>', $formName)."\n";
					$this->saveSnippet($templateIndex, $stack, '/website/views/scripts/forms/'.$formName.'.php', __DIR__.'/templates/pimcore/porta/form.phtml');
					$continue = true;
				}
			} while ($continue);
		} else {
			end($snippetStack);
			$snippetStack[key($snippetStack)]['skip'] = false;
			$snippetStack[key($snippetStack)]['break'] = true;
			$snippetStack[key($snippetStack)]['visited'] = false;
			$tmp = $this->parser->DFS($templateIndex, $snippetStack, Parser::ACTION_COPY_TREE);
			$content = $tmp['output'];
		}

		$h = fopen($this->outputDir.$path, 'w');
		if ($templatePath) {
			$template = file_get_contents($templatePath);
			$content = vsprintf($template, $content);
		}
		fwrite($h, $content);
		fclose($h);
		$this->savedSnippetPaths[] = $this->outputDir.$path;
	}

	private function genSnippetName($type, $file = null)
	{
		if (!isset($this->snippetIndexes[$type])) {
			$this->snippetIndexes[$type] = 0;
		}

		$this->snippetIndexes[$type] += 1;

		$snippetName = $type;
		if ($file) {
			$snippetName .= '-' . $file;
		}
		if ($this->snippetIndexes[$type] > 1) {
			$snippetName .= sprintf('-%d', $this->snippetIndexes[$type]);
		}

		return $snippetName;
	}

	private function checkDirectories()
	{
		if (!$this->fs->exists($this->inputDir)) {
			throw new \Exception(sprintf('Input directory [%s] does not exist.',
				$this->inputDir
			));
		}
		if (!$this->fs->exists($this->outputDir)) {
			$this->fs->mkdir($this->outputDir);
		}
		//test write permissions
		$tmpDir = $this->outputDir.'/tmp';
		if (!$this->fs->exists($tmpDir)) {
			$this->fs->mkdir($tmpDir);
			$this->fs->remove($tmpDir);
		}
		//some structure dirs
		if (!$this->fs->exists($this->outputDir.'/website/views/scripts/cms')) {
			$this->fs->mkdir($this->outputDir.'/website/views/scripts/cms');
		}
		if (!$this->fs->exists($this->outputDir.'/website/views/scripts/forms')) {
			$this->fs->mkdir($this->outputDir.'/website/views/scripts/forms');
		}
		if (!$this->fs->exists($this->outputDir.'/website/views/layouts/snippets')) {
			$this->fs->mkdir($this->outputDir.'/website/views/layouts/snippets');
		}
	}

	private function copyStaticFiles()
	{
		//copy static files
		if ($this->fs->exists($this->inputDir.'/static')) {
			$this->fs->mirror($this->inputDir.'/static', $this->outputDir.'/static');
		}
		if (!$this->fs->exists($this->outputDir.'/static')) {
			$this->fs->mkdir($this->outputDir.'/static');
		}
		$staticDirs = ['css', 'scss', 'sass', 'images', 'img', 'js', 'font', 'fonts', 'svg'];
		foreach ($staticDirs as $staticDir) {
			if ($this->fs->exists($this->inputDir.'/'.$staticDir)) {
				$this->fs->mirror(
					$this->inputDir.'/'.$staticDir,
					$this->outputDir.'/static/'.$staticDir
				);
			}
		}
		//copy frontend stack configuration
		$configs = ['package.json', 'Gruntfile.js', 'bower.json', 'gulpfile.js'];
		foreach ($configs as $config) {
			if ($this->fs->exists($this->inputDir.'/'.$config)) {
				$this->fs->copy($this->inputDir.'/'.$config, $this->outputDir.'/'.$config);
			}
		}
	}

	private function generateIncludes($css, $js, $phpMin = false)
	{
		$cssIncludes = '';
		$jsIncludes = '';
		if ($phpMin) {
			// css
			$cssIncludes .= "<?\n";
			$cssIncludes .= sprintf("%secho \$this->minifyHeadLink(PIMCORE_DEBUG)\n",
				str_repeat($this->parser->indent, 3)
			);
			foreach ($css as $i => $cssPath) {
				$cssIncludes .= sprintf("%s->appendStylesheet('%s', 'screen')\n",
					str_repeat($this->parser->indent, 4),
					$cssPath
				);
			}
			$cssIncludes .= sprintf("%s->setVersion('1');\n",
				str_repeat($this->parser->indent, 4)
			);
			$cssIncludes .= sprintf("%s?>",
				str_repeat($this->parser->indent, 2)
			);

			// js
			$jsIncludes .= "<?\n";
			$jsIncludes .= sprintf("%secho \$this->minifyHeadScript(PIMCORE_DEBUG)\n",
				str_repeat($this->parser->indent, 3)
			);
			foreach ($js as $i => $jsPath) {
				$jsIncludes .= sprintf("%s->appendFile('%s')\n",
					str_repeat($this->parser->indent, 4),
					$jsPath
				);
			}
			$jsIncludes .= sprintf("%s->setVersion('1');\n",
				str_repeat($this->parser->indent, 4)
			);
			$jsIncludes .= sprintf("%s?>",
				str_repeat($this->parser->indent, 2)
			);
		} else {
			$debugSwitch = '<?if (!PIMCORE_DEBUG) echo \'.min\';?>';

			// css
			foreach ($css as $i => $cssPath) {
				$start = -4;
				$length = 0;
				if (substr($cssPath, -8, 4) == '.min') {
					$start = -8;
					$length = 4;
				}
				$cssIncludes .= sprintf('%s<link href="%s" media="screen" rel="stylesheet" type="text/css">%s',
					($i > 0) ? str_repeat($this->parser->indent, 2) : '',
					substr_replace($cssPath, $debugSwitch, $start, $length),
					($i == (count($css) - 1)) ? "" : "\n"
				);
			}

			// js
			foreach ($js as $i => $jsPath) {
				$start = -3;
				$length = 0;
				if (substr($jsPath, -7, 4) == '.min') {
					$start = -7;
					$length = 4;
				}
				$jsIncludes .= sprintf('%s<script type="text/javascript" src="%s"></script>%s',
					($i > 0) ? str_repeat($this->parser->indent, 2) : '',
					substr_replace($jsPath, $debugSwitch, $start, $length),
					($i == (count($js) - 1)) ? "" : "\n"
				);
			}
		}

		return [$cssIncludes, $jsIncludes];
	}
}
