<?php

/**
 * Cart handling class, persists to session and to cookies.
 * All data is synchronized with the DB on each request using the recalcuate() method.
 * 
 * @author Martin Kuric <martin.kuric@portadesign.cz>
 */

namespace Website\Model;

class Cart implements \Countable
{

	protected $language = null;
	/**
	 *
	 * @var \Pimcore\Translate\Website
	 */
	protected $translator = null;
	/**
	 *
	 * @var OrderManager 
	 */
	protected $shopModel = null;
	protected $order = null;

	/**
	 * @var Currency 
	 */
	protected $currencyModel = null;

	/**
	 * holds order ID in session after successful order creation to pair it for payment purposes
	 * @var int
	 */
	protected $orderId = null;

	/**
	 * for product Pimcore objects, which should not be stored in the session
	 * @var array
	 */
	protected $productObjects = [];

	/**
	 * @var DiscountCode discount code object
	 */
	protected $productsDiscount = null;

	/**
	 * @var array current and next shipping discount steps
	 */
	protected $shippingDiscountSteps = null;

	/**
	 * for product comparing purposes (might be moved from session to cookies
	 * @var array 
	 */
	protected $compare = [];
	protected $lastViewed = [];
	protected $sessionNamespace;

	const MAX_COMPARED_ITEMS = 3;
	const MAX_LAST_VIEWED_ITEMS = 4;
	//out of stock constants
	const OUT_OF_STOCK_IGNORE = 'ignore';
	const OUT_OF_STOCK_BLOCK_PURCHASE = 'block';
	const OUT_OF_STOCK_INFORM = 'inform';

	public function __construct($translator, $currencyModel, $shopModel, $noAutoMessages = false)
	{
		$this->language = substr($translator->getLocale(), 0, 2);
		$this->translator = $translator;
		$this->currencyModel = $currencyModel;
		$this->shopModel = $shopModel;

		$this->productsDiscount = new \stdClass();
		$this->productsDiscount->code = null;
		$this->productsDiscount->discountedPrice = 0;
		$this->productsDiscount->discountValue = 0;

		$this->loadSession();

		$this->recalculate($noAutoMessages);
	}

	public function setSessionNs(\Zend_Session_Namespace $ns)
	{
		$this->sessionNamespace = $ns;
	}

	public function getSessionNs()
	{
		if (null === $this->sessionNamespace) {
			$this->setSessionNs(new \Zend_Session_Namespace(__CLASS__));
		}
		return $this->sessionNamespace;
	}

	public function loadSession()
	{
		if (isset($this->getSessionNs()->order)) {
			$this->order = $this->getSessionNs()->order;
		} else {
			$this->order = new \stdClass();
			$this->order->country = $this->currencyModel->getCountry();
			$this->order->products = [];
			$this->order->shippingPrice = 0;
			$this->order->baseShippingPrice = 0;
			$this->order->paymentPrice = 0;
			$this->order->basePaymentPrice = 0;
			$this->order->productsPrice = 0;
			$this->order->contactInfo = [];
			$this->order->accepted = false;
			//set default shiping type
			$this->order->shippingType = \Website\Model\OrderManager::SHIPPING_PICK_UP;
			$shippingOptions = (array)current(\Website\Tool\Utils::getEshopSettings()->{'getShippingOptions'.$this->currencyModel->getCountry()}()->getData());
			foreach($shippingOptions as $key => $value) {
				if ($value >= 0) {
					$this->order->shippingType = $key;
					break;
				}
			}
			//set default payment type
			$this->order->paymentType = \Website\Model\OrderManager::PAYMENT_CASH;
			$paymentOptions = \Website\Tool\Utils::getEshopSettings()->{'getPaymentOptions'.$this->currencyModel->getCountry()}()->getData();
			foreach($paymentOptions[$this->order->shippingType] as $key => $value) {
				if ($value >= 0) {
					$this->order->paymentType = $key;
					break;
				}
			}
			$this->order->discountCode = null;
		}

		$this->orderId = $this->getSessionNs()->orderId;
		$this->compare = (array)$this->getSessionNs()->compare;
		$this->lastViewed = (array)$this->getSessionNs()->lastViewed;
	}

	public function persist()
	{
		//save order to session
		$this->getSessionNs()->order = $this->order;
		$this->getSessionNs()->orderId = $this->orderId;
		$this->getSessionNs()->compare = $this->compare;
		$this->getSessionNs()->lastViewed = $this->lastViewed;
	}

	public function addItem($id, $count = 1)
	{
		$count = (int) $count;
		if ($count < 1)
			$count = 1;

		$countAlreadyInCart = isset($this->order->products[$id]->count) ? $this->order->products[$id]->count : 0;

		if (!array_key_exists($id, $this->order->products)) {
			$this->order->products[$id] = new \stdClass();
			$this->order->products[$id]->lastCount = 0;
		}

		$this->order->products[$id]->count = $count + $countAlreadyInCart;

		$this->order->accepted = false;

		//this starts "a new order" after a previously completed one which we keep in session for various reasons (e.g. e-commerce)
		$this->setOrderId(null);

		$this->recalculate();

		return $count + $countAlreadyInCart;
	}

	public function updateItem($id, $count)
	{
		if (isset($this->order->products[$id])) {
			$this->order->products[$id]->count = $count;

			$this->order->accepted = false;

			//$this->persist();
			$this->setOrderId(null); //NOTICE test this
			$this->recalculate();
		}

		return true;
	}

	public function setDiscountCode($code)
	{
		$this->order->discountCode = $code;

		$this->persist();

		return true;
	}

	public function setPaymentType($type)
	{
		$this->order->paymentType = $type;

		$this->order->accepted = false;

		$this->persist();

		return true;
	}

	public function setShippingType($type)
	{
		$this->order->shippingType = $type;

		$this->order->accepted = false;

		$this->persist();

		return true;
	}

	public function setCountry($country)
	{
		$this->order->country = $country;

		$this->order->accepted = false;

		$this->persist();

		return true;
	}

	/**
	 * 
	 * @param array $data contact info data from \Website\Form\CartContactForm
	 * @return boolean
	 */
	public function setContactInfo($data)
	{
		$this->order->contactInfo = $data;

		$this->order->accepted = false;

		$this->persist();

		return true;
	}

	/**
	 * this method is called on each request so the notification messages are popped also on server changes...
	 */
	public function recalculate($noAutoMessages = false)
	{
		$flashMessenger = \Zend_Controller_Action_HelperBroker::getStaticHelper('FlashMessenger');
		$this->order->productsPrice = 0;

		foreach ($this->order->products as $id => $product) {

			//check if product exists and is not a master product
			$productObject = Product::getById($id);
			//we turn off the inheritance, because on the master product flag would otherwise be inherited, since we do not explicitly set a false value
			\Website\Tool\Utils::forceInheritanceAndFallbackValues(false);
			if (!$productObject || $productObject->getIsMasterProduct()) {
				unset($this->order->products[$id]);
				unset($this->productObjects[$id]);
				if (!$noAutoMessages) {
					$flashMessenger->setNamespace('info')
						->addMessage(sprintf($this->translator->translate('msg_cart_item_removed_id'), $id));
				}
				continue;
			}
			\Website\Tool\Utils::restorInheritanceAndFallbackValues();

			//remove product if count is set to 0
			$count = $product->count;
			if ($count < 1) {
				unset($this->order->products[$id]);
				unset($this->productObjects[$id]);
				if ($product->lastCount > 0) {
					if (!$noAutoMessages && $productObject instanceof Product) {
						$flashMessenger->setNamespace('success')
								->addMessage(sprintf($this->translator->translate('msg_cart_item_removed'), $productObject->getVariantName($this->language)));
					} else {
						$flashMessenger->setNamespace('info')
								->addMessage(sprintf($this->translator->translate('msg_cart_item_removed_id'), $id));
					}
				}
				continue;
			}

			//check the stock following the outOfStockBehaviour settings
			$inStock = $productObject->getInStock();
			$eshopSettings = \Website\Tool\Utils::getEshopSettings();
			$outOfStockBehaviour = $eshopSettings->getOutOfStockBehaviour();
			if ($outOfStockBehaviour == self::OUT_OF_STOCK_BLOCK_PURCHASE && ($count) > $inStock) {
				if (!$noAutoMessages) {
					$flashMessenger->setNamespace('error')
							->addMessage(sprintf($this->translator->translate('msg_out_of_stock_block'), $productObject->getVariantName($this->language), $inStock));
				}
				$this->order->products[$id]->count = $inStock;
				$this->order->products[$id]->lastCount = $inStock;
			} else {
				if (!$noAutoMessages) {
					if ($outOfStockBehaviour == self::OUT_OF_STOCK_INFORM && ($count) > $inStock && $count != $product->lastCount) {
						$flashMessenger->setNamespace('info')
								->addMessage(sprintf($this->translator->translate('msg_out_of_stock_inform'), $productObject->getVariantName($this->language), $inStock));
					}
					//check if product was updated or added and inform about it
					if ($product->lastCount == 0) {
						$flashMessenger->setNamespace('success')
								->addMessage(sprintf($this->translator->translate('msg_cart_item_added_to_cart'), $productObject->getVariantName($this->language)));
					} elseif ($product->lastCount != $count) {
						$flashMessenger->setNamespace('success')
								->addMessage(sprintf($this->translator->translate('msg_cart_item_updated'), $productObject->getVariantName($this->language), $product->count));
					}
				}
				$this->order->products[$id]->lastCount = $product->count;
			}

			//update price
			$this->productObjects[$id] = $productObject;
			list(, $discountedPrice) = $this->currencyModel->getProductPrices($productObject);
			$this->order->productsPrice += $discountedPrice * $product->count;
			if ($outOfStockBehaviour == self::OUT_OF_STOCK_BLOCK_PURCHASE && $inStock == 0) {
				unset($this->productObjects[$id]);
				unset($this->order->products[$id]);
			}

		}

		//apply discount code
		if ($this->order->discountCode !== null) {
			$tmp = $this->shopModel->applyDiscountCode($this->order->discountCode, $this->order->productsPrice, (boolean)$this->orderId);
			if ($tmp != false) {
				list($this->productsDiscount->discountValue, $this->productsDiscount->discountedPrice, $this->productsDiscount->code) = $tmp;
			} else {
				$this->productsDiscount->code = null;
				$this->order->discountCode = null;
			}
		}

		//should we apply the possible discount overflow on shipping too?
		$applyDiscountOnShipping =($this->order->discountCode && $this->productsDiscount->code->getApplyOnShipping()) ? true : false;

		//calculate the shipping and payment prices; check for free shipping settings and apply discount overflow if needed
		try {
			list($this->order->baseShippingPrice, $this->order->shippingPrice) = $this->shopModel->getShippingPrices(
				$this->order->country,
				$this->order->shippingType,
				$this->order->productsPrice,
				$this->productsDiscount->discountValue,
				$applyDiscountOnShipping
			);
			list($this->order->basePaymentPrice, $this->order->paymentPrice) = $this->shopModel->getPaymentPrices(
				$this->order->country,
				$this->order->shippingType,
				$this->order->paymentType,
				$this->order->productsPrice,
				$this->productsDiscount->discountValue,
				$applyDiscountOnShipping
			);
		} catch (\Exception $e) {
			$this->clear();
		}

		$this->persist();
	}

	public function addToComparison($productId)
	{
		if (count($this->compare) >= self::MAX_COMPARED_ITEMS) {
			return ['status' => -1, 'key' => 'msg_product_comparison_full', 'args' => [self::MAX_COMPARED_ITEMS]];
		}

		$product = Product::getById($productId);

		if (!$product instanceof Product) {
			return ['status' => -1, 'key' => 'msg_product_does_not_exist'];
		}

		if (!in_array($productId, $this->compare)) {
			$this->compare[] = $productId;

			$this->persist();

			return ['status' => 1, 'key' => 'msg_product_added_to_comparison', 'args' => [$product->getVariantName($this->language)]];
		} else {
			return ['status' => 0, 'key' => 'msg_product_already_in_comparison', 'args' => [$product->getVariantName($this->language)]];
		}
	}

	public function removeFromComparison($productId)
	{
		$key = array_search($productId, $this->compare);

		if ($key !== false) {
			unset($this->compare[$key]);

			$this->persist();

			return true;
		}

		return false;
	}

	public function getComparedProducts()
	{
		$products = [];
		//products to compare
		foreach ($this->compare as $key => $id) {
			$tmp = Product::getById($id);

			if (!$tmp || !$tmp->isPublished()) {
				unset($this->compare[$key]);
				continue;
			}

			$products[] = $tmp;
		}

		$this->persist();

		return $products;
	}

	public function addToLastViewed($productId)
	{
		if (!in_array($productId, $this->lastViewed)) {
			array_unshift($this->lastViewed, $productId);

			if (isset($this->lastViewed[self::MAX_LAST_VIEWED_ITEMS])) {
				unset($this->lastViewed[self::MAX_LAST_VIEWED_ITEMS]);
			}
		}

		$this->persist();
	}

	public function getLastViewedProducts()
	{
		$search = ElasticSearch::getInstance($this->currencyModel, $this->language, '');

		return $search->getLastViewedProducts($this->lastViewed);
	}

	public function accept()
	{
		$this->order->accepted = true;
		$this->persist();
	}

	public function isAccepted()
	{
		return $this->order->accepted;
	}

	public function getOrder()
	{
		return $this->order;
	}

	public function getCompare()
	{
		return $this->compare;
	}

	public function getProductObjects()
	{
		return $this->productObjects;
	}

	public function getProductsDiscount()
	{
		return $this->productsDiscount;
	}

	public function setOrderId($id)
	{
		$this->orderId = $id;

		$this->persist();
	}

	public function getOrderId()
	{
		return $this->orderId;
	}

	//do not clear orderId
	public function clear()
	{
		$this->order = null;
		$this->productObjects = [];
		$this->productsDiscount = null;
		$this->persist();
	}

	public function clearCompareAndLastViewed()
	{
		$this->compare = [];
		$this->lastViewed = [];
	}

	public function count()
	{
		$count = 0;

		if (isset($this->order->products)) {
			foreach ($this->order->products as $product)
				$count += $product->count;
		}

		return $count;
	}

}
