<?php

/**
 * currency wrapper, loads currency configurations from administration
 * supports automatic conversions from base to current currency
 * conversion can be based on exchange service or manual.
 */
namespace Website\Model;

class Currency
{
	/**
	 * currency used in the request.
	 *
	 * @var \Zend_Currency
	 */
	protected $currency = null;

	/**
	 * currency used in the request, but frozen (exchange rate = 1); used for product prices,
	 * because they are already calculated in the backend because of promotion sales...
	 *
	 * @var \Zend_Currency
	 */
	protected $frozenCurrency = null;

	/**
	 * this is the currency the prices are stored in.
	 *
	 * @var \Zend_Currency
	 */
	protected $baseCurrency = null;

	/**
	 * system currencies; \Zend_Currency objects.
	 *
	 * @var array
	 */
	protected $systemCurrencies = null;

	/**
	 * @param string $locale locale with language and country code e.g.(cs_CZ)
	 */
	public function __construct($locale = false)
	{
		$systemCurrenciesOptions = $this->getSystemCurrenciesOptions(true);

		if (!$systemCurrenciesOptions) {
			throw new \Exception('No system currencies set.');

			return;
		}

		$baseLocale = $this->getBaseLocale();
		$tmp = new \Zend_Currency(null, $baseLocale); //just because we need the
		//set-up currencies
		foreach ($systemCurrenciesOptions as $currencyLocale => $currencyOptions) {
			$currency = new \Zend_Currency($currencyOptions, $currencyLocale);
			//add exchange service
			if ($currencyLocale != $baseLocale) {
				$exchangeService = new ExchangeService($tmp->getShortName(), $systemCurrenciesOptions);
				$currency->setService($exchangeService);
			}
			$this->systemCurrencies[$currencyLocale] = $currency;
		}

		//link base, current and create current-frozen currencies
		$this->baseCurrency = $this->systemCurrencies[$baseLocale];
		if ($locale && isset($this->systemCurrencies[$locale])) {
			$this->currency = $this->systemCurrencies[$locale];
		} else {
			$this->currency = $this->baseCurrency;
		}
		$this->frozenCurrency = $this->createFrozenCurrency($this->currency->getLocale());

		\Zend_Registry::set('\Zend_Currency', $this->currency);
	}

	/**
	 * @var float|\Zend_Currency
	 * @var \Zend_Currency       $currency
	 *
	 * @return \Zend_Currency
	 */
	public function getCurrency($value = null, \Zend_Currency $currency = null)
	{
		if (!$currency) {
			$currency = $this->currency;
		}
		if ($value === null) {
			return $currency;
		} else {
			return $currency->setValue($value);
		}
	}

	/**
	 * @return \Zend_Currency
	 */
	public function getBaseCurrency($value = null)
	{
		if ($value === null) {
			return $this->baseCurrency;
		} else {
			return $this->baseCurrency->setValue($value);
		}
	}

	/**
	 * parses the eshop settings exchange rate table for (active) currencies.
	 *
	 * @var bool
	 *
	 * @return array associative array with locale as keys and some currency info array as values
	 */
	public function getSystemCurrenciesOptions($alsoInactive = false)
	{
		$currencies = [];

		$config = \Website\Tool\Utils::getEshopSettings();

		foreach ($config->getExchangeRate()->data as $locale => $data) {
			$localeParts = explode('_', $locale);
			if (($data['isactive'] || $alsoInactive) && count($localeParts) == 2) {
				//fix cs_CZ precision
				if ($locale == 'cs_cz') {
					$data['precision'] = 0;
				}
				//strtoupper because on structured table data component row keys Pimcore converts upper case to lower case
				$currencies[$localeParts[0].'_'.strtoupper($localeParts[1])] = $data;
			}
		}

		return $currencies;
	}

	/**
	 * @return array of \Zend_Currency objects (keys are locales)
	 */
	public function getSystemCurrencies()
	{
		return $this->systemCurrencies;
	}

	/**
	 * @param string $locale locale
	 *
	 * @return array currency options for \Zend_Currency constructor
	 */
	public function getCurrencyOptions($locale)
	{
		$systemCurrenciesOptions = $this->getSystemCurrenciesOptions(true);

		return isset($systemCurrenciesOptions[$locale]) ? $systemCurrenciesOptions[$locale] : null;
	}

	/**
	 * create a currency with options from backend, but conversion rate 1
	 * used for localizing hard values like saved order price.
	 *
	 * @param string $locale the locale string
	 *
	 * @return \Zend_Currency
	 */
	public function createFrozenCurrency($locale)
	{
		$currency = new \Zend_Currency($this->getCurrencyOptions($locale), $locale);
		$exchangeRate = new ExchangeService(null, null, true);
		$currency->setService($exchangeRate);

		return $currency;
	}

	/**
	 * @return string country from locale string of active currency
	 */
	public function getCountry()
	{
		return substr($this->currency->getLocale(), 3);
	}

	/**
	 * returns products price and discounted price after applying some global modifications to it;
	 * passed product may be and Product or its elastic search related search result.
	 *
	 * @param Product|array $product
	 *
	 * @return array
	 */
	public function getProductPrices($product)
	{
		$country = $this->getCountry();

		//switch based on product type (object vs array - from elastic search)
		if ($product instanceof Product) {
			$price = $product->{'getPrice'.$country}();
			$discountedPrice = $product->{'getDiscountedPrice'.$country}();
		} else {
			$price = $product['price'.$country];
			$discountedPrice = $product['discountedPrice'.$country];
		}

		$price = \Website\Tool\Utils::applyGlobalPriceChange($price);
		$discountedPrice = \Website\Tool\Utils::applyGlobalPriceChange($discountedPrice);

		return [$price, $discountedPrice];
	}

	/**
	 * localizes the base prices from base currency to current currency.
	 *
	 * @param float $value         the price
	 * @param mixed $magicCurrency could be NULL => current currency will be used; or TRUE => current frozen currency will be used or \Zend_Currency..
	 * @param bool  $onlyValue     only the raw value
	 * @param array $options       additional \Zend_Currency options
	 *
	 * @return type
	 */
	public function localizeCurrency($baseValue, $magicCurrency = null, $onlyValue = false, $options = [])
	{
		if ($magicCurrency === true) {
			$magicCurrency = $this->getFrozenCurrency();
		}

		if ($onlyValue) {
			return $this->getCurrency($this->getBaseCurrency($baseValue), $magicCurrency)->getValue();
		} else {
			return $this->getCurrency($this->getBaseCurrency($baseValue), $magicCurrency)->toCurrency(null, $options);
		}
	}

	/**
	 * @return string base locale
	 */
	public function getBaseLocale()
	{
		$systemCurrenciesOptions = $this->getSystemCurrenciesOptions(true);

		foreach ($systemCurrenciesOptions as $locale => $data) {
			if ($data['isdefault']) {
				return $locale;
			}
		}

		return key($systemCurrenciesOptions);
	}

	/**
	 * @return \Zend_Currency
	 */
	public function getFrozenCurrency()
	{
		return $this->frozenCurrency;
	}
}
