#!/bin/bash
#NOTICE: watch for line endings if you are using windows or mac, use LF only
#NOTICE: this script uses --login-path=deployment for remote mysql access

###Load remote conf
source ./website/tools/Deployment/remote.conf
if [ -z "${HOSTNAME}" ]; then
	echo 'Hostname not set - check your remote.conf'
	exit;
fi
if [ -z "${HOSTUSERNAME}" ]; then
	echo 'Host username not set - check your remote.conf'
	exit;
fi
if [ -z "${HOSTGROUP}" ]; then
	echo 'Host group not set - check your remote.conf'
	exit;
fi
if [ -z "${APACHEUSER}" ]; then
	echo 'Apache user not set - check your remote.conf'
	exit;
fi
if [ -z "${HOSTROOT}" ]; then
	echo 'Host root not set - check your remote.conf'
	exit;
fi
if [ -z "${WEBNAME}" ]; then
	echo 'Web name not set - check your remote.conf'
	exit;
fi
if [ -z "${VHOSTCONFDIR}" ]; then
	echo 'Virtual host directory not set - check your remote.conf'
	exit;
fi

WEBROOT="$HOSTROOT/$WEBNAME/html"

###Check git
echo "Checking remote Git remote..."
GIT_ERR=$(git ls-remote 2>&1 >/dev/null);
if [[ $GIT_ERR == *fatal:* ]]; then
	echo $GIT_ERR;
	exit;
fi
echo "It's OK"

###Sync only git sources
if [ "$1" = "-g" ]; then
	echo "Checking local Git remote..."
	GIT_ERR=$(git ls-remote 2>&1 >/dev/null);
	if [[ $GIT_ERR == *fatal:* ]]; then
		echo $GIT_ERR;
		exit;
	fi
	echo "It's OK"

	echo -e "\n\nAdding local changes to Git..."
	git add -A
	git commit -u
	git push -u origin --all
	git push -u origin --tags
	echo "Push complete"

	echo -e "\n\nPulling Git changes..."
	ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && git pull"
	echo "Pull complete"

	echo -e "\n\nSaving definitions..."
	ssh $HOSTUSERNAME@$HOSTNAME "sudo chown -R $APACHEUSER:$HOSTGROUP $WEBROOT/website/var;sudo chmod -R 775 $WEBROOT/website/var;"
	ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && php website/tools/Deployment/save-definitions.php"

	echo -e "\n\nDone."
	exit;
fi


###create temp dir
mkdir ./deployment_tmp 2>/dev/null


###Check remote mysql access
MYSQL_ERR=$(ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment -e \"GRANT USAGE ON *.* TO '$REMOTEMYSQLUSER'@'localhost';\"" 2>&1 >/dev/null);
if [ ! -z "${MYSQL_ERR}" ]; then
	echo -e "\nRemote mysql access denied, exiting...\n"
	exit;
fi

###Create remote DB, user and
echo -e "\n\nCreating remote database and user..."
MYSQL_ERR=$(ssh $HOSTUSERNAME@$HOSTNAME "mysql  --login-path=deployment -e \"USE $WEBNAME;\"" 2>&1 >/dev/null);
if [ -z "${MYSQL_ERR}" ]; then
	read -p "Database \"$WEBNAME\" already exists, do you want to REMOVE IT and continue? (y/n) " CONT
	if [ "$CONT" != "y" ]; then
		echo "Exiting..."
		exit;
	else
		ssh $HOSTUSERNAME@$HOSTNAME "mysqladmin --login-path=deployment -f drop $WEBNAME"
	fi
fi
MYSQL_RESULT=$(ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment -e \"SELECT 1 FROM mysql.user WHERE user = '$WEBNAME';\"" 2>&1);
if [ ! -z "${MYSQL_RESULT}" ]; then
	read -p "Database user \"$WEBNAME\" already exists, do you want to REMOVE IT and continue? (y/n) " CONT
	if [ "$CONT" != "y" ]; then
		echo "Exiting..."
		exit;
	else
		ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment -e \"drop user '$WEBNAME'@'localhost';\""
		echo "Database user \"$WEBNAME\" removed".
	fi
fi

REMOTEMSQLUSERPASSWORD=$(cat /dev/urandom | tr -dc 'a-zA-Z0-9' | head -c 16);
ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment -e \"CREATE DATABASE $WEBNAME CHARACTER SET utf8 COLLATE utf8_general_ci;\""
ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment -e \"CREATE USER '$WEBNAME'@'localhost' identified by '$REMOTEMSQLUSERPASSWORD';\"";
ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment -e \"GRANT ALL PRIVILEGES on $WEBNAME.* to '$WEBNAME'@'localhost';\"";
ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment -e \"GRANT SELECT,SUPER on *.* to '$WEBNAME'@'localhost';\"";
echo "Remote database \"$WEBNAME\" and user \"$WEBNAME\" created"

###Import local DB to remote
echo -e "\n\nImporting local database to remote..."
#extract local mysql credentials
MYSQLUSER=$(cat ./website/var/config/system.xml | tr -d '[\n\t ]' | sed -e "s/.*host><username>\(.*\)<\/username><password>.*<\/password><dbname>.*/\1/")
MYSQLPASSWORD=$(cat ./website/var/config/system.xml | tr -d '[\n\t ]' | sed -e "s/.*<\/username><password>\(.*\)<\/password><dbname>.*/\1/")
DBNAME=$(cat ./website/var/config/system.xml | tr -d '[\n\t ]' | sed -e "s/.*<dbname>\(.*\)<\/dbname>.*/\1/")
mysqldump -u$MYSQLUSER -p$MYSQLPASSWORD $DBNAME > ./deployment_tmp/tmpdb.sql 2>/dev/null
#sed replace the MySQL DEFINER to avoid errors
cat ./deployment_tmp/tmpdb.sql | sed "s/\`$MYSQLUSER\`/\`$WEBNAME\`/g" > ./deployment_tmp/tmpdb2.sql
#sed replace the IP to localhost 
cat ./deployment_tmp/tmpdb2.sql | sed "s/\`127.0.0.1\`/\`localhost\`/g" > ./deployment_tmp/db.sql
scp ./deployment_tmp/db.sql $HOSTUSERNAME@$HOSTNAME:~/db.sql
ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment $WEBNAME < ~/db.sql";
echo "Remote DB imported"

###Do the deployment push to git repo
echo -e "\n\nPushing repo, this might take a while..."
git add -A
git commit -u -m "deployment"
git push -u origin --all
git push -u origin --tags
echo "Repo push complete"

###Create remote directory structure
echo -e "\n\nCreating remote directory structure..."
if (ssh $HOSTUSERNAME@$HOSTNAME "[ -d $HOSTROOT/$WEBNAME ]"); then
	read -p "Directory \"$HOSTROOT/$WEBNAME\" already exists. Do you want to REMOVE IT and continue? (y/n) " CONT
	if [ "$CONT" != "y" ]; then
		echo "Exiting..."
		exit;
	else
		ssh $HOSTUSERNAME@$HOSTNAME "sudo rm -rf $HOSTROOT/$WEBNAME"
	fi
fi
ssh $HOSTUSERNAME@$HOSTNAME "sudo mkdir $HOSTROOT/$WEBNAME;sudo mkdir $WEBROOT;sudo mkdir $HOSTROOT/$WEBNAME/tmp;sudo mkdir $HOSTROOT/$WEBNAME/logs"
ssh $HOSTUSERNAME@$HOSTNAME "sudo chown -R $APACHEUSER:$HOSTGROUP $HOSTROOT/$WEBNAME;sudo chmod -R 775 $HOSTROOT/$WEBNAME"
echo "Remote directory structure built"

###Pull the repo on remote host
GITREPO=$(awk -F "=" '/url/ {print $2}' ./.git/config);
echo -e "\n\nCloning git repo on remote machine, this might take a while..."
ssh $HOSTUSERNAME@$HOSTNAME "git clone $GITREPO $WEBROOT"
ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && git config core.filemode false"
echo "Cloning complete"

###Copy Pimcore files, which are not tracked with Git
echo -e "\n\nSynchronizing the FS..."
#Sub the mysql credentials in system.xml
sed -e "s/<username>$MYSQLUSER<\/username>/<username>$WEBNAME<\/username>/" -e "s/<password>$MYSQLPASSWORD<\/password>/<password>$REMOTEMSQLUSERPASSWORD<\/password>/" -e "s/<dbname>$DBNAME<\/dbname>/<dbname>$WEBNAME<\/dbname>/" ./website/var/config/system.xml > ./deployment_tmp/system.xml 
#create some var subdirectories, just to be sure they exist
ssh $HOSTUSERNAME@$HOSTNAME "mkdir -p $WEBROOT/website/var/backup; mkdir -p $WEBROOT/website/var/search; mkdir -p $WEBROOT/website/var/cache; mkdir -p $WEBROOT/website/var/tmp; mkdir -p $WEBROOT/website/var/assets; mkdir -p $WEBROOT/website/var/email; mkdir -p $WEBROOT/website/var/classes/Object; mkdir -p $WEBROOT/website/var/versions;"
#copy stuff
scp ./.htaccess $HOSTUSERNAME@$HOSTNAME:$WEBROOT/
scp ./deployment_tmp/system.xml $HOSTUSERNAME@$HOSTNAME:$WEBROOT/website/var/config/system.xml
scp ./website/var/config/cache.xml $HOSTUSERNAME@$HOSTNAME:$WEBROOT/website/var/config/cache.xml
tar -czf ./deployment_tmp/classes.tar.gz -C ./website/var/classes/Object .
scp ./deployment_tmp/classes.tar.gz $HOSTUSERNAME@$HOSTNAME:~/classes.tar.gz
ssh $HOSTUSERNAME@$HOSTNAME "tar -xzf ~/classes.tar.gz -C $WEBROOT/website/var/classes/Object/"
tar -czf ./deployment_tmp/assets.tar.gz -C ./website/var/assets .
scp ./deployment_tmp/assets.tar.gz $HOSTUSERNAME@$HOSTNAME:~/assets.tar.gz
ssh $HOSTUSERNAME@$HOSTNAME "tar -xzf ~/assets.tar.gz -C $WEBROOT/website/var/assets/"
tar -czf ./deployment_tmp/emails.tar.gz -C ./website/var/email .
scp ./deployment_tmp/emails.tar.gz $HOSTUSERNAME@$HOSTNAME:~/emails.tar.gz
ssh $HOSTUSERNAME@$HOSTNAME "tar -xzf ~/emails.tar.gz -C $WEBROOT/website/var/email/"
tar -czf ./deployment_tmp/versions.tar.gz -C ./website/var/versions .
scp ./deployment_tmp/versions.tar.gz $HOSTUSERNAME@$HOSTNAME:~/versions.tar.gz
ssh $HOSTUSERNAME@$HOSTNAME "tar -xzf ~/versions.tar.gz -C $WEBROOT/website/var/versions/"
#composer
ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && composer install"
#set rights
ssh $HOSTUSERNAME@$HOSTNAME "sudo chown -R $APACHEUSER:$HOSTGROUP $WEBROOT; sudo chmod -R 775 $WEBROOT"
echo "FS synchronized"

###Set up remote apache vhost
echo -e "\n\nSetting up apache vhost..."
ssh $HOSTUSERNAME@$HOSTNAME "test -e $VHOSTCONFDIR/$WEBNAME.vhost"
if [ $? -ne 0 ]; then
	cat "./website/tools/Deployment/vhost_skeleton" | sed "s|###SITE_NAME###|$WEBNAME|g" | sed "s|###HOST_ROOT###|$HOSTROOT|g" > ./deployment_tmp/$WEBNAME.vhost
	scp ./deployment_tmp/$WEBNAME.vhost $HOSTUSERNAME@$HOSTNAME:~/$WEBNAME.vhost
	ssh $HOSTUSERNAME@$HOSTNAME "sudo mv ~/$WEBNAME.vhost $VHOSTCONFDIR/$WEBNAME.vhost"
	if [[ $VHOSTCONFDIR == *sites-available* ]]; then
		VHOSTENABLEDALIAS=$(echo $VHOSTCONFDIR | sed -e "s/sites-available/sites-enabled\/$WEBNAME.vhost/");
		ssh $HOSTUSERNAME@$HOSTNAME "sudo ln -s $VHOSTCONFDIR/$WEBNAME.vhost $VHOSTENABLEDALIAS";
	fi
	echo "Vhost set up"
else
	echo "Vhost already exists"
fi

###Cleanup (remote and locally)
echo -e "\n\nCleaning up..."
rm -rf ./deployment_tmp
ssh $HOSTUSERNAME@$HOSTNAME "rm ~/db.sql; rm ~/assets.tar.gz; rm ~/classes.tar.gz; rm ~/emails.tar.gz; rm ~/versions.tar.gz;"
echo "Done."

###Reload remote web server
echo -e "\n\n";
read -p "Would you like to reload the remote web server service? (y/n) " CONT
if [ "$CONT" == "y" ]; then
	ssh $HOSTUSERNAME@$HOSTNAME "sudo service apache2 reload;"
	echo "Service reloaded. Web should be running."
fi

echo -e "\n\nDeployment complete. You might want to remove the git password from remote git config. And maybe tweak the .htaccess."
echo -e "\nDo not forget to set up pimcore maintenance cron job."
