<?php

/**
 * simple class to help with elastic searching.
 *
 * @author Martin Kuric <martin.kuric@portadesign.cz>
 */
namespace Website\Model;

class ElasticSearch
{
	private static $instances;

	/**
	 * @var \Elasticsearch\Client
	 */
	protected $elasticClient = null;
	/**
	 * @var \Website\Model\Currency
	 */
	protected $currencyModel = null;
	protected $index = null;
	protected $language = 'cs';
	protected $urlPrefix = null;

	/**
	 * @param string $language
	 * @param string $urlPrefix
	 *
	 * @return ElasticSearch
	 */
	public static function getInstance(\Website\Model\Currency $currencyModel, $language, $urlPrefix = null)
	{
		$key = $language.$urlPrefix;
		if (empty(self::$instances[$key])) {
			self::$instances[$key] = new self($currencyModel, $language, $urlPrefix);
		}

		return self::$instances[$key];
	}

	public function __construct(\Website\Model\Currency $currencyModel, $language, $urlPrefix = null)
	{
		$this->currencyModel = $currencyModel;
		$this->language = $language;
		$this->elasticClient = \Website\Tool\ElasticSearch::getClient();
		$this->index = \Website\Tool\ElasticSearch::getMainIndexName(false);
		$this->urlPrefix = $urlPrefix;

		if (!$this->elasticClient->indices()->existsAlias(['name' => $this->index])) {
			throw new \Exception('Elastic search main index not found.');
		}
	}

	public function getProducts($path, $filter = [])
	{
		$country = $this->currencyModel->getCountry();
		$showVariantsInList = (\Website\Tool\Utils::getEshopSettings()->getShowVariantsInLists());

		/*	PREPARE QUERY BASE	*/

		$filteredQuery = $this->queryBase('product_'.$this->language, true, $showVariantsInList);
		//category path restriction
		if (!empty($path)) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['searchPaths'] = $path;
		}
		//fulltext query param
		if (!empty($filter['query'])) {
			$query = [];
			$query['multi_match']['query'] = $filter['query'];
			$query['multi_match']['fields'] = ['name^3', 'description'];
			$filteredQuery['body']['query']['filtered']['query'] = $query;
		} else {
			$filteredQuery['body']['query']['filtered']['query']['match_all'] = [];
		}

		/*	PREPARE FILTER PARTS	*/

		$filterParts = [];
		//brands
		if (!empty($filter['brand'])) {
			$filterParts['brands'] = ['terms' => ['brand' => $filter['brand']]];
		}
		//colors
		if (!empty($filter['color'])) {
			if ($showVariantsInList) {
				$filterParts['colors'] = ['terms' => ['color' => $filter['color']]];
			} else {
				$filterParts['colors'] = ['terms' => ['colors' => $filter['color']]];
			}
		}
		//sizes
		if (!empty($filter['size'])) {
			if ($showVariantsInList) {
				$filterParts['sizes'] = ['terms' => ['sizes' => $filter['size']]];
			} else {
				//@TODO complete this after multi-level variant attributes propagation finished
			}
		}
		//price ranges
		if (!empty($filter['price'])) {
			$priceFilter = [];
			$priceBuckets = \Website\Tool\Utils::getFilterPriceBuckets($country, true);
			foreach ($filter['price'] as $key) {
				if (!empty($priceBuckets[$key])) {
					$priceFilter['or'][]['range']['discountedPrice'.$country] = $priceBuckets[$key];
				}
			}
			if (!empty($priceFilter['or'])) {
				$filterParts['prices'] = $priceFilter;
			}
		}
		//new
		if (!empty($filter['new'])) {
			$filterParts['new'] = ['term' => ['isNew' => true]];
		}
		//on sale
		if (!empty($filter['sale'])) {
			$filterParts['sale'] = ['term' => ['isOnSale' => true]];
		}

		/* AGGREGATIONS */

		$aggregations = [];
		list($aggregations['priceMin'], $aggregations['priceMax']) = $this->priceRangeAggregation($filteredQuery);
		$aggregations['price'] = $this->priceBucketsAggregation($filteredQuery, $filterParts, $country);
		$aggregations['color'] = $this->colorsAggregation($filteredQuery, $filterParts, $showVariantsInList);
		$aggregations['size'] = $this->sizesAggregation($filteredQuery, $filterParts, $showVariantsInList);
		$aggregations['brand'] = $this->brandsAggregation($filteredQuery, $filterParts);
		$aggregations += $this->singleFieldAggregations($filteredQuery, $filterParts);

		/*	PAGINATION, SORT, FINAL QUERY	*/

		//add filter parts to base query
		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'] = array_merge($filteredQuery['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		//get total count
		$countQuery = ['index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => ['query' => $filteredQuery['body']['query']]];
		$countResult = $this->elasticClient->count($countQuery);
		$totalCount = $countResult['count'];
		//switch to folded fulltext analyzer if no results
		if (!empty($filter['query']) && !$totalCount) {
			$filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'] = ['name.folded^3', 'description.folded'];
			$countQuery = ['index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => ['query' => $filteredQuery['body']['query']]];
			$countResult = $this->elasticClient->count($countQuery);
			$totalCount = $countResult['count'];
		}
		//pagination
		$page = !empty($filter['page']) ? intval($filter['page']) : 1;
		$limit = !empty($filter['limit'])
			? intval($filter['limit'])
			: \Website\Tool\Utils::getEshopSettings()->getProductPageLimit();
		if ((int) $page > ceil($totalCount / $limit)) {
			$page = ceil($totalCount / $limit);
		}
		if ($page < 1) {
			$page = 1;
		}
		$filteredQuery['body']['from'] = ($page - 1) * $limit;
		$filteredQuery['body']['size'] = $limit;
		//only sort when no searchQuery is set and some results matched...
		if ($totalCount && empty($filter['query'])) {
			$filteredQuery['body']['sort'][]['ordering'] = 'asc';
		}
		//query
		$resultSet = $this->elasticClient->search($filteredQuery);
		$resultSet['aggregations'] = $aggregations;

		return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
	}

	public function suggest($query, $size = 10)
	{
		$suggestQuery = [
			'index' => $this->index,
			'body' => [
				'suggestions' => [
					'text' => $query,
					'completion' => ['field' => 'name_suggest', 'size' => $size],
				],
			],
		];
		//$this->elasticClient
		$suggestions = [];
		$resultSet = $this->elasticClient->suggest($suggestQuery);
		foreach ($resultSet['suggestions'][0]['options'] as $result) {
			$suggestions[] = $result['text'];
		}

		return $suggestions;
	}

	public function getMostSoldProducts($limit = 3)
	{
		$filteredQuery = $this->queryBase('product_'.$this->language);

		$filteredQuery['body']['query']['filtered']['query']['match_all'] = [];

		$filteredQuery['body']['from'] = 0;
		$filteredQuery['body']['size'] = $limit;
		$filteredQuery['body']['sort'][]['soldCount'] = 'desc';

		$resultSet = $this->elasticClient->search($filteredQuery);

		return $this->constructSimplePagination($resultSet, $limit, 1, 0);
	}

	//@TODO multi level variants fix
	public function getRelatedProducts($path, $id, $limit = 3)
	{
		$functionScore = $this->queryBase('product_'.$this->language);

		$functionScore['body']['query']['filtered']['query']['match_all'] = [];
		$functionScore['body']['query']['function_score']['query'] = $functionScore['body']['query'];
		$functionScore['body']['query']['function_score']['functions'][0]['random_score'] = new \stdClass();
		unset($functionScore['body']['query']['filtered']);

		$functionScore['body']['query']['function_score']['query']['filtered']['filter']['bool']['must_not'][]['term']['id'] = $id;
		if ($path) {
			$functionScore['body']['query']['function_score']['query']['filtered']['filter']['bool']['must'][]['term']['searchPaths'] = $path;
		}

		$functionScore['body']['from'] = 0;
		$functionScore['body']['size'] = $limit;

		$resultSet = $this->elasticClient->search($functionScore);

		return $this->constructSimplePagination($resultSet, $limit, 1, 0);
	}

	public function getLastViewedProducts($idList)
	{
		if (empty($idList)) {
			return [];
		}

		$filteredQuery = $this->queryBase('product_'.$this->language);

		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['ids'] = [
			'type' => $filteredQuery['type'],
			'values' => $idList,
		];

		$resultSet = $this->elasticClient->search($filteredQuery);

		return $resultSet['hits']['hits'];
		//return $this->constructSimplePagination($resultSet['hits']['hits'], 999, 1, 0);
	}

	public function getVariants($productId)
	{
		$filteredQuery['index'] = $this->index;
		$filteredQuery['type'] = 'product_'.$this->language;

		$filter = [];
		$filter['bool']['must'][0]['term']['published'] = true;
		$filter['bool']['must'][1]['term']['isVariant'] = true;
		$filter['bool']['must'][2]['term']['parentId'] = (int) $productId;

		$filteredQuery['body']['query']['filtered']['filter'] = $filter;
		$filteredQuery['body']['from'] = 0;
		$filteredQuery['body']['size'] = 100;

		$resultSet = $this->elasticClient->search($filteredQuery);

		//@TODO multi-level support
		$hits = $resultSet['hits']['hits'];
		foreach ($hits as $key => $hit) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][2]['term']['parentId'] = $hit['_source']['id'];
			$resultSet = $this->elasticClient->search($filteredQuery);
			if (!empty($resultSet['hits']['hits'])) {
				unset($hits[$key]);
				foreach ($resultSet['hits']['hits'] as $tmpHit) {
					$hits[] = $tmpHit;
				}
			}
		}

		return $hits;
		//return $this->constructSimplePagination($resultSet['hits']['hits'], 999, 1, 0);
	}

	public function fulltextSearch($query, $page = 1, $limit = 10, $onlyPublished = true)
	{
		if (empty($query)) {
			return $this->constructSimplePagination([], $limit, $page, 0);
		}

		$types = [
			\Website\Tool\ElasticSearch::DOCUMENT_FULLTEXT_TYPE_KEY.'_'.$this->language,
			\Website\Tool\ElasticSearch::OBJECT_FULLTEXT_TYPE_KEY.'_'.$this->language,
		];
		$filteredQuery['index'] = $this->index;
		$filteredQuery['type'] = $types;

		$filter = [];
		if ($onlyPublished) {
			$filter['bool']['must'][]['term']['published'] = true;
		}
		$filteredQuery['body']['query']['filtered']['filter'] = $filter;
		if (!empty($query)) {
			$multiMatchQuery = [];
			$multiMatchQuery['multi_match']['query'] = $query;
			$multiMatchQuery['multi_match']['fields'] = ['title^4', 'keywords^3', 'description^2', 'content'];
			$filteredQuery['body']['query']['filtered']['query'] = $multiMatchQuery;
		} else {
			$filteredQuery['body']['query']['filtered']['query']['match_all'] = [];
		}

		//count
		$countQuery = ['index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => ['query' => $filteredQuery['body']['query']]];
		$countResult = $this->elasticClient->count($countQuery);
		$totalCount = $countResult['count'];
		//switch to folded fulltext analyzer if no results
		if (!$totalCount) {
			$filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'] = ['title.folded^4', 'keywords.folded^3', 'description.folded^2', 'content.folded'];
			$countQuery = ['index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => ['query' => $filteredQuery['body']['query']]];
			$countResult = $this->elasticClient->count($countQuery);
			$totalCount = $countResult['count'];
		}
		//pagination
		if ((int) $page > ceil($totalCount / $limit)) {
			$page = ceil($totalCount / $limit);
		}
		$page = (int) max([1, $page]);
		$filteredQuery['body']['from'] = ($page - 1) * $limit;
		$filteredQuery['body']['size'] = $limit;

		$resultSet = $this->elasticClient->search($filteredQuery);

		return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
	}

	/**
	 * @param type $published
	 * @param type $isMaserProduct
	 *
	 * @return \Elastica\Query\Bool
	 */
	private function queryBase($type, $published = true, $showVariantsInList = false)
	{
		$query['index'] = $this->index;
		$query['type'] = $type;

		$filter = [];
		$filter['bool']['must'][]['term']['published'] = $published;
		if ($showVariantsInList) {
			$tmpCount = count($filter['bool']['must']);
			//is variant and is not hidden (like size)
			$filter['bool']['must'][$tmpCount]['or'][0]['and'][]['term']['isVariant'] = true;
			$filter['bool']['must'][$tmpCount]['or'][0]['and'][]['term']['size'] = 0;
			//is master product and has no visible variants (like color)
			$filter['bool']['must'][$tmpCount]['or'][1]['and'][]['term']['isMasterProduct'] = true;
			$filter['bool']['must'][$tmpCount]['or'][1]['and'][]['missing']['field'] = 'colors';
			//is no master product nor variant
			$filter['bool']['must'][$tmpCount]['or'][2]['and'][]['term']['isVariant'] = false;
			$filter['bool']['must'][$tmpCount]['or'][2]['and'][]['term']['isMasterProduct'] = false;
		} else {
			$filter['bool']['must'][]['term']['isVariant'] = false;
		}

		$query['body']['query']['filtered']['filter'] = $filter;

		return $query;
	}

	private function priceRangeAggregation($query)
	{
		$aggQuery = [];
		$aggQuery['min_price']['min']['field'] = 'discountedPrice';
		$aggQuery['max_price']['max']['field'] = 'discountedPrice';
		$query['body']['aggs'] = $aggQuery;
		$result = $this->elasticClient->search($query);

		return [$result['aggregations']['min_price']['value'], $result['aggregations']['max_price']['value']];
	}

	private function priceBucketsAggregation($query, $filterParts, $country)
	{
		unset($filterParts['prices']);

		$priceBuckets = \Website\Tool\Utils::getFilterPriceBuckets($country);
		$aggQuery = [];
		$aggQuery['price']['range']['field'] = 'discountedPrice'.$country;
		$aggQuery['price']['range']['ranges'] = $priceBuckets;

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$prices = [];
		foreach ($result['aggregations']['price']['buckets'] as $key => $bucket) {
			$prices[$key] = $bucket['doc_count'];
		}

		return $prices;
	}

	private function colorsAggregation($query, $filterParts, $showVariantsInList)
	{
		unset($filterParts['colors']);

		//colors
		$aggQuery = [];
		$aggQuery['color']['terms']['size'] = 100;
		if ($showVariantsInList) {
			$aggQuery['color']['terms']['field'] = 'color';
		} else {
			$aggQuery['color']['terms']['field'] = 'colors';
		}

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$colors = [];
		foreach ($result['aggregations']['color']['buckets'] as $bucket) {
			if ($bucket['key']) {
				$colors[$bucket['key']] = $bucket['doc_count'];
			}
		}

		return $colors;
	}

	private function sizesAggregation($query, $filterParts, $showVariantsInList)
	{
		$tmp = !empty($filterParts['sizes']) ? $filterParts['sizes'] : null;
		unset($filterParts['sizes']);

		//sizes
		$aggQuery = [];
		if ($showVariantsInList) {
			$aggQuery['size']['terms']['size'] = 100;
			$aggQuery['size']['terms']['field'] = 'sizes';
		} else {
			//@TODO complete this after multi-level variant attributes propagation finished
		}

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		if ($tmp) {
			$query['body']['query']['filtered']['filter']['bool']['must_not'][] = $tmp;
		}
		$result = $this->elasticClient->search($query);

		$sizes = [];
		if (!empty($result['aggregations']['size'])) {
			foreach ($result['aggregations']['size']['buckets'] as $bucket) {
				if ($bucket['key']) {
					$sizes[$bucket['key']] = $bucket['doc_count'];
				}
			}
		}

		return $sizes;
	}

	private function brandsAggregation($query, $filterParts)
	{
		unset($filterParts['brands']);
		//brands
		$aggQuery = [];
		$aggQuery['brand']['terms']['size'] = 100;
		$aggQuery['brand']['terms']['field'] = 'brand';

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$brands = [];
		foreach ($result['aggregations']['brand']['buckets'] as $bucket) {
			$brands[$bucket['key']] = $bucket['doc_count'];
		}

		return $brands;
	}

	private function singleFieldAggregations($query, $filterParts)
	{
		$aggQuery = [];
		//news
		$aggQuery['new']['filter']['term']['isNew'] = true;
		//on sale
		$aggQuery['sale']['filter']['term']['isOnSale'] = true;

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		return [
			'new' => $result['aggregations']['new']['doc_count'],
			'sale' => $result['aggregations']['sale']['doc_count'],
		];
	}

	private function constructSimplePagination($resultSet, $limit, $page, $totalCount)
	{
		$pagination = new \stdClass();
		$pagination->current = $page;
		$pagination->pageCount = ($limit) ? (int) ceil($totalCount / $limit) : 1;
		if ($page > 1) {
			$pagination->previous = $page - 1;
		}
		if ($page < $pagination->pageCount) {
			$pagination->next = $page + 1;
		}
		$pagination->totalCount = $totalCount;
		$pagination->first = 1;
		$pagination->last = $pagination->pageCount;
		$pagination->items = empty($resultSet) ? [] : $resultSet['hits']['hits'];
		$pagination->aggregations = !empty($resultSet['aggregations']) ? $resultSet['aggregations'] : [];
		$pagination->urlprefix = $this->urlPrefix;

		return $pagination;
	}

	private function query($filteredQuery, $filter, $foldedFallback = true, $paginate = true, $highlighting = false)
	{
		//add folded fallback fields
		$fulltextFields = [];
		if ($foldedFallback && !empty($filter['query'])) {
			$fulltextFields = $filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'];
			$fields = [];
			foreach ($fulltextFields as $field) {
				$parts = explode('^', $field);
				$foldedField = $parts[0].'.folded';
				if (!empty($parts[1])) {
					$foldedField .= '^'.$parts[1];
				}
				$fields[] = $field;
				$fields[] = $foldedField;
			}
			$filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'] = $fields;
		}

		//get total count
		$countQuery = ['index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => ['query' => $filteredQuery['body']['query']]];
		$countResult = $this->elasticClient->count($countQuery);
		$totalCount = $countResult['count'];

		//sort, only when no searchQuery is set and some results matched and some sort is defined...
		if ($totalCount && empty($filter['query']) && !empty($filter['sort'])
		&& empty($filter['ids_score'])) {
			$sortStruct = $filter['sort'];
			if (is_array($sortStruct)) {
				foreach ($sortStruct as $key => $sort) {
					$filteredQuery['body']['sort'][][$key] = $sort;
				}
			} elseif (stristr($sortStruct, ':')) {
				$gps = explode(':', $sortStruct);
				$filteredQuery['body']['sort'][]['_geo_distance'] = [
					'location' => str_replace(':', ',', $sortStruct),
					'order' => 'asc',
					'unit' => 'km',
					'distance_type' => 'plane',
				];
			} else {
				//TODO define sorting types
			}
		}

		//add highlighting
		if ($highlighting && $fulltextFields && !empty($filter['query'])) {
			$highlight = ['fields' => []];
			foreach ($fulltextFields as $field) {
				$parts = explode('^', $field);
				$highlight['fields'][$parts[0]] = [
					'matched_fields' => [$parts[0]],
					'type' => 'fvh',
				];
				if ($foldedFallback) {
					$highlight['fields'][$parts[0]]['matched_fields'][] = $parts[0].'.folded';
				}
			}
			$filteredQuery['body']['highlight'] = $highlight;
		}

		//pagination
		$limit = !empty($filter['limit']) ? intval($filter['limit']) : 999;
		if ($paginate && $limit) {
			$page = !empty($filter['page']) ? intval($filter['page']) : 1;
			if ((int) $page > ceil($totalCount / $limit)) {
				$page = ceil($totalCount / $limit);
			}
			if ($page < 1) {
				$page = 1;
			}
			$filteredQuery['body']['from'] = isset($filter['offset']) ? ($page - 1) * $limit : 0;
			$filteredQuery['body']['size'] = isset($filter['offset']) ? $limit : $page * $limit;
		}
		//query
		$resultSet = $this->elasticClient->search($filteredQuery);
		if ($paginate) {
			return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
		} else {
			return $resultSet;
		}
	}
}
