<?php

use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

$serversConfigPath = __DIR__.DIRECTORY_SEPARATOR.'config.php';
if (!file_exists($serversConfigPath)) {
	fwrite(STDERR, sprintf("Looks like %s is missing.\n", $serversConfigPath));
	exit;
}
require_once __DIR__.DIRECTORY_SEPARATOR.'config.php';
require_once __DIR__.'/../../../pimcore/config/startup.php';

/*
 * Common parameters.
 */
set('local_web_directory', realpath(__DIR__.'/../../../'));
set('local_tmp_directory', get('local_web_directory').DIRECTORY_SEPARATOR.'deployment_tmp');

/*
 * Environment vars
 */
env('sudo', 'sudo ');
env('last_sync_stamp', 0);
env('last_db_sync_stamp', 0);

/*
 * Binaries
 */

/*
 * Default arguments and options.
 */
argument(
	'stage',
	InputArgument::OPTIONAL,
	'Run tasks only on this stage (may be multiple servers) or server.',
	get('default_stage')
);
option('all', 'a', InputOption::VALUE_NONE, 'Full deployment or sync.');
option('db', null, InputOption::VALUE_NONE, 'Deploy or sync the DB.');

/*
 * Tasks
 */
task('prepare', function () {
	$options = parseOptions();

	pwrite('Checking local requirements');
	$systemConfigPath = get('local_web_directory').'/website/var/config/system.php';
	if (file_exists($systemConfigPath)) {
		$lastSync = \Pimcore\Model\Tool\TmpStore::get('deployer_last_sync_'.env('server.name'));
		if ($lastSync) {
			env('last_sync_stamp', (int) $lastSync->getData());
		}
		$lastDbSync = \Pimcore\Model\Tool\TmpStore::get('deployer_last_db_sync_'.env('server.name'));
		if ($lastDbSync) {
			env('last_db_sync_stamp', (int) $lastDbSync->getData());
		}
	}
	if (!get('web_name')) {
		throw new \RuntimeException('Looks like the web_name parameter is not set');
	}
	if (input()->getArgument('command') == 'deployment:deploy' && ($options->all || $options->db)) {
		if (!file_exists($systemConfigPath)) {
			throw new \RuntimeException('Looks like the local system config is missing.');
		}
	}
	binariesCheck(true);
	if ($options->all || $options->db) {
		mysqlAccessCheck(true);
	}
	gitCheck(true);
	success();

	pwrite('Checking server requirements');
	binariesCheck();
	if ($options->all || $options->db || !file_exists($systemConfigPath)) {
		mysqlAccessCheck();
		if (input()->getArgument('command') == 'deployment:deploy') {
			mysqlDbAndUserCheck(false, $options->all, $options->db);
		} elseif (input()->getArgument('command') == 'deployment:sync') {
			mysqlDbAndUserCheck(false, false, true);
		}
	}
	if (input()->getArgument('command') == 'deployment:deploy' && $options->all) {
		serverDirMissing(env('hosting_path'), true);
		serverDirExists(getServerProjectPath(), true);
		if (env()->has('vhosts_path')) {
			serverDirMissing(env('vhosts_path'), true);
			serverFileExists(getServerVhostPath(), true);
		}
		if (env()->has('vhosts_symlink_path')) {
			serverDirMissing(env('vhosts_symlink_path'), true);
			serverFileExists(getServerVhostPath(true), true);
		}
	} elseif (input()->getArgument('command') == 'deployment:sync') {
		serverDirMissing(getServerProjectPath(), true);
	}
	createLocalDirectory(get('local_tmp_directory'));
	if (!$options->all || input()->getArgument('command') == 'deployment:sync') {
		createServerDirectory(getServerProjectPath(true, true));
	}
	if ((input()->getArgument('command') == 'deployment:sync' && !$options->db)
			|| !$options->all) {
		gitCheck();
	}
	success();
})
->desc('Check local and server requirements.')
->setPrivate();

task('deploy:vhost', function () {
	$options = parseOptions();
	if (!$options->all) {
		return;
	}

	pwrite('Creating server directory structure');
	createServerDirectory(getServerProjectPath(true, true));
	createServerDirectory(getServerProjectPath().'/logs');
	createServerDirectory(getServerProjectPath().'/tmp');
	run(sprintf('ln -s %s %s', getServerProjectPath(true), getServerProjectPath().'/www'));
	setServerPermissions(getServerProjectPath());
	success();

	if (env()->has('vhosts_path') && env()->has('vhosts_config_template')) {
		pwriteln('Creating server apache vhost config');
		$tmpVhostPath = getServerProjectPath(true, true).'/'.get('web_name').'.vhost';
		createServerFile(
			$tmpVhostPath,
			env('vhosts_config_template')
		);
		run(sprintf('{{sudo}}mv %s %s', $tmpVhostPath, getServerVhostPath()));
		if (env()->has('vhosts_symlink_path')) {
			run(sprintf('{{sudo}}ln -s %s %s', getServerVhostPath(), getServerVhostPath(true)));
		}
		success();
	}
})
->desc('Create an Apache vhost and web directory structure.')
->setPrivate();

task('deploy:db', function () {
	$options = parseOptions();
	if (!$options->all && !$options->db) {
		return;
	}

	if ($options->db && env('last_db_sync_stamp')) {
		$changes = createMigrationFile(env('last_db_sync_stamp'));
		if ($changes) {
			pwriteln('Detected DB changes in these tables:');
			foreach ($changes['db'] as $table => $change) {
				pwriteln(sprintf('<comment>%s</comment> - %d new entries, %s updated entries',
					$table,
					isset($change['new']) ? $change['new'] : 0,
					isset($change['updated']) ? $change['updated'] : 0
				));
			}
			pwriteln();
			pwriteln(sprintf(
				'You can check the generated migration file <comment>%s</comment> for more details.',
				$changes['migrationFilePath']
			));
			$confirmation = paskConfirmation('Do you want to deploy these changes to server?');
			if ($confirmation) {
				$serverMigrationPath = getServerProjectPath(true).'/website/var/tmp/'
					.basename($changes['migrationFilePath']);
				pupload(
					$changes['migrationFilePath'],
					$serverMigrationPath
				);
				pwrite('Applying migration DB changes');
				cd(getServerProjectPath(true));
				run(sprintf('{{sudo}}-u {{http_user}} php %s', $serverMigrationPath));
				run(sprintf('rm -rf %s', $serverMigrationPath));
				success();
			}
		}
	} else {
		pwrite('Creating server MySQL database and user');
		$isMysql57 = version_compare(getMysqlVersion(), '5.7.0', '>=');
		$password = generateRandomString(16);
		set('server_mysql_password', $password);
		run(sprintf(
			'{{bin/mysql}} -e "CREATE DATABASE %s CHARACTER SET utf8 COLLATE utf8_general_ci;"',
			get('web_name')
		));
		if ($isMysql57) {
			run(sprintf(
				'{{bin/mysql}} -e "CREATE USER IF NOT EXISTS \'%s\'@\'localhost\';'
					.'SET PASSWORD FOR \'%s\'@\'localhost\' = PASSWORD(\'%s\');"',
				get('web_name'),
				get('web_name'),
				$password
			));
		} else {
			run(sprintf(
				'{{bin/mysql}} -e "CREATE USER \'%s\'@\'localhost\' identified by \'%s\';"',
				get('web_name'),
				$password
			));
		}
		run(sprintf('{{bin/mysql}} -e "GRANT ALL PRIVILEGES on %s.* to \'%s\'@\'localhost\';"',
			get('web_name'),
			get('web_name')
		));
		success();

		pwriteln('Importing local MySQL database to server MySQL database');
		$localMysqlUser = pimcoreSystemConfig('read', 'username');
		$localMysqlDb = pimcoreSystemConfig('read', 'dbname');
		$localDbPath = get('local_tmp_directory').DIRECTORY_SEPARATOR.'db.sql';
		runLocally(sprintf(
			'{{bin/local/mysqldump}} %s > %s 2>&1',
			$localMysqlDb,
			$localDbPath
		), 600);
		$db = str_replace(
			'127.0.0.1',
			'localhost',
			str_replace(
				$localMysqlUser,
				get('web_name'),
				file_get_contents($localDbPath)
			)
		);
		file_put_contents($localDbPath, $db);

		$serverDbPath = getServerProjectPath(true, true).'/db.sql';
		pupload($localDbPath, $serverDbPath);
		run(sprintf('{{bin/mysql}} %s < %s', get('web_name'), $serverDbPath));
		success();
	}

	$lifetime = 86400 * 365 * 5;
	\Pimcore\Model\Tool\TmpStore::delete('deployer_last_db_sync_'.env('server.name'));
	\Pimcore\Model\Tool\TmpStore::add('deployer_last_db_sync_'.env('server.name'), time(), 'deployer', $lifetime);
})
->desc('Upload and install DB to server.')
->setPrivate();

task('deploy:vcs', function () {
	$options = parseOptions();

	$gulpfilePath = get('local_web_directory').'/gulpfile.js';
	if (file_exists($gulpfilePath)) {
		$compileGulp = false;
		$gulpfile = file_get_contents($gulpfilePath);
		$tokensToCheck = [
			'task(\'dev\',', 'task(\'prod\',', 'styles.css', 'styles.min.css', './static/css',
			'scripts.js', 'scripts.min.js', './static/js',
		];
		$checkPassed = true;
		foreach ($tokensToCheck as $token) {
			if (stristr($gulpfile, $token) === false) {
				$checkPassed = false;
				break;
			}
		}
		if ($checkPassed) {
			$pairsToCheck = [
				'/static/js/scripts.js' => '/static/js/scripts.min.js',
				'/static/css/styles.css' => '/static/css/styles.min.css',
			];
			foreach ($pairsToCheck as $devFile => $prodFile) {
				$devFilePath = get('local_web_directory').$devFile;
				$prodFilePath = get('local_web_directory').$prodFile;
				if (!file_exists($devFilePath) || !file_exists($prodFilePath)
					|| filemtime($devFilePath) > filemtime($prodFilePath)) {
					$compileGulp = paskConfirmation(
						'Looks like you forgot to run gulp prod, do you want to run it now?'
					);
					break;
				}
			}
		}
		if ($compileGulp) {
			pwrite('Running gulp prod task');
			runLocally(sprintf(
				'cd %s && gulp prod',
				get('local_web_directory')
			), 600);
			success();
		}
	}

	$somethingToCommit = runLocally('git status --porcelain')->toString();
	if ($somethingToCommit) {
		pwrite('Adding local changes to Git');
		runLocally('git add -A', 600);
		success();
		$commitMessage = trim(pask('Please enter commit message', 'increment'));
		if (!$commitMessage) {
			$commitMessage = 'increment';
		}
		runLocally(sprintf('git commit -q -u -m "%s"', addslashes($commitMessage)), 600);
	}

	runLocally('git remote update');
	$somethingToPush = ($options->all || trim(runLocally(
		'git diff --stat --cached origin/master', 600)->toString()
	));
	if ($somethingToPush) {
		pwrite('Pushing to Git from local');
		if ($options->all) {
			runLocally('git push -q -u origin --all', 600);
		} else {
			runLocally('git push -q', 600);
		}
		success();
	}

	if ($options->all) {
		pwrite('Cloning Git repository on server');
		$config = parse_ini_file(get('local_web_directory').'/.git/config');
		$repositoryUrl = $config['url'];
		cd(getServerProjectPath(true));
		run('rm -rf '.getServerProjectPath(true, true));
		run(sprintf('git clone -q %s %s', $repositoryUrl, getServerProjectPath(true)));
		run('git config core.filemode false');
		createServerDirectory(getServerProjectPath(true, true));
		setServerPermissions(getServerProjectPath(true));
		success();
	}

	cd(getServerProjectPath(true));
	setServerPermissions(getServerProjectPath(true).'/.git/', true);
	$somethingToPull = trim(run('git fetch --dry-run')->toString());
	if ($somethingToPull) {
		pwrite('Pulling from Git to server');
		run('git pull');
		setServerPermissions(getServerProjectPath(true).'/.git/', true);
		if ($somethingToCommit) {
			$rows = explode("\n", $somethingToCommit);
			foreach ($rows as $row) {
				$tmp = explode(' ', $row);
				if (!$tmp || count($tmp) < 2 || in_array(trim($tmp[1]), ['M', 'D'])) {
					continue;
				}
				end($tmp);
				setServerPermissions(getServerProjectPath(true).'/'.trim(current($tmp)));
			}
		}
		success();
	}
})
->desc('Push code to server throught the VCS.')
->setPrivate();

task('deploy:ignores', function () {
	$options = parseOptions();
	if (!$options->all) {
		return;
	}

	pwriteln('Copying ignored files');
	$dirsToCreate = ['/website/var/backup', '/website/var/search', '/website/var/cache',
		'/website/var/tmp', '/website/var/assets', '/website/var/email',
		'/website/var/classes/Object', '/website/var/versions',
	];
	foreach ($dirsToCreate as $relativeDirectoryPath) {
		run(sprintf('mkdir -p %s', getServerProjectPath(true).$relativeDirectoryPath));
	}
	copy(
		get('local_web_directory').'/website/var/config/system.php',
		get('local_tmp_directory').'/system.php'
	);
	pimcoreSystemConfig('write', 'username', get('web_name'), true);
	pimcoreSystemConfig('write', 'password', get('server_mysql_password'), true);
	pimcoreSystemConfig('write', 'dbname', get('web_name'), true);
	pupload(
		realpath(get('local_tmp_directory').'/system.php'),
		getServerProjectPath(true).'/website/var/config/system.php'
	);
	if (env()->has('cache_config')) {
		$cacheConfig = "<?php \n\nreturn ".var_export_pretty_tabs(env('cache_config')).";\n";
	} else {
		$cache = ['frontend' => $cacheFrontend];
		$cacheConfig = "<?php \n\nreturn ".var_export_pretty_tabs($cache).";\n";
	}
	createLocalFile(get('local_tmp_directory').'/cache.php', $cacheConfig);
	pupload(
		realpath(get('local_tmp_directory').'/cache.php'),
		getServerProjectPath(true).'/website/var/config/cache.php'
	);

	$toUpload = [
		'/.htaccess',
		'/website/var/classes/Object',
		'/website/var/assets',
		'/website/var/email',
		'/website/var/versions',
	];
	foreach ($toUpload as $relativePath) {
		pupload(
			realpath(get('local_web_directory').$relativePath),
			getServerProjectPath(true).$relativePath
		);
	}
	setServerPermissions(getServerProjectPath());
})
->desc('Upload files and directories ignored by the VCS.')
->setPrivate();

task('deploy:vendors', function () {
	$options = parseOptions();

	if ($options->all) {
		pwrite('Installing composer libraries on server (this can take a while)');
		cd(getServerProjectPath(true));
		run('composer install --no-dev --optimize-autoloader --quiet');
		run(sprintf('git checkout %s', getServerProjectPath(true).'/composer.lock'));
		setServerPermissions(getServerProjectPath(true).'/vendor');
		success();
	} else {
		$composerLockStamp = filemtime(get('local_web_directory').'/composer.lock');
		if (env('last_sync_stamp') && $composerLockStamp
				&& $composerLockStamp > env('last_sync_stamp')) {
			pwrite('Updating composer libraries on server (this can take a while)');
			setServerPermissions(getServerProjectPath(true).'/vendor');
			setServerPermissions(getServerProjectPath(true).'/pimcore');
			cd(getServerProjectPath(true));
			run('composer update --no-dev --optimize-autoloader --quiet');
			run(sprintf('git checkout %s', getServerProjectPath(true).'/composer.lock'));
			setServerPermissions(getServerProjectPath(true).'/vendor');
			setServerPermissions(getServerProjectPath(true).'/pimcore');
			success();
		}
	}
})
->desc('Install 3rd party libraries.')
->setPrivate();

task('deploy:pimcore_classes', function () {
	$options = parseOptions();
	if ($options->all) {
		return;
	}

	if (env('last_sync_stamp')) {
		$changedClasses = array_filter(array_unique(explode("\n", runLocally(sprintf(
			'git log --name-only --oneline --since %d --pretty=format:',
			env('last_sync_stamp')
		), 600)->toString())), function ($path) {
			return stristr($path, '.psf') && (
				stristr($path, 'website/var/classes/definition')
				|| stristr($path, 'website/var/classes/fieldcollections')
				|| stristr($path, 'website/var/classes/objectbricks'))
			;
		});
		if ($changedClasses) {
			setServerPermissions(getServerProjectPath(true).'/website/var/classes');

			foreach ($changedClasses as $definitionPath) {
				pimcoreUpdateDefinition($definitionPath);
			}
		}
	}
})
->desc('Updates required pimcore classes on server.')
->setPrivate();

task('sync:db', function () {
	$options = parseOptions();
	$systemConfigPath = get('local_web_directory').'/website/var/config/system.php';
	if (!$options->all && !$options->db && file_exists($systemConfigPath)) {
		return;
	}

	pwrite('Creating local MySQL database and user');
	download(
		get('local_tmp_directory').DIRECTORY_SEPARATOR.'system.php',
		getServerProjectPath(true).'/website/var/config/system.php'
	);
	if (!file_exists($systemConfigPath)) {
		copy(
			get('local_tmp_directory').DIRECTORY_SEPARATOR.'system.php',
			get('local_web_directory').'/website/var/config/system.php'
		);
	}
	$localDbName = pimcoreSystemConfig('read', 'dbname');
	$localDbUser = pimcoreSystemConfig('read', 'username');
	$localDbPassword = pimcoreSystemConfig('read', 'password');
	$remoteDbName = pimcoreSystemConfig('read', 'dbname', null, true);
	$remoteDbUser = pimcoreSystemConfig('read', 'username', null, true);
	$isMysql57 = version_compare(getMysqlVersion(true), '5.7.0', '>=');
	if ($localDbUser != 'root') {
		if (!$isMysql57) {
			runLocally(sprintf(
				'{{bin/local/mysql}} -e "GRANT USAGE ON *.* TO \'%s\'@\'localhost\';'
					.'DROP USER \'%s\'@\'localhost\';'
					.'CREATE USER \'%s\'@\'localhost\' IDENTIFIED BY \'%s\';'
					.'GRANT ALL PRIVILEGES ON *.* TO \'%s\'@\'localhost\';" 2>&1',
				$localDbUser,
				$localDbUser,
				$localDbUser,
				$localDbPassword,
				$localDbUser
			), 600);
		} else {
			runLocally(sprintf(
				'{{bin/local/mysql}} -e "CREATE USER IF NOT EXISTS \'%s\'@\'localhost\';'
					.'SET PASSWORD FOR \'%s\'@\'localhost\' = PASSWORD(\'%s\');'
					.'GRANT ALL PRIVILEGES ON *.* TO \'%s\'@\'localhost\';" 2>&1',
				$localDbUser,
				$localDbUser,
				$localDbPassword,
				$localDbUser
			), 600);
		}
	}
	list($dbExists, $userExists) = mysqlDbAndUserCheck(true, false, false, $localDbName, $localDbUser);
	if ($dbExists) {
		runLocally('{{bin/local/mysqladmin}} -f drop '.$localDbName, 600);
	}
	runLocally(sprintf(
		'{{bin/local/mysql}} -e "CREATE DATABASE %s CHARACTER SET utf8 COLLATE utf8_general_ci" 2>&1',
		$localDbName
	), 600);
	success();

	pwrite('Importing server database to local');
	run(sprintf(
		'{{bin/mysqldump}} --routines %s > %s 2>&1',
		$remoteDbName,
		getServerProjectPath(true, true).'/'.$remoteDbName.'.sql'
	));
	$tmpDbPath = get('local_tmp_directory').DIRECTORY_SEPARATOR.$remoteDbName.'.sql';
	download(
		$tmpDbPath,
		getServerProjectPath(true, true).'/'.$remoteDbName.'.sql'
	);
	$db = str_replace(
		'127.0.0.1',
		'localhost',
		str_replace(
			'`'.$remoteDbUser.'`',
			'`'.$localDbUser.'`',
			file_get_contents($tmpDbPath)
		)
	);
	file_put_contents($tmpDbPath, $db);
	runLocally(sprintf('{{bin/local/mysql}} %s < %s 2>&1', $localDbName, $tmpDbPath), 600);

	$lifetime = 86400 * 365 * 5;
	\Pimcore\Model\Tool\TmpStore::delete('deployer_last_db_sync_'.env('server.name'));
	\Pimcore\Model\Tool\TmpStore::add('deployer_last_db_sync_'.env('server.name'), time(), 'deployer', $lifetime);

	success();
})
->desc('Download and install DB from server.')
->setPrivate();

task('sync:vcs', function () {
	$options = parseOptions();

	cd(getServerProjectPath(true));
	$somethingToCommit = run('git status --porcelain')->toString();
	if ($somethingToCommit) {
		pwriteln('There are uncommitted changes on server: ');
		writeln($somethingToCommit);
		if (paskConfirmation('Do you want to commit them?')) {
			$commitMessage = trim(pask('Please enter commit message', 'increment'));
			if (!$commitMessage) {
				$commitMessage = 'increment';
			}
			pwrite('Adding server changes to Git');
			setServerPermissions(getServerProjectPath(true).'/.git/', true);
			run('git add -A');
			run(sprintf('git commit -q -u -m "%s"', addslashes($commitMessage)));
			success();
		}
	}

	run('git remote update');
	$somethingToPush = trim(run('git diff --stat --cached origin/master')->toString());
	if ($somethingToPush) {
		pwrite('Pushing to Git from server');
		run('git push -q');
		success();
	}

	runLocally('git remote update', 600);
	$somethingToPull = trim(runLocally('git log HEAD..origin/master --oneline')->toString());
	if ($somethingToPull) {
		pwrite('Pulling from Git to local');
		runLocally('git pull', 600);
		success();
	}
})
->desc('Pull code from server throught the VCS.')
->setPrivate();

task('sync:ignores', function () {
	$options = parseOptions();
	$serverProjectPath = getServerProjectPath(true);

	pwrite('Copying ignored files'.(($options->all) ? ' (this can take a while)' : ''));
	$dirsToCreate = ['/website/var/backup', '/website/var/search', '/website/var/cache',
		'/website/var/tmp', '/website/var/assets', '/website/var/email',
		'/website/var/classes/Object', '/website/var/versions',
		'website/var/versions/asset', 'website/var/versions/document',
		'website/var/versions/object',
	];
	foreach ($dirsToCreate as $relativeDirectoryPath) {
		if (!file_exists(get('local_web_directory').'/'.$relativeDirectoryPath)) {
			mkdir(get('local_web_directory').'/'.$relativeDirectoryPath, 0777, true);
		}
	}
	if (!file_exists(get('local_web_directory').'/.htaccess')) {
		download(
			get('local_web_directory').'/.htaccess',
			$serverProjectPath.'/.htaccess'
		);
	}
	if (!file_exists(get('local_web_directory').'/website/var/config/cache.php')) {
		download(
			get('local_web_directory').'/website/var/config/cache.php',
			$serverProjectPath.'/website/var/config/cache.php'
		);
	}
	$directoriesToDownload = [
		'website/var/classes/Object',
	];
	if ($options->all) {
		$directoriesToDownload[] = 'website/var/assets';
		$directoriesToDownload[] = 'website/var/email';
		$directoriesToDownload[] = 'website/var/versions';
	}
	cd($serverProjectPath);
	foreach ($directoriesToDownload as $relativePath) {
		$files = explode("\n", run(sprintf(
			'find %s -type f',
			$relativePath
		))->toString());
		foreach ($files as $file) {
			if (!$file) {
				continue;
			}
			$parentDirectory = dirname(get('local_web_directory').'/'.$file);
			if (!file_exists($parentDirectory)) {
				mkdir($parentDirectory, 0777, true);
			}
			download(
				get('local_web_directory').'/'.$file,
				$serverProjectPath.'/'.$file
			);
		}
	}
	success();
})
->desc('Download files ignored by the VCS.')
->setPrivate();

task('sync:vendors', function () {
	$options = parseOptions();

	if (!file_exists(get('local_web_directory').'/vendor')) {
		pwrite('Installing composer libraries on local (this can take a while)');
		runLocally(sprintf(
			'cd %s && composer install --optimize-autoloader --quiet',
			get('local_web_directory')
		), 600);
		runLocally(sprintf(
			'cd %s && git checkout %s',
			get('local_web_directory'),
			get('local_web_directory').'/composer.lock'
		), 600);
		success();
	} else {
		$composerLockStamp = filemtime(get('local_web_directory').'/composer.lock');
		if (!env('last_sync_stamp') || ($composerLockStamp
				&& $composerLockStamp > env('last_sync_stamp'))) {
			pwrite('Updating composer libraries on local (this can take a while)');
			runLocally(sprintf(
				'cd %s && composer update --optimize-autoloader --quiet',
				get('local_web_directory')
			), 600);
			runLocally(sprintf(
				'cd %s && git checkout %s',
				get('local_web_directory'),
				get('local_web_directory').'/composer.lock'
			), 600);
			success();
		}
	}
})
->desc('Install 3rd party libraries.')
->setPrivate();

task('cleanup', function () {
	$options = parseOptions();

	$lifetime = 86400 * 365 * 5;
	\Pimcore\Model\Tool\TmpStore::delete('deployer_last_sync_'.env('server.name'));
	\Pimcore\Model\Tool\TmpStore::add('deployer_last_sync_'.env('server.name'), time(), 'deployer', $lifetime);

	if ($options->all && input()->getArgument('command') == 'deployment:deploy') {
		$crontab = run('{{sudo}}crontab -u {{http_user}} -l')->toString();
		$phpBin = run('which php');
		$crontask = sprintf(
			'*/15 * * * *	%s %s',
			$phpBin,
			getServerProjectPath(true).'/pimcore/cli/maintenance.php'
		);
		if (stristr($crontab, getServerProjectPath(true)) === false) {
			pwriteln(sprintf('Adding new task to %s crontab', env('http_user')));
			$crontab .= "\n".$crontask."\n";
			createServerFile(getServerProjectPath(true, true).'/crontab', $crontab);
			run(sprintf(
				'cat %s | {{sudo}}crontab -u {{http_user}} -',
				getServerProjectPath(true, true).'/crontab'
			));
			success();
		}
	}

	pwrite('Removing temporary files and directories');
	runLocally('rm -rf '.get('local_tmp_directory'), 600);
	run('rm -rf '.getServerProjectPath(true, true));
	success();

	if ($options->all && input()->getArgument('command') == 'deployment:deploy') {
		$warmUpCache = paskConfirmation('Do you want to warm up the cache?');
		if ($warmUpCache) {
			cd(getServerProjectPath(true));
			try {
				run('{{sudo}}-u {{http_user}} php pimcore/cli/console.php cache:warming');
			} catch (\RuntimeException $e) {
			}
			success();
		}

		if (env()->has('apache_reload_command')) {
			pwrite('Reloading apache web server');
			run('{{sudo}}{{apache_reload_command}}');
			success();
		}
	}
})
->desc('Clean up temporary files after deploy or sync.')
->setPrivate();

task('deployment:remove', function () {
	$serverProjectPath = getServerProjectPath();
	$confirmation = paskConfirmation(sprintf(
		'Are you sure to remove %s, related DB and vhost from %s?',
		$serverProjectPath,
		env('server.host')
	));

	if ($confirmation) {
		binariesCheck();

		createLocalDirectory(get('local_tmp_directory'));

		$crontab = run('{{sudo}}crontab -u {{http_user}} -l')->toString();
		$crontabLines = explode("\n", $crontab);
		foreach ($crontabLines as $i => $line) {
			if (stristr($line, getServerProjectPath(true)) !== false) {
				pwriteln(sprintf('Removing task from %s crontab', env('http_user')));
				unset($crontabLines[$i]);
				$crontab = implode("\n", $crontabLines)."\n";
				createServerFile(getServerProjectPath(true, true).'/crontab', $crontab);
				run(sprintf(
					'cat %s | {{sudo}}crontab -u {{http_user}} -',
					getServerProjectPath(true, true).'/crontab'
				));
				success();
				break;
			}
		}

		if (serverDirExists($serverProjectPath)) {
			pwrite('Removing files and directories');
			run('{{sudo}}rm -rf '.$serverProjectPath);
			success();
		}

		if (env()->has('vhosts_path') && serverFileExists(getServerVhostPath())) {
			pwrite('Removing vhost');
			if (env()->has('vhosts_symlink_path') && serverFileExists(getServerVhostPath(true))) {
				run('{{sudo}}rm -rf '.getServerVhostPath(true));
			}
			run('{{sudo}}rm -rf '.getServerVhostPath());
			success();
		}

		mysqlAccessCheck();
		list($dbExists, $dbUserExists) = mysqlDbAndUserCheck(false, false);
		if ($dbExists || $dbUserExists) {
			pwrite('Removing DB');
			if ($dbUserExists) {
				run(sprintf(
					'{{bin/mysql}} -e "DROP user \'%s\'@\'localhost\';"',
					get('web_name')
				));
			}
			if ($dbExists) {
				run(sprintf(
					'{{bin/mysqladmin}} -f drop %s',
					get('web_name')
				));
			}
			success();
		}
		if (env()->has('apache_reload_command')) {
			pwrite('Reloading apache web server');
			run('{{sudo}}{{apache_reload_command}}');
			success();
		}

		runLocally('rm -rf '.get('local_tmp_directory'), 600);
	} else {
		return;
	}
})->desc('Remove all parts of project from server.');

task('deployment:deploy', [
	'prepare',
	'deploy:vhost',
	'deploy:db',
	'deploy:vcs',
	'deploy:ignores',
	'deploy:vendors',
	'deploy:pimcore_classes',
	'cleanup',
])->desc('Deploy project to the server. Only VCS changes are deployed by default.');

task('deployment:sync', [
	'prepare',
	'sync:db',
	'sync:vcs',
	'sync:ignores',
	'sync:vendors',
	'cleanup',
])->desc('Download and setup project from the server. Only VCS changes are synchronized by default.');

/************************************\
-------------- HELPERS --------------*

\************************************/

function pwrite($message = '', $prefix = '  ')
{
	write($prefix.$message);
}

function pwriteln($message = '', $prefix = '  ')
{
	writeln($prefix.$message);
}

function pupload($local, $server)
{
	pwrite();
	pwrite();
	upload($local, $server);
}

function paskConfirmation($message, $default = false)
{
	pwrite();

	return askConfirmation($message, $default);
}

function pask($message, $default = null)
{
	pwrite();

	return ask($message, $default);
}

function paskHiddenResponse($message)
{
	pwrite();

	return askHiddenResponse($message);
}

function parseOptions()
{
	$options = [
		'all' => input()->hasOption('all') ? input()->getOption('all') : false,
		'db' => input()->hasOption('db') ? input()->getOption('db') : false,
	];

	return (object) $options;
}

function generateRandomString($length = 10, $lower = false)
{
	$randomString = substr(
		str_shuffle('0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ'),
		0,
		$length
	);

	if ($lower) {
		$randomString = strtolower($randomString);
	}

	return $randomString;
}

function getServerProjectPath($addWebRoot = false, $addDeploymentTmp = false)
{
	$path = env('hosting_path').'/'.get('web_name');
	if ($addWebRoot || $addDeploymentTmp) {
		$path  .= '/html';
	}
	if ($addDeploymentTmp) {
		$path .= '/deployment_tmp';
	}

	return $path;
}

function getServerVhostPath($symlink = false)
{
	if ($symlink && env()->has('vhosts_symlink_path')) {
		return env('vhosts_symlink_path').'/'.get('web_name').'.vhost';
	} else {
		return env('vhosts_path').'/'.get('web_name').'.vhost';
	}
}

function serverFileExists($path, $throwException = false)
{
	$exists = run(sprintf('if [ -f %s ]; then echo "true"; fi', $path))->toBool();
	if ($throwException && $exists) {
		throw new \RuntimeException(sprintf(
			'Looks like file <comment>%s</comment> already exists on the server.',
			$path
		));
	}

	return $exists;
}

function serverFileMissing($path, $throwException = false)
{
	$missing = run(sprintf('if [ ! -f %s ]; then echo "true"; fi', $path))->toBool();
	if ($throwException && $missing) {
		throw new \RuntimeException(sprintf(
			'Looks like file <comment>%s</comment> does not exist on the server.',
			$path
		));
	}

	return $missing;
}

function serverDirExists($path, $throwException = false)
{
	$exists = run(sprintf('if [ -d %s ]; then echo "true"; fi', $path))->toBool();
	if ($throwException && $exists) {
		throw new \RuntimeException(sprintf(
			'Looks like directory <comment>%s</comment> already exists on the server.',
			$path
		));
	}

	return $exists;
}

function serverDirMissing($path, $throwException = false)
{
	$missing = run(sprintf('if [ ! -d %s ]; then echo "true"; fi', $path))->toBool();
	if ($throwException && $missing) {
		throw new \RuntimeException(sprintf(
			'Looks like directory <comment>%s</comment> does not exist on the server.',
			$path
		));
	}

	return $missing;
}

function createLocalDirectory($path)
{
	if (realpath($path) === false) {
		if (mkdir($path, '0664', true) === false) {
			throw new \RuntimeException(sprintf(
				'Looks like local directory <comment>%s</comment> could not be created.',
				$path
			));
		}
	}
}

function createLocalFile($path, $content)
{
	if (file_put_contents($path, $content) === false) {
		throw new \RuntimeException(sprintf(
			'Looks like local file <comment>%s</comment> could not be created.',
			$path
		));
	}
}

function createServerDirectory($path)
{
	run('mkdir -p '.$path);
	run(sprintf('{{sudo}}chmod 775 '.$path));
}

function createServerFile($path, $content)
{
	$fileName = basename($path);
	$localPath = get('local_tmp_directory').DIRECTORY_SEPARATOR
		.generateRandomString(4, true).'-'.$fileName;
	createLocalFile($localPath, $content);
	pupload($localPath, $path);
	run(sprintf('{{sudo}}chmod 664 '.$path));
}

function success($msg = null, $newLine = true)
{
	if ($newLine) {
		pwriteln($msg.'<info>✔</info>');
	} else {
		pwrite($msg.'<info>✔</info>');
	}
}

function fail($msg = null, $newLine = true)
{
	if ($newLine) {
		pwriteln($msg.'<fg=red;options=bold>x</>');
	} else {
		pwrite($msg.'<fg=red;options=bold>x</>');
	}
}

function localCommandExist($command)
{
	try {
		$output = runLocally('which '.$command);

		return !(stristr($output, $command) === false);
	} catch (\RuntimeException $e) {
		$formatter = \Deployer\Deployer::get()->getHelper('formatter');
		pwrite($formatter->formatBlock(sprintf(
			'Running local command %s: %s | %s',
			$command,
			$e->getMessage(),
			$e->getTraceAsString()
		), 'error', true));

		return false;
	}
}

function binariesCheck($local = false)
{
	$existsCommand = ($local) ? 'localCommandExist' : 'commandExist';
	$where = ($local) ? 'local' : 'server';

	if (!$local) {
		// Check if shell is POSIX-compliant (copied from deployer's common task)
		\Deployer\Task\Context::get()->getServer()->connect();
		try {
			cd(''); // To run command as raw.
			$result = run('echo $0')->toString();
			if ($result == 'stdin: is not a tty') {
				throw new RuntimeException(
					'Looks like ssh inside another ssh.\n'.
					'Help: http://goo.gl/gsdLt9'
				);
			}
		} catch (\RuntimeException $e) {
			$formatter = \Deployer\Deployer::get()->getHelper('formatter');
			$errorMessage = [
				'Shell on your server is not POSIX-compliant. Please change to sh, bash or similar.',
				'Usually, you can change your shell to bash by running: chsh -s /bin/bash',
			];
			pwrite($formatter->formatBlock($errorMessage, 'error', true));

			throw $e;
		}
	}

	$binariesToCheck = ['which', 'cat', 'grep', 'awk', 'git', 'mysql',
		'mysqldump', 'mysqladmin', 'composer', 'printf', ];
	if ($local) {
		$binariesToCheck['gulp'];
	} else {
		$binariesToCheck[] = 'find';
		$binariesToCheck[] = 'php';
		if (env('sudo')) {
			$binariesToCheck[] = 'sudo';
			$binariesToCheck[] = 'crontab';
		}
	}
	foreach ($binariesToCheck as $command) {
		if (!$existsCommand($command)) {
			throw new \RuntimeException(sprintf(
				'Looks like %s bash command %s is missing.',
				$where,
				$command
			));
		}
	}

	if (!$local && env('sudo')) {
		$sudoError = trim(run('{{sudo}}-n echo 2>&1'));
		if ($sudoError != '') {
			throw new \RuntimeException(sprintf(
				'Looks like %s@%s is not a sudoer with NOPASSWD set.',
				\Deployer\Tosk\Context::get()->getServer()->getConfiguration()->getUser(),
				env('server.host')
			));
		}
	}
}

function gitCheck($local = false, $throwException = true)
{
	$runCommand = ($local) ? 'runLocally' : 'run';
	$where = ($local) ? 'local' : 'server';

	if (!$local) {
		cd(getServerProjectPath(true));
	}
	try {
		$gitCheck = $runCommand('git ls-remote 2>&1');
	} catch (\RuntimeException $e) {
		$gitCheck = $e->getMessage();
	}
	$noAccess = (stristr($gitCheck, 'fatal') !== false || stristr($gitCheck, 'error') !== false);
	if (!$noAccess && $local) {
		$config = parse_ini_file(get('local_web_directory').'/.git/config');
		if ($config === false || !isset($config['url'])) {
			$noAccess = true;
		}
	}
	if ($throwException && $noAccess) {
		throw new \RuntimeException(sprintf(
			'Looks like %s git repository has no access to the remote.',
			$where
		));
	}

	return !$noAccess;
}

function getMysqlVersion($local = false)
{
	return trim(($local)
		? runLocally('mysql --version|awk \'{ printf "%s",$5 }\'|awk -F\, \'{ printf "%s",$1 }\'')
		: run('mysql --version|awk \'{ printf "%s",$5 }\'|awk -F\, \'{ printf "%s",$1 }\''));
}

function mysqlAccessCheck($local = false, $throwException = true)
{
	$isMysql57 = version_compare(getMysqlVersion($local), '5.7.0', '>=');
	$runCommand = ($local) ? 'runLocally' : 'run';
	$where = ($local) ? 'local' : 'server';

	if (!$local) {
		if (env()->has('mysql_login_path')) {
			env('bin/mysql', 'mysql --login-path='.env('mysql_login_path'));
			env('bin/mysqldump', 'mysqldump --login-path='.env('mysql_login_path'));
			env('bin/mysqladmin', 'mysqladmin --login-path='.env('mysql_login_path'));
		} else {
			if (env()->has('mysql_user')) {
				$serverMysqlUser = env('mysql_user');
			} else {
				pwriteln();
				$serverMysqlUser = trim(pask('Please enter server MySQL root username'));
			}
			if (env()->has('mysql_password')) {
				$serverMysqlPassword = env('mysql_password');
			} else {
				pwriteln();
				$serverMysqlPassword = trim(paskHiddenResponse('Please enter server MySQL root password'));
			}
			env('bin/mysql', sprintf('export MYSQL_PWD=%s; mysql -u %s', $serverMysqlPassword, $serverMysqlUser));
			env('bin/mysqldump', sprintf('export MYSQL_PWD=%s; mysqldump -u %s', $serverMysqlPassword, $serverMysqlUser));
			env('bin/mysqladmin', sprintf('export MYSQL_PWD=%s; mysqladmin -u %s', $serverMysqlPassword, $serverMysqlUser));
		}
	}

	try {
		if ($isMysql57) {
			$mysqlError = $runCommand(sprintf(
				'{{bin%s/mysql}} -e "CREATE USER IF NOT EXISTS '
					.'\'dummyuser\'@\'localhost\'; DROP USER \'dummyuser\'@\'localhost\';" 2>&1',
				($local) ? '/local' : ''
			))->toString();
		} else {
			$mysqlError = $runCommand(sprintf(
				'{{bin%s/mysql}} -e "GRANT USAGE ON *.* TO '
					.'\'dummyuser\'@\'localhost\'; DROP USER \'dummyuser\'@\'localhost\';" 2>&1',
				($local) ? '/local' : ''
			))->toString();
		}
	} catch (\RuntimeException $e) {
		$mysqlError = ($e->getMessage()) ? $e->getMessage() : 'Unknown';
	}

	if ($throwException) {
		if ($mysqlError) {
			throw new \RuntimeException(
				sprintf('Looks like you do not have root access to %s MySQL database. Reason: %s',
				$where,
				$mysqlError
			));
		}
	} else {
		return !$mysqlError;
	}
}

function mysqlDbAndUserCheck($local = false, $throwExceptionOnExistence = true,
	$throwExceptionOnNonExistence = false, $dbName = null, $username = null)
{
	$runCommand = ($local) ? 'runLocally' : 'run';
	$where = ($local) ? 'local' : 'server';
	if (!$username) {
		$username = get('web_name');
	}
	if (!$dbName) {
		$dbName = get('web_name');
	}

	try {
		$dbMissing = (bool) $runCommand(sprintf(
			'{{bin%s/mysql}} -e "USE %s;" 2>&1',
			($local) ? '/local' : '',
			$dbName
		))->toString();
	} catch (\RuntimeException $e) {
		$dbMissing = true;
	}
	$userExists = (bool) $runCommand(sprintf(
		'{{bin%s/mysql}} -e "SELECT 1 FROM mysql.user WHERE user = \'%s\';" 2>&1',
		($local) ? '/local' : '',
		$username
	))->toString();

	if ($throwExceptionOnExistence) {
		if (!$dbMissing) {
			throw new \RuntimeException(sprintf(
				'Looks like database <comment>%s</comment> already exists on %s MySQL',
				$dbName,
				$where
			));
		}
		if ($userExists) {
			throw new \RuntimeException(sprintf(
				'Looks like database user <comment>%s</comment> already exists on %s MySQL',
				$username,
				$where
			));
		}
	} elseif ($throwExceptionOnNonExistence) {
		if ($dbMissing) {
			throw new \RuntimeException(sprintf(
				'Looks like database <comment>%s</comment> does not exist on %s MySQL',
				$dbName,
				$where
			));
		}
		if (!$userExists) {
			throw new \RuntimeException(sprintf(
				'Looks like database user <comment>%s</comment> does not exist on %s MySQL',
				$username,
				$where
			));
		}
	} else {
		return [!$dbMissing, $userExists];
	}
}

function setServerPermissions($path, $setExecutables = false)
{
	run(sprintf('{{sudo}}find %s -type d -exec chmod 775 {} \;', $path));
	run(sprintf('{{sudo}}find %s -type f -exec chmod 664 {} \;', $path));
	if ($setExecutables) {
		run(sprintf('{{sudo}}find %s -type d -exec chmod 775 {} \;', getServerProjectPath(true).'/.git/'));
		run(sprintf('{{sudo}}find %s -type f -exec chmod 774 {} \;', getServerProjectPath(true).'/.git/'));
		run(sprintf(
			'{{sudo}}chmod 664 %s %s',
			getServerProjectPath(true).'/composer.json',
			getServerProjectPath(true).'/composer.lock'
		));
	}
	run(sprintf(
		'{{sudo}}chown -R %s:%s %s',
		env('http_user'),
		env('user_group'),
		getServerProjectPath()
	));
}

function pimcoreSystemConfig($task, $param, $value = null, $useTmpConfig = false)
{
	if ($useTmpConfig) {
		$configPath = get('local_tmp_directory').'/system.php';
	} else {
		$configPath = get('local_web_directory').'/website/var/config/system.php';
	}
	$config = include $configPath;

	if ($task == 'read') {
		$value = null;
		switch ($param) {
			case 'username':
				$value = $config['database']['params']['username'];
				break;
			case 'password':
				$value = $config['database']['params']['password'];
				break;
			case 'dbname':
				$value = $config['database']['params']['dbname'];
				break;
		}

		return $value;
	} elseif ($task == 'write') {
		switch ($param) {
			case 'username':
				$config['database']['params']['username'] = $value;
				break;
			case 'password':
				$config['database']['params']['password'] = $value;
				break;
			case 'dbname':
				$config['database']['params']['dbname'] = $value;
				break;
		}
		$newConfig = "<?php \n\nreturn ".var_export_pretty_tabs($config).";\n";
		file_put_contents($configPath, $newConfig);
	}
}

// copied from pimcore helpers
function var_export_pretty_tabs($var, $indent = '')
{
	switch (gettype($var)) {
		case 'string':
			return '"'.addcslashes($var, "\\\$\"\r\n\t\v\f").'"';
		case 'array':
			$indexed = array_keys($var) === range(0, count($var) - 1);
			$r = [];
			foreach ($var as $key => $value) {
				$r[] = "$indent\t"
					.($indexed ? '' : var_export_pretty_tabs($key).' => ')
					.var_export_pretty_tabs($value, "$indent\t");
			}

			return "[\n".implode(",\n", $r)."\n".$indent.']';
		case 'boolean':
			return $var ? 'TRUE' : 'FALSE';
		default:
			return var_export($var, true);
	}
}

function pimcoreUpdateDefinition($definitionPath)
{
	cd(getServerProjectPath(true));
	$result = null;
	$definitionFileExists = file_exists(get('local_web_directory').'/'.$definitionPath);
	if (stristr($definitionPath, 'classes/fieldcollections') !== false) {
		$key = current(explode('.', basename($definitionPath)));
		if ($definitionFileExists) {
			pwrite(sprintf('Updating fieldcollection %s', $key));
			$result = run(sprintf(
				'php pimcore/cli/console.php deployment:fieldcollection-update %s',
				$key
			));
		} else {
			pwriteln(sprintf('Please remove fieldcollection %s DB tabes from server', $key));
		}
	} elseif (stristr($definitionPath, 'classes/objectbricks') !== false) {
		$key = current(explode('.', basename($definitionPath)));
		if ($definitionFileExists) {
			pwrite(sprintf('Updating objectbrick %s', $key));
			$result = run(sprintf(
				'php pimcore/cli/console.php deployment:objectbrick-update %s',
				$key
			));
		} else {
			pwriteln(sprintf('Please remove objectbrick %s DB tabes from server', $key));
		}
	} elseif (stristr($definitionPath, 'classes/definition') !== false) {
		$key = current(explode('.', basename($definitionPath)));
		$id = (int) explode('_', $key)[1];
		if ($definitionFileExists) {
			$class = \Pimcore\Model\Object\ClassDefinition::getById($id);
			pwrite(sprintf('Updating class %s [%d]', $class->getName(), $id));
			$result = run(sprintf(
				'php pimcore/cli/console.php deployment:class-update %d %s %s',
				$id,
				$class->getName(),
				$class->getIcon()
			));
		} else {
			pwrite(sprintf('Removing class %d', $id));
			$result = run(sprintf(
				'php pimcore/cli/console.php deployment:class-update %d %s',
				$id,
				'Dummy'
			));
		}
	}

	if ($result) {
		if (stristr($result, 'created') || stristr($result, 'updated') || stristr($result, 'deleted')) {
			success();
		} else {
			fail();
			pwriteln($result);
		}
	}
}

function createMigrationFile($sinceStamp)
{
	$db = \Pimcore\Db::get();
	$result = [
		'migrationFilePath' => realpath(get('local_web_directory').'/website/var/tmp')
			.DIRECTORY_SEPARATOR.'migration-'.date('Y-m-d-H-i-s').'.php',
		'db' => [],
		'migrationFile' => '',
	];

	$simpleTables = [
		'custom_layouts' => [],
		'glossary' => ['check_autoincrement' => true],
		'redirects' => ['check_autoincrement' => true],
		'translations_admin' => [],
		'translations_website' => [],
		'website_settings' => ['check_autoincrement' => true],
	];
	foreach ($simpleTables as $table => $options) {
		$newEntries = $db->select()->from($table)
			->where('creationDate > ?', $sinceStamp)
			->query()->fetchAll();
		if ($newEntries) {
			$result['migrationFile'] .= sprintf("// %s - new entries\n", $table);
			foreach ($newEntries as $entry) {
				$result['migrationFile'] .= sprintf('$db->insertOrUpdate(\'%s\', ', $table)
					.var_export_pretty_tabs($entry).");\n";
			}
			$result['db'][$table]['new'] = count($newEntries);
		}
		if (!isset($options['check_autoincrement'])
				|| autoIncrementMatch($table, count($newEntries))) {
			$updatedEntries = $db->select()->from($table)
				->where('creationDate < ?', $sinceStamp)
				->where('modificationDate > ?', $sinceStamp)
				->query()->fetchAll();
			if ($updatedEntries) {
				$result['migrationFile'] .= sprintf("// %s - updated entries\n", $table);
				foreach ($updatedEntries as $entry) {
					$result['migrationFile'] .= sprintf('$db->insertOrUpdate(\'%s\', ', $table)
						.var_export_pretty_tabs($entry).");\n";
				}
				$result['db'][$table]['updated'] = count($updatedEntries);
			}
		}
	}

	if ($result['migrationFile']) {
		$result['migrationFile'] = "<?php\n\n"
			."set_time_limit(60000);\n"
			."include_once __DIR__.'/../../../pimcore/cli/startup.php';\n"
			."\$db = Pimcore\Db::get();\n\n"
			.$result['migrationFile'];
		file_put_contents($result['migrationFilePath'], $result['migrationFile']);

		return $result;
	}

	return;
}

function autoIncrementMatch($table, $increment = 0)
{
	$lastServerAutoincrement = trim(run(sprintf(
		'{{bin/mysql}} -sN -e "SELECT AUTO_INCREMENT FROM INFORMATION_SCHEMA.TABLES'
		.' WHERE TABLE_SCHEMA = \'%s\' AND TABLE_NAME = \'%s\';"',
		get('web_name'),
		$table
	))->toString());
	$lastLocalAutoincrement = trim(runLocally(sprintf(
		'{{bin/local/mysql}} -sN -e "SELECT AUTO_INCREMENT FROM INFORMATION_SCHEMA.TABLES'
		.' WHERE TABLE_SCHEMA = \'%s\' AND TABLE_NAME = \'%s\';"',
		get('web_name'),
		$table
	))->toString());

	return (int) $lastServerAutoincrement === ((int) $lastLocalAutoincrement - $increment);
}
