<?php

/**
 * simple class to help with elastic searching.
 */
namespace Website\Model;

class ElasticSearch
{
	private static $instances;

	/**
	 * @var \Elasticsearch\Client
	 */
	protected $elasticClient = null;
	/**
	 * @var \Website\Model\Currency
	 */
	protected $currencyModel = null;
	protected $index = null;
	protected $language = 'cs';
	protected $urlPrefix = null;

	/**
	 * @param string $language
	 * @param string $urlPrefix
	 *
	 * @return ElasticSearch
	 */
	public static function getInstance(\Website\Model\Currency $currencyModel, $language, $urlPrefix = null)
	{
		$key = $language.$urlPrefix;
		if (empty(self::$instances[$key])) {
			self::$instances[$key] = new self($currencyModel, $language, $urlPrefix);
		}

		return self::$instances[$key];
	}

	public function __construct(\Website\Model\Currency $currencyModel, $language, $urlPrefix = null)
	{
		$this->currencyModel = $currencyModel;
		$this->language = $language;
		$this->elasticClient = \Website\Tool\ElasticSearch::getClient();
		$this->index = \Website\Tool\ElasticSearch::getMainIndexName(false);
		$this->urlPrefix = $urlPrefix;

		if (!$this->elasticClient->indices()->existsAlias(['name' => $this->index])) {
			throw new \Exception('Elastic search main index not found.');
		}
	}

	public function getProducts($path, $filter = [])
	{
		/*	PREPARE QUERY BASE	*/

		// query base
		$baseOptions = ['type' => 'product_'.$this->language];
		if (!empty($filter['query'])) {
			$baseOptions['query']['multi_match']['query'] = $filter['query'];
			$baseOptions['query']['multi_match']['fields'] = ['name^3', 'description'];
		}
		$filteredQuery = $this->queryBase($baseOptions, $filter);

		//category path restriction
		$baseFilter = (isset($filteredQuery['body']['query']['filtered']['filter']))
			? $filteredQuery['body']['query']['filtered']['filter']
			: [];
		if (!empty($path)) {
			$baseFilter['bool']['must'][]['term']['searchPaths'] = $path;
		}

		// product name has to be set
		$baseFilter['bool']['must'][]['exists']['field'] = 'name';

		// variants setup
		$showVariantsInList = (\Website\Tool\Utils::getEshopSettings()->getShowVariantsInLists());
		if ($showVariantsInList) {
			$tmpCount = count($baseFilter['bool']['must']);
			//is variant and is not hidden (like size)
			$baseFilter['bool']['must'][$tmpCount]['or'][0]['and'][]['term']['isVariant'] = true;
			$baseFilter['bool']['must'][$tmpCount]['or'][0]['and'][]['term']['size'] = 0;
			//is master product and has no visible variants (like color)
			$baseFilter['bool']['must'][$tmpCount]['or'][1]['and'][]['term']['isMasterProduct'] = true;
			$baseFilter['bool']['must'][$tmpCount]['or'][1]['and'][]['missing']['field'] = 'colors';
			//is no master product nor variant
			$baseFilter['bool']['must'][$tmpCount]['or'][2]['and'][]['term']['isVariant'] = false;
			$baseFilter['bool']['must'][$tmpCount]['or'][2]['and'][]['term']['isMasterProduct'] = false;
		} else {
			$baseFilter['bool']['must'][]['term']['isVariant'] = false;
		}
		$filteredQuery['body']['query']['filtered']['filter'] = $baseFilter;

		/*	PREPARE FILTER PARTS	*/

		$country = $this->currencyModel->getCountry();
		$filterParts = [];
		//brands
		if (!empty($filter['brand'])) {
			$filterParts['brands'] = ['terms' => ['brand' => $filter['brand']]];
		}
		//colors
		if (!empty($filter['color'])) {
			if ($showVariantsInList) {
				$filterParts['colors'] = ['terms' => ['color' => $filter['color']]];
			} else {
				$filterParts['colors'] = ['terms' => ['colors' => $filter['color']]];
			}
		}
		//sizes
		if (!empty($filter['size'])) {
			if ($showVariantsInList) {
				$filterParts['sizes'] = ['terms' => ['sizes' => $filter['size']]];
			} else {
				//@TODO complete this after multi-level variant attributes propagation finished
			}
		}
		//price ranges
		if (!empty($filter['price'])) {
			$priceFilter = [];
			$priceBuckets = \Website\Tool\Utils::getFilterPriceBuckets($country, true);
			foreach ($filter['price'] as $key) {
				if (!empty($priceBuckets[$key])) {
					$priceFilter['or'][]['range']['discountedPrice'.$country] = $priceBuckets[$key];
				}
			}
			if (!empty($priceFilter['or'])) {
				$filterParts['prices'] = $priceFilter;
			}
		}
		//new
		if (!empty($filter['new'])) {
			$filterParts['new'] = ['term' => ['isNew' => true]];
		}
		//on sale
		if (!empty($filter['sale'])) {
			$filterParts['sale'] = ['term' => ['isOnSale' => true]];
		}

		/* AGGREGATIONS */

		$aggregations = [];
		list($aggregations['priceMin'], $aggregations['priceMax']) = $this->priceRangeAggregation($filteredQuery);
		$aggregations['price'] = $this->priceBucketsAggregation($filteredQuery, $filterParts, $country);
		$aggregations['color'] = $this->colorsAggregation($filteredQuery, $filterParts, $showVariantsInList);
		$aggregations['size'] = $this->sizesAggregation($filteredQuery, $filterParts, $showVariantsInList);
		$aggregations['brand'] = $this->brandsAggregation($filteredQuery, $filterParts);
		$aggregations += $this->singleFieldAggregations($filteredQuery, $filterParts);

		//add filter parts to base query
		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'] = array_merge(
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'],
			array_values($filterParts)
		);

		$pagination = $this->query($filteredQuery, $filter);
		$pagination->aggregations = $aggregations;

		return $pagination;
	}

	public function getMostSoldProducts($limit = 3)
	{
		$filteredQuery = $this->queryBase([
			'type' => 'product_'.$this->language,
		]);

		return $this->query($filteredQuery, [
			'limit' => $limit,
			'sort' => ['soldCount' => ['order' => 'desc']],
		], false);
	}

	//@TODO multi level variants fix
	public function getRelatedProducts($path, $id, $limit = 3)
	{
		$filteredQuery = $this->queryBase([
			'type' => 'product_'.$this->language,
			'query' => 'random_score',
		]);

		$filteredQuery['body']['query']['filtered']['filter']['bool']['must_not'][]['term']['id'] = $id;
		if ($path) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['searchPaths'] = $path;
		}

		return $this->query($filteredQuery, ['limit' => $limit], false);
	}

	public function getLastViewedProducts($idList)
	{
		if (empty($idList)) {
			return [];
		}

		$filteredQuery = $this->queryBase(
			['type' => 'product_'.$this->language],
			['ids' => $idList, 'ids_score' => true]
		);

		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['ids']['values'] = $idList;

		$resultSet = $this->query($filteredQuery, [], false, false);

		return $resultSet['hits']['hits'];
	}

	public function getVariants($productId)
	{
		$filteredQuery = $this->queryBase([
			'type' => 'product_'.$this->language,
		]);

		$filter = (isset($filteredQuery['body']['query']['filtered']['filter']))
			? $filteredQuery['body']['query']['filtered']['filter']
			: [];
		$tmpCount = count($filter['bool']['must']);
		$filter['bool']['must'][$tmpCount]['term']['published'] = true;
		$filter['bool']['must'][$tmpCount + 1]['term']['isVariant'] = true;
		$filter['bool']['must'][$tmpCount + 2]['term']['parentId'] = (int) $productId;

		$filteredQuery['body']['query']['filtered']['filter'] = $filter;

		$resultSet = $this->query($filteredQuery, [], false, false);

		//@TODO multi-level support
		$hits = $resultSet['hits']['hits'];
		foreach ($hits as $key => $hit) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][$tmpCount + 2]['term']['parentId'] = $hit['_source']['id'];
			$resultSet = $this->elasticClient->search($filteredQuery);
			if (!empty($resultSet['hits']['hits'])) {
				unset($hits[$key]);
				foreach ($resultSet['hits']['hits'] as $tmpHit) {
					$hits[] = $tmpHit;
				}
			}
		}

		return $hits;
	}

	public function fulltextSearch($filter = [])
	{
		// query base
		$baseOptions = ['type' => [
			\Website\Tool\ElasticSearch::DOCUMENT_FULLTEXT_TYPE_KEY.'_'.$this->language,
			\Website\Tool\ElasticSearch::OBJECT_FULLTEXT_TYPE_KEY.'_'.$this->language,
		]];
		if (!empty($filter['query'])) {
			$baseOptions['query']['multi_match']['query'] = $filter['query'];
			$baseOptions['query']['multi_match']['fields'] = ['title^4', 'keywords^3', 'description^2', 'content'];
		}
		$filteredQuery = $this->queryBase($baseOptions, $filter);

		$aggregations = [];
		$aggregations['type'] = $this->fulltextTypeAggregation($filteredQuery);

		if (!empty($filter['type'])) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['type'] = $filter['type'];
		}

		$pagination = $this->query($filteredQuery, $filter, true, true, true);
		$pagination->aggregations = $aggregations;

		return $pagination;
	}

	private function fulltextTypeAggregation($query)
	{
		$aggQuery = [];
		$aggQuery['type']['terms']['size'] = 100;
		$aggQuery['type']['terms']['field'] = 'type';

		$query['body']['aggs'] = $aggQuery;
		$result = $this->elasticClient->search($query);

		$types = [];
		foreach ($result['aggregations']['type']['buckets'] as $bucket) {
			$types[$bucket['key']] = $bucket['doc_count'];
		}

		return $types;
	}

	private function priceRangeAggregation($query)
	{
		$aggQuery = [];
		$aggQuery['min_price']['min']['field'] = 'discountedPrice';
		$aggQuery['max_price']['max']['field'] = 'discountedPrice';
		$query['body']['aggs'] = $aggQuery;
		$result = $this->elasticClient->search($query);

		return [$result['aggregations']['min_price']['value'], $result['aggregations']['max_price']['value']];
	}

	private function priceBucketsAggregation($query, $filterParts, $country)
	{
		unset($filterParts['prices']);

		$priceBuckets = \Website\Tool\Utils::getFilterPriceBuckets($country);
		$aggQuery = [];
		$aggQuery['price']['range']['field'] = 'discountedPrice'.$country;
		$aggQuery['price']['range']['ranges'] = $priceBuckets;

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$prices = [];
		foreach ($result['aggregations']['price']['buckets'] as $key => $bucket) {
			$prices[$key] = $bucket['doc_count'];
		}

		return $prices;
	}

	private function colorsAggregation($query, $filterParts, $showVariantsInList)
	{
		unset($filterParts['colors']);

		//colors
		$aggQuery = [];
		$aggQuery['color']['terms']['size'] = 100;
		if ($showVariantsInList) {
			$aggQuery['color']['terms']['field'] = 'color';
		} else {
			$aggQuery['color']['terms']['field'] = 'colors';
		}

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$colors = [];
		foreach ($result['aggregations']['color']['buckets'] as $bucket) {
			if ($bucket['key']) {
				$colors[$bucket['key']] = $bucket['doc_count'];
			}
		}

		return $colors;
	}

	private function sizesAggregation($query, $filterParts, $showVariantsInList)
	{
		$tmp = !empty($filterParts['sizes']) ? $filterParts['sizes'] : null;
		unset($filterParts['sizes']);

		//sizes
		$aggQuery = [];
		if ($showVariantsInList) {
			$aggQuery['size']['terms']['size'] = 100;
			$aggQuery['size']['terms']['field'] = 'sizes';
		} else {
			//@TODO complete this after multi-level variant attributes propagation finished
			$aggQuery = new \stdClass();
		}

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		if ($tmp) {
			$query['body']['query']['filtered']['filter']['bool']['must_not'][] = $tmp;
		}
		$result = $this->elasticClient->search($query);

		$sizes = [];
		if (!empty($result['aggregations']['size'])) {
			foreach ($result['aggregations']['size']['buckets'] as $bucket) {
				if ($bucket['key']) {
					$sizes[$bucket['key']] = $bucket['doc_count'];
				}
			}
		}

		return $sizes;
	}

	private function brandsAggregation($query, $filterParts)
	{
		unset($filterParts['brands']);
		//brands
		$aggQuery = [];
		$aggQuery['brand']['terms']['size'] = 100;
		$aggQuery['brand']['terms']['field'] = 'brand';

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$brands = [];
		foreach ($result['aggregations']['brand']['buckets'] as $bucket) {
			$brands[$bucket['key']] = $bucket['doc_count'];
		}

		return $brands;
	}

	private function singleFieldAggregations($query, $filterParts)
	{
		$aggQuery = [];
		//news
		$aggQuery['new']['filter']['term']['isNew'] = true;
		//on sale
		$aggQuery['sale']['filter']['term']['isOnSale'] = true;

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		return [
			'new' => $result['aggregations']['new']['doc_count'],
			'sale' => $result['aggregations']['sale']['doc_count'],
		];
	}

	public function suggest($query, $size = 10)
	{
		$suggestQuery = [
			'index' => $this->index,
			'body' => [
				'suggestions' => [
					'text' => $query,
					'completion' => ['field' => 'name_suggest', 'size' => $size],
				],
			],
		];

		$suggestions = [];
		$resultSet = $this->elasticClient->suggest($suggestQuery);
		foreach ($resultSet['suggestions'][0]['options'] as $result) {
			$suggestions[] = $result['text'];
		}

		return $suggestions;
	}

	private function queryBase($options = [], $filter = [])
	{
		$filteredQuery['index'] = $this->index;
		$filteredQuery['type'] = $options['type'];

		// set query for filtered query
		$query = (!empty($options['query'])) ? $options['query'] : null;
		if (is_array($query)) {
			$filteredQuery['body']['query']['filtered']['query'] = $query;
		} elseif ($query == 'random_score') {
			$filteredQuery['body']['query']['filtered']['query']['function_score']['functions'][0]['random_score'] = new \stdClass();
		} elseif (!empty($filter['ids_score']) && !empty($filter['ids'])) {
			$functionScore = [
				'boost_mode' => 'replace',
				'script_score' => [
					'lang' => 'groovy',
					'script' => "-ids.indexOf(doc['id'].value.toInteger())",
					'params' => [
						'ids' => $filter['ids'],
					],
				],
			];
			$filteredQuery['body']['query']['filtered']['query']['function_score'] = $functionScore;
		} else {
			$filteredQuery['body']['query']['filtered']['query']['match_all'] = [];
		}

		// add filters
		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['published'] = true;

		return $filteredQuery;
	}

	private function constructSimplePagination($resultSet, $limit, $page, $totalCount)
	{
		$pagination = new \stdClass();
		$pagination->current = $page;
		$pagination->pageCount = ($limit) ? (int) ceil($totalCount / $limit) : 1;
		if ($page > 1) {
			$pagination->previous = $page - 1;
		}
		if ($page < $pagination->pageCount) {
			$pagination->next = $page + 1;
		}
		$pagination->totalCount = $totalCount;
		$pagination->first = 1;
		$pagination->last = $pagination->pageCount;
		$pagination->items = empty($resultSet) ? [] : $resultSet['hits']['hits'];
		$pagination->urlprefix = $this->urlPrefix;

		return $pagination;
	}

	private function query($filteredQuery, $filter, $foldedFallback = true, $paginate = true, $highlighting = false)
	{
		//add folded fallback fields
		$fulltextFields = [];
		if ($foldedFallback && !empty($filter['query'])) {
			$fulltextFields = $filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'];
			$fields = [];
			foreach ($fulltextFields as $field) {
				$parts = explode('^', $field);
				$foldedField = $parts[0].'.folded';
				if (!empty($parts[1])) {
					$foldedField .= '^'.$parts[1];
				}
				$fields[] = $field;
				$fields[] = $foldedField;
			}
			$filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'] = $fields;
		}

		//get total count
		$countQuery = ['index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => ['query' => $filteredQuery['body']['query']]];
		$countResult = $this->elasticClient->count($countQuery);
		$totalCount = $countResult['count'];

		//sort, only when no searchQuery is set and some results matched and some sort is defined...
		if ($totalCount && empty($filter['query']) && !empty($filter['sort'])
		&& empty($filter['ids_score'])) {
			$sortStruct = $filter['sort'];
			if (is_array($sortStruct)) {
				foreach ($sortStruct as $key => $sort) {
					$filteredQuery['body']['sort'][$key] = $sort;
				}
			} elseif (stristr($sortStruct, ':')) {
				$gps = explode(':', $sortStruct);
				$filteredQuery['body']['sort']['_geo_distance'] = [
					'location' => str_replace(':', ',', $sortStruct),
					'order' => 'asc',
					'unit' => 'km',
					'distance_type' => 'plane',
				];
			} else {
				//TODO define sorting types
			}
		}

		//add highlighting
		if ($highlighting && $fulltextFields && !empty($filter['query'])) {
			$highlight = ['fields' => []];
			foreach ($fulltextFields as $field) {
				$parts = explode('^', $field);
				$highlight['fields'][$parts[0]] = [
					'matched_fields' => [$parts[0]],
					'type' => 'fvh',
				];
				if ($foldedFallback) {
					$highlight['fields'][$parts[0]]['matched_fields'][] = $parts[0].'.folded';
				}
			}
			$filteredQuery['body']['highlight'] = $highlight;
		}

		//pagination
		$limit = !empty($filter['limit']) ? intval($filter['limit']) : 999;
		if ($paginate) {
			$page = !empty($filter['page']) ? intval($filter['page']) : 1;
			if ((int) $page > ceil($totalCount / $limit)) {
				$page = ceil($totalCount / $limit);
			}
			if ($page < 1) {
				$page = 1;
			}
			$filteredQuery['body']['from'] = isset($filter['offset']) ? ($page - 1) * $limit : 0;
			$filteredQuery['body']['size'] = isset($filter['offset']) ? $limit : $page * $limit;
		} else {
			$filteredQuery['body']['size'] = $limit;
		}
		//query
		$resultSet = $this->elasticClient->search($filteredQuery);
		if ($paginate) {
			return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
		} else {
			return $resultSet;
		}
	}
}
