<?php

namespace Website\Model;

class Product extends \Pimcore\Model\Object\Product implements IElasticObject, IElasticObjectFulltext
{
	use ElasticTrait;

	protected $noChain = false;
	protected $noMappedCode = false;

	/* HELPERS */

	public function getCategory()
	{
		$category = parent::getParent();
		while ($category instanceof self) {
			$category = $category->getParent();
		}

		return ($category instanceof Category) ? $category : null;
	}

	public function getPreviewImage()
	{
		$images = parent::getGallery();
		if (is_array($images)) {
			foreach ($images as $image) {
				if ($image instanceof \Pimcore\Model\Asset\Image) {
					return $image;
				}
			}
		}

		return;
	}

	//descrease inStock, NOTICE - (string) because 0 can't be set in pimcore number data component - say's its not set...
	public function decreaseInStockSize($count = 1)
	{
		$this->setInStock((string) max([0, $this->getInStock() - $count]));
		$this->setSoldCount((string) $this->getSoldCount() + $count);
		$this->setNoMappedCode(false); //for elastic search soldCount
		parent::save();
		$this->elasticSearchUpdate();
	}

	/*								Mapped										*/

	public function save()
	{
		$systemLanguages = \Pimcore\Tool::getValidLanguages();

		if (parent::getId() && !\Website\Tool\ElasticSearch::isUpdateRequest() && !$this->noMappedCode) {
			$parentObject = parent::getParent();
			//get variant level
			$variantLevel = 0;
			$upperLevel = $parentObject;
			while ($upperLevel instanceof self) {
				$variantLevel += 1;
				$upperLevel = $upperLevel->getParent();
			}
			//check variant restrictions upon product creation
			$variantAttributes = null;
			if ($variantLevel > 0 && parent::isPublished()) {
				if ($variantLevel > 2) {
					$msg = \Pimcore\Model\Translation\Admin::getByKeyLocalized(
						'error_not_implemented_yet', false, true
					);
					throw new \Pimcore\Model\Element\ValidationException($msg);

					return false;
				}
				//variant attribute has to be set
				$variantAttributes = parent::getVariantAttributes()->getItems(true);
				if (!count($variantAttributes) || count($variantAttributes) < $variantLevel) {
					$msg = \Pimcore\Model\Translation\Admin::getByKeyLocalized(
						'error_no_variant_attribute_set', false, true
					);
					throw new \Pimcore\Model\Element\ValidationException($msg);

					return false;
				}
				if (count($variantAttributes) > $variantLevel) {
					$msg = \Pimcore\Model\Translation\Admin::getByKeyLocalized(
						'error_too_many_variant_attributes', false, true
					);
					throw new \Pimcore\Model\Element\ValidationException($msg);

					return false;
				}
				//other same-level variants have to have same variant attributes set
				$variants = $parentObject->getChildren([\Pimcore\Model\Object\AbstractObject::OBJECT_TYPE_VARIANT]);
				$variantAttributesForComparison = $this->prepareVariantAttributesForComparison($variantAttributes);
				foreach ($variants as $variant) {
					$tmpVariantAttributes = $variant->getVariantAttributes()->getItems(true);
					if ($variant instanceof self && $variant->isPublished() && $this->prepareVariantAttributesForComparison($tmpVariantAttributes) != $variantAttributesForComparison) {
						$msg = \Pimcore\Model\Translation\Admin::getByKeyLocalized(
							'error_different_variant_attribute_types', false, true
						);
						throw new \Pimcore\Model\Element\ValidationException($msg);

						return false;
					}
				}
				//multi level variants hierarchy restrictions
				//1) color variants cannot be children of size variants
				if ($variantLevel == 2 && $this->getVariantAttributes()->getColor() instanceof \Pimcore\Model\Object\Objectbrick\Data\Color && $this->getVariantAttributes()->getColor()->getColor()) {
					$msg = \Pimcore\Model\Translation\Admin::getByKeyLocalized(
						'error_variant_hierarchy_restriction_size_color', false, true
					);
					throw new \Pimcore\Model\Element\ValidationException($msg);

					return false;
				}
			}

			//variant stuff (master product handling, variant attribute Ids grouping on parent products)
			//@TODO grouping of grouped Ids on parents parent... and multi-level support of that...
			$supportedAttributes = ['color', 'size'];
			if ($variantLevel > 0) {
				$saveParent = false;
				//get sibblings
				$sibblingList = new Product\Listing();
				$sibblingList->setObjectTypes([\Pimcore\Model\Object\AbstractObject::OBJECT_TYPE_VARIANT]);
				$sibblingList->setCondition('o_parentId = '.$parentObject->getId().' AND o_published = 1 AND o_id <> '.parent::getId());
				$sibblingList->load();
				//get all same level variants
				$variants = $sibblingList->getItems(0, 0);
				if (parent::getPublished()) {
					$variants[] = $this;
				}
				$variantCount = count($variants);
				//change parents master product status if needed
				$parentIsMasterProduct = ($variantCount);
				if ($parentIsMasterProduct != ($parentObject->getIsMasterProduct())) {
					$parentObject->setNoChain(true);
					$parentObject->setNoMappedCode(true);
					$parentObject->setIsMasterProduct($parentIsMasterProduct);
					$saveParent = true;
				}
				//group the variants and save a serialized list on parent if needed
				foreach ($supportedAttributes as $attribute) {
					$serialized = $this->getSerializedVariantAttributeIds($variants, $attribute);
					if ($serialized != $parentObject->{'get'.ucfirst($attribute).'s'}()) {
						$parentObject->{'set'.ucfirst($attribute).'s'}($serialized);
						$saveParent = true;
					}
				}
				//save parent if needed
				if ($saveParent) {
					$parentObject->setOmitMandatoryCheck(true);
					$parentObject->save();
				}
			} else {
				foreach ($supportedAttributes as $attribute) {
					$this->{'set'.ucfirst($attribute).'s'}('');
				}
				$this->setIsMasterProduct(false);
			}

			//localized stuff
			foreach ($systemLanguages as $lang) {
				//variant name (for now, it is the same as name)
				$variantName = $this->getName($lang);
				parent::setVariantName($variantName, $lang);
				//translated path
				$translatedPath = parent::getTranslatedPath($lang);
				$category = $this->getCategory();
				$slug = \Website\Tool\Utils::webalize($variantName);
				if ($category instanceof Category) {
					$newTranslatedPath = $category->getTranslatedPath($lang).'/'.$slug;
				} else {
					$newTranslatedPath = $slug;
				}
				//check if translated path is unique
				$list = $this->getList();
				$list->setCondition('o_id <> '.(int) parent::getId().' AND translatedPath = '.$list->quote($newTranslatedPath));
				if ($list->count()) {
					$msg = \Pimcore\Model\Translation\Admin::getByKeyLocalized(
						'error_product_name_not_unique', false, true
					);
					throw new \Pimcore\Model\Element\ValidationException($msg);

					return;
				}
				//translatedPath is going to change => add old path to redirects
				if (($translatedPath != $newTranslatedPath) && !empty($translatedPath)) {
					\Website\Tool\Utils::addToOldEshopUrls(parent::getId(), parent::getClassId(), $translatedPath, $lang);
				}
				parent::setTranslatedPath($newTranslatedPath, $lang);
			}

			//calculate exchange rate base prices and sale promotions
			\Website\Tool\Utils::forceInheritanceAndFallbackValues();
			$currencyModel = \Website\Tool\Utils::getCurrencyModel();
			$systemCurrencies = $currencyModel->getSystemCurrencies();
			$baseLocale = $currencyModel->getBaseLocale();
			$basePriceGetter = 'getPrice'.substr($baseLocale, 3);
			$baseCostGetter = 'getCost'.substr($baseLocale, 3);
			$baseDiscountedPriceGetter = 'getDiscountedPrice'.substr($baseLocale, 3);
			foreach ($currencyModel->getSystemCurrenciesOptions(true) as $locale => $currencyOptions) {
				$country = substr($locale, 3);
				$priceGetter = 'getPrice'.$country;
				$priceSetter = 'setPrice'.$country;
				$costSetter = 'setCost'.$country;
				$ignoreSalePromotionsGetter = 'getIgnoreSalePromotions'.$country;
				$discountedPriceSetter = 'setDiscountedPrice'.$country;

				//calculate exchange rate based prices and costs
				if ($locale != $baseLocale && $currencyOptions['rate'] && $currencyOptions['rate'] != 1) {
					parent::$priceSetter($currencyModel->localizeCurrency(parent::$basePriceGetter(), $systemCurrencies[$locale], true));
					parent::$costSetter($currencyModel->localizeCurrency(parent::$baseCostGetter(), $systemCurrencies[$locale], true));
					parent::$discountedPriceSetter($currencyModel->localizeCurrency(parent::$baseDiscountedPriceGetter(), $systemCurrencies[$locale], true));
				}

				//sale promotion stuff... yeah, it's ugly
				if (!parent::$ignoreSalePromotionsGetter()) {
					$price = parent::$priceGetter();
					$discount = 0;
					foreach (\Website\Model\DiscountManager::getSalePromotions() as $path => $promotions) {
						if (strpos(parent::getFullPath().'/', $path.'/') === 0) {
							foreach ($promotions as $promotion) {
								if ($promotion['isPercentual']) {
									$percents = min(100, $promotion['discount'.$country]);
									$discount += $price / 100 * $percents;
								} else {
									$discount += $promotion['discount'.$country];
								}
							}
						}
					}
					parent::$discountedPriceSetter(max([0, $price - $discount]));
				}
			}
			\Website\Tool\Utils::restorInheritanceAndFallbackValues();
		}

		//save the object
		parent::save();

		if (!$this->noChain) {
			//update all childs; expensive because it creates an chain saving...
			foreach (parent::getChildren([\Pimcore\Model\Object\AbstractObject::OBJECT_TYPE_VARIANT]) as $child) {
				if ($child->isPublished()) {
					$child->setOmitMandatoryCheck(true);
					$child->save();
				}
			}
		}

		$this->elasticSearchUpdate();
		$this->elasticSearchUpdateFulltext();
	}

	private function prepareVariantAttributesForComparison(&$variantAttributes)
	{
		$attributes = [];
		foreach ($variantAttributes as $attribute) {
			$attributes[] = get_class($attribute);
		}

		sort($attributes);

		return $attributes;
	}

	private function getSerializedVariantAttributeIds(&$list, $attribute)
	{
		$getter = 'get'.ucfirst($attribute);
		$ids = [];
		foreach ($list as $item) {
			if ($item->getVariantAttributes()->$getter() && $item->getVariantAttributes()->$getter()->$getter()) {
				$ids[] = $item->getVariantAttributes()->$getter()->$getter()->getId();
			}
		}

		if (empty($ids)) {
			return '0'; // we return '0' because empty string would not be saved on variant if parent is not empty due to inheritance bug in pimcore
		} else {
			sort($ids);

			return implode(',', $ids);
		}
	}

	public function delete()
	{
		//unpublish (for possible master product deactivation)
		try {
			if ($this->getPublished()) {
				$this->setNoChain(true);
				$this->setPublished(false);
				$this->setOmitMandatoryCheck(true);
				$this->save();
			}
		} catch (\Exception $e) {
			\Pimcore\Log\Simple::log('elasticsearch', 'Error when saving object before deleting it, ID: '.$this->getId());
		}

		$this->elasticSearchDelete();
		$this->elasticSearchDeleteFulltext();

		parent::delete();
	}

	public function setNoChain($noChain)
	{
		$this->noChain = $noChain;
	}

	public function setNoMappedCode($noMappedCode)
	{
		$this->noMappedCode = $noMappedCode;
	}

	/* 								For elastic search							 */

	public function elasticSearchLoad($language, $extend = [])
	{
		//add grouped variant attributes
		//read without inheritance
		\Website\Tool\Utils::forceInheritanceAndFallbackValues(false);
		$colors = parent::getColors();
		$colors = empty($colors) ? [] : explode(',', parent::getColors());
		$sizes = parent::getSizes();
		$sizes = empty($sizes) ? [] : explode(',', parent::getSizes());
		\Website\Tool\Utils::restorInheritanceAndFallbackValues();

		\Website\Tool\Utils::forceInheritanceAndFallbackValues();

		//add categories
		$mainCategory = $this->getCategory();
		if ($mainCategory) {
			$searchPaths = [$mainCategory->getTranslatedPath($language)];
		}
		foreach ((array) $this->getAdditionalCategories() as $category) {
			$searchPaths[] = $category->getTranslatedPath($language);
		}

		//add variant attributes
		$attributeColor = $this->getVariantAttributes()->getColor();
		if ($attributeColor instanceof \Pimcore\Model\Object\Objectbrick\Data\Color && $attributeColor->getColor() instanceof \Pimcore\Model\Object\ProductColor) {
			$color = $attributeColor->getColor()->getId();
		} else {
			$color = 0;
		}
		$attributeSize = $this->getVariantAttributes()->getSize();
		if ($attributeSize instanceof \Pimcore\Model\Object\Objectbrick\Data\Size && $attributeSize->getSize() instanceof \Pimcore\Model\Object\ProductSize) {
			$size = $attributeSize->getSize()->getId();
		} else {
			$size = 0;
		}

		$data = array_merge($this->getElasticBaseData(), [
			'name' => $this->getVariantName($language),
			'name_suggest' => $this->createElasticSuggestion($this->getVariantName($language), ['id' => (int) parent::getId()]),
			'description' => $this->normalizeElasticText(parent::getDescription($language)),
			'translatedPath' => parent::getTranslatedPath($language),
			'searchPaths' => $searchPaths,
			'previewImage' => \Website\Tool\Utils::image($this->getPreviewImage(), 'ProductPreviewImage', null, ['class' => 'img-responsive']),
			'priceCZ' => (float) parent::getPriceCZ(),
			'priceSK' => (float) parent::getPriceSK(),
			'discountedPriceCZ' => (float) parent::getDiscountedPriceCZ(),
			'discountedPriceSK' => (float) parent::getDiscountedPriceSK(),
			'inStock' => (int) parent::getInStock(),
			'isNew' => (bool) parent::getIsNew(),
			'isOnSale' => (parent::getDiscountedPriceCZ() < parent::getPriceCZ()), // maybe differ between localized prices
			'soldCount' => (int) parent::getSoldCount(),
			'isVariant' => parent::getType() == \Pimcore\Model\Object\Concrete::OBJECT_TYPE_VARIANT,
			'brand' => (int) (parent::getBrand() instanceof \Pimcore\Model\Object\Brand) ? parent::getBrand()->getId() : 0,
			'color' => $color,
			'size' => $size,
			'colors' => $colors,
			'sizes' => $sizes,
			'ordering' => (int) parent::getOrdering(),
		]);

		\Website\Tool\Utils::restorInheritanceAndFallbackValues();

		\Website\Tool\Utils::forceInheritanceAndFallbackValues(false);
		$data['isMasterProduct'] = (bool) parent::getIsMasterProduct(); // read without inheritance
		\Website\Tool\Utils::restorInheritanceAndFallbackValues();

		$data = array_replace_recursive($data, $extend);

		return $data;
	}

	public function elasticSearchLoadFulltext($lang)
	{
		\Website\Tool\Utils::forceInheritanceAndFallbackValues();

		$data = array_merge($this->getElasticBaseData(), [
			'path' => $this->getUrl($lang),
			'title' => $this->getVariantName($lang),
			'description' => $this->getSeoDescription($lang),
			'content' => $this->normalizeElasticText($this->getDescription($lang)),
		]);

		\Website\Tool\Utils::restorInheritanceAndFallbackValues();

		return $data;
	}
}
