<?php

namespace Website\Tool;

class Utils
{
	//hardcoded pimcore IDs

	const FOLDER_CATALOG = 2;
	const FOLDER_DISCOUNT_CODES = 9;
	const FOLDER_SALE_PROMOTIONS = 11;
	const DOCUMENT_ROOT = 2;
	const DOCUMENT_ACCOUNT = 12;
	const DOCUMENT_CART = 17;
	const DOCUMENT_SEARCH_RESULTS = 27;
	const DOCUMENT_COMPARE_PRODUCTS = 20;
	const OBJECT_ESHOP_SETTINGS = 12;

	/**
	 * @var \Website\Model\Currency
	 */
	protected static $currencyModel = null;
	private static $inheritance;
	private static $fallback;
	private static $languageSwitch = null;
	public static $defaultLanguage = null;
	public static $shippingCountries = null;

	/**
	 * defines the valid frontend languages as a subset of valid admin languages
	 * used when we need to add another language in the admin, but don't want to break the site while filling in all the localized data...
	 *
	 * @return array valid languages (ISO 639-1)
	 */
	public static function getValidFrontendLanguages()
	{
		$config = \Pimcore\Config::getWebsiteConfig();
		$allowed = explode(',', $config->get('frontendLanguages', self::getDefaultLanguage()));
		$combined = [];

		foreach (\Pimcore\Tool::getValidLanguages() as $language) {
			if (in_array($language, $allowed)) {
				$combined[] = $language;
			}
		}

		return $combined;
	}

	/**
	 * returns the default website language (the first from system settings)
	 * default language should match the language property of master document tree.
	 *
	 * @return string default language ISO 639-1
	 */
	public static function getDefaultLanguage()
	{
		if (self::$defaultLanguage === null) {
			$config = \Pimcore\Config::getSystemConfig();
			self::$defaultLanguage = current(explode(',', $config->general->validLanguages));
		}

		return self::$defaultLanguage;
	}

	public static function detectLanguage()
	{
		$language = (\Zend_Registry::isRegistered('Zend_Locale'))
			? \Zend_Registry::get('Zend_Locale')->getLanguage()
			: self::getDefaultLanguage();

		return $language;
	}

	/**
	 * @return \Pimcore\Translate\Website
	 */
	public static function getTranslator()
	{
		$translator = null;
		if (\Zend_Registry::isRegistered('Zend_Translate')) {
			$translator = \Zend_Registry::get('Zend_Translate');
		} else {
			$language = self::detectLanguage();
			$translator = new \Pimcore\Translate\Website($language);
		}

		return $translator;
	}

	public static function thumbnail($image, $thumbnail, $fallbackImage = null)
	{
		if ($image instanceof \Pimcore\Model\Document\Tag\Image && $image->getImage()) {
			return (string) $image->getThumbnail($thumbnail);
		} elseif ($image instanceof \Pimcore\Model\Object\Data\Hotspotimage && $image->getImage()) {
			return (string) $image->getThumbnail($thumbnail);
		} elseif ($image instanceof \Pimcore\Model\Asset\Image) {
			return (string) $image->getThumbnail($thumbnail);
		} elseif ($fallbackImage instanceof \Pimcore\Model\Asset\Image) {
			return (string) $fallbackImage->getThumbnail($thumbnail);
		} elseif ($fallbackImage) {
			$image = \Pimcore\Model\Asset\Image::getById($fallbackImage);
			if (!$image) {
				$image = \Pimcore\Model\Asset\Image::getByPath($fallbackImage);
			}
			if ($image) {
				return (string) $image->getThumbnail($thumbnail);
			}
		} else {
			return '';
		}
	}

	/**
	 * @param mixed  $image             image asset or hotspot image
	 * @param string $thumbnail         image thumbnail pipeline name
	 * @param array  $additionalOptions additional options (see pimcore document image editable documentation)
	 * @param mixed  $fallbackImage     image asset or image asset ID or image asset path
	 *
	 * @return string HTML image element like the document image editable
	 */
	public static function image($image, $thumbnail, $fallbackImage = null, $additionalOptions = [])
	{
		$html = null;
		$imageTag = new \Pimcore\Model\Document\Tag\Image();
		if ($image instanceof \Pimcore\Model\Document\Tag\Image && $image->getImage()) {
			$imageTag = $image;
		} elseif ($image instanceof \Pimcore\Model\Asset\Image) {
			$imageTag->setImage($image);
		} elseif ($image instanceof \Pimcore\Model\Object\Data\Hotspotimage && $image->getImage()) {
			$html = $image->getThumbnail($thumbnail)->getHTML($additionalOptions);
		} elseif ($fallbackImage instanceof \Pimcore\Model\Asset\Image) {
			$imageTag->setImage($fallbackImage);
		} elseif ($fallbackImage) {
			$image = \Pimcore\Model\Asset\Image::getById($fallbackImage);
			if (!$image) {
				$image = \Pimcore\Model\Asset\Image::getByPath($fallbackImage);
			}
			if ($image) {
				$imageTag->setImage($image);
			}
		}

		if (!$html) {
			$additionalOptions['thumbnail'] = $thumbnail;
			$imageTag->setOptions($additionalOptions);
			$html = $imageTag->frontend();
		}

		return $html;
	}

	public static function link($link, $additionalAttributes = [], $text = null)
	{
		if ($link instanceof \Pimcore\Model\Document\Tag\Link) {
			$link->setOptions($additionalAttributes);
		} elseif ($link instanceof \Pimcore\Model\Object\Data\Link) {
			$link->setValues($additionalAttributes);
		}
		$tag = (string) $link;
		if ($text) {
			$tag = preg_replace('#(<a.*?>).*?(</a>)#', '$1'.$text.'$2', $tag);
		}

		return $tag;
	}

	/**
	 * @return \Website\Model\EshopSettings eshop settings
	 */
	public static function getEshopSettings()
	{
		if (\Zend_Registry::isRegistered('eshop_settings')) {
			return \Zend_Registry::get('eshop_settings');
		} else {
			if (!$settings = \Pimcore\Cache::load('eshop_settings')) {
				$settings = \Website\Model\EshopSettings::getById(self::OBJECT_ESHOP_SETTINGS);

				if (!$settings) {
					throw new \Exception('Eshop settings could not be loaded.');
				}

				\Pimcore\Cache::save($settings, 'eshop_settings', $settings->getCacheTags());
			}

			\Zend_Registry::set('eshop_settings', $settings);

			return $settings;
		}

		return $settings;
	}

	public static function getFilterPriceBuckets($country = 'CZ', $forFilter = false)
	{
		$to = 'to';
		$from = 'from';

		if ($forFilter) {
			$to = 'lt';
			$from = 'gte';
		}
		switch ($country) {
			case 'CZ': {
					return [
						0 => [$to => 500],
						1 => [$from => 501, $to => 1000],
						2 => [$from => 1001, $to => 2000],
						3 => [$from => 2001],
					];
				};
				break;
			case 'SK': {
					return [
						0 => [$to => 10],
						1 => [$from => 10, $to => 20],
						2 => [$from => 20],
					];
				};
				break;
			default: {
					return [];
				}
		}
	}

	public static function getFilterBrands()
	{
		$result = \Pimcore\Cache::load('filter_brands');
		if (!$result) {
			$result = [];
			$list = new \Pimcore\Model\Object\Brand\Listing();
			$list->load();
			foreach ($list as $item) {
				$result[$item->getId()] = $item->getName();
			}
			\Pimcore\Cache::save($result, 'filter_brands', ['output']);
		}

		return $result;
	}

	public static function getFilterColors($lang = 'cs')
	{
		$result = \Pimcore\Cache::load('filter_colors_'.$lang);
		if (empty($result)) {
			$result = [];
			$list = new \Pimcore\Model\Object\ProductColor\Listing();
			$list->load();
			foreach ($list as $item) {
				$result[$item->getId()] = $item->getName($lang);
			}
			\Pimcore\Cache::save($result, 'filter_colors_'.$lang, ['output']);
		}

		return $result;
	}

	public static function getFilterSizes($lang = 'cs')
	{
		$result = \Pimcore\Cache::load('filter_sizes_'.$lang);
		if (empty($result)) {
			$result = [];
			$list = new \Pimcore\Model\Object\ProductSize\Listing();
			$list->load();
			foreach ($list as $item) {
				$result[$item->getId()] = $item->getName($lang);
			}
			\Pimcore\Cache::save($result, 'filter_sizes_'.$lang, ['output']);
		}

		return $result;
	}

	/**
	 * @TODO countries
	 * for statistics
	 *
	 * @param string $paymentType
	 * @param string $shippingType
	 *
	 * @return float
	 */
	public static function getOurShippingPrice($paymentType, $shippingType, $country = 'CZ')
	{
		$settings = self::getEshopSettings();
		$ourShippingCost = $settings->{'getShippingCostOptions'.$country}()->data;
		$price = (int) $ourShippingCost[$paymentType][$shippingType];

		return $price;
	}

	//for global price modifications
	public static function applyGlobalPriceChange($price)
	{
		return $price;
	}

	//for administration
	public static function getCurrencyModel()
	{
		if (!self::$currencyModel) {
			self::$currencyModel = new \Website\Model\Currency();
		}

		return self::$currencyModel;
	}

	public static function url($staticRoute, $params = [])
	{
		if (!$staticRoute instanceof \Pimcore\Model\Staticroute) {
			$staticRoute = \Pimcore\Model\Staticroute::getByName($staticRoute, (\Pimcore\Model\Site::isSiteRequest()) ? \Pimcore\Model\Site::getCurrentSite()->getId() : 0);
		}

		if (!$staticRoute) {
			return '/';
		}

		return $staticRoute->assemble($params, true);
	}

	public static function docUrl($realFullPath, $language, $rootDoc = null)
	{
		$languageSwitch = self::getLanguageSwitch($language, $realFullPath, $rootDoc);

		return (isset($languageSwitch[$language])) ? $languageSwitch[$language] : '/';
	}

	public static function getLanguageSwitch($language, $section, $rootDoc = null, $hpFallback = false)
	{
		if (!$rootDoc) {
			$rootDoc = \Pimcore\Model\Document\Page::getById(1);
		}
		$siteId = (\Pimcore\Model\Site::isSiteRequest())
			? \Pimcore\Model\Site::getCurrentSite()->getRootId()
			: $rootDoc->getId();
		$cacheKey = 'language_switcher_'.$rootDoc->getId().'_'.$siteId;
		$switcher = (self::$languageSwitch[$rootDoc->getId()])
			? self::$languageSwitch[$rootDoc->getId()]
			: \Pimcore\Cache::load($cacheKey);
		if ($switcher === false) {
			$switcher = [];
			$languages = \Website\Tool\Utils::getValidFrontendLanguages();
			$originLanguage = $rootDoc->getProperty('language');
			$queue = [$rootDoc];
			$service = new \Pimcore\Model\Document\Service();
			while (!empty($queue)) {
				$doc = array_shift($queue);
				foreach ($doc->getChildren(true) as $child) {
					$queue[] = $child;
				}
				if (!$doc->getPublished()) {
					continue;
				}

				$docRealFullPath = $doc->getRealFullPath();
				$translations = $service->getTranslations($doc);
				if (!isset($translations[$originLanguage])) {
					$translations[$originLanguage] = $doc->getId();
				}

				foreach ($translations as $language => $translationDocId) {
					$translationDoc = \Pimcore\Model\Document::getById($translationDocId);
					if ($translationDoc && $translationDoc->isPublished()) {
						$switcher[$docRealFullPath][$language] = $translationDoc->getFullPath();
					}
				}
			}

			// global cache, for frontend requests
			if (!\Pimcore\Tool::isFrontentRequestByAdmin()) {
				\Pimcore\Cache::save($switcher, $cacheKey, ['output']);
			}
		}

		// per-request cache
		self::$languageSwitch[$rootDoc->getId()] = $switcher;

		$sectionSwitch = (is_array($switcher) && isset($switcher[$section])) ? $switcher[$section] : [];

		// add HPs for non-existing page localizations
		if ($hpFallback && is_array($switcher)) {
			$langParamPosition = stripos($section, '/'.$language.'/');
			if ($langParamPosition !== false) {
				$hpKey = substr($section, 0, $langParamPosition).'/'.$language;
				if (isset($switcher[$hpKey])) {
					$sectionSwitch += $switcher[$hpKey];
				}
			}
		}

		return $sectionSwitch;
	}

	public static function addToOldEshopUrls($id, $classId, $translatedPath, $lang)
	{
		$db = \Pimcore\Db::get();
		$alreadyThere = $db->select()->from('eshop_old_paths')
						->where('translatedPath = ?', $translatedPath)
						->where('class = ?', $classId)
						->where('language = ?', $lang)
						->query()->fetch();
		if (empty($alreadyThere)) {
			try {
				$db->insert('eshop_old_paths', [
					'o_id' => $id,
					'class' => $classId,
					'translatedPath' => $translatedPath,
					'language' => $lang,
				]);
			} catch (\Exception $e) {
				\Pimcore\Logger::error('REDIRECTS: Could not insert product with ID '.$id.' - '.$e->getMessage().' | '.$e->getTraceAsString());
			}
		}
	}

	public static function forceInheritanceAndFallbackValues($inherit = true)
	{
		self::$inheritance = \Pimcore\Model\Object\AbstractObject::getGetInheritedValues();
		self::$fallback = \Pimcore\Model\Object\Localizedfield::getGetFallbackValues();
		\Pimcore\Model\Object\AbstractObject::setGetInheritedValues($inherit);
		\Pimcore\Model\Object\Localizedfield::setGetFallbackValues($inherit);
	}

	public static function restorInheritanceAndFallbackValues()
	{
		\Pimcore\Model\Object\AbstractObject::setGetInheritedValues(self::$inheritance);
		\Pimcore\Model\Object\Localizedfield::setGetFallbackValues(self::$fallback);
	}

	public static function profilerStart()
	{
		$profiler = \Pimcore\Db::get()->getProfiler();
		$profiler->setEnabled(true);
	}

	public static function profilerStop()
	{
		$profiler = \Pimcore\Db::get()->getProfiler();
		if ($profiler->getQueryProfiles()) {
			foreach ($profiler->getQueryProfiles() as $query) {
				echo $query->getQuery().', time: '.$query->getElapsedSecs().'<br>';
			}
		} else {
			echo 'nada';
		}
		exit;
	}

	public static function loadAssets($rootFolderOrAssetArray = [], $allowedTypes = ['image'])
	{
		$queue = [];
		if ($rootFolderOrAssetArray instanceof \Pimcore\Model\Asset\Folder) {
			$queue[] = $rootFolderOrAssetArray;
		} elseif (is_array($rootFolderOrAssetArray)) {
			$queue = $rootFolderOrAssetArray;
		} else {
			$root = \Pimcore\Model\Asset\Folder::getById($rootFolderOrAssetArray);
			if ($root) {
				$queue[] = $root;
			}
		}

		$assets = [];
		while ($asset = array_shift($queue)) {
			if (!$asset instanceof \Pimcore\Model\Asset) {
				continue;
			}

			if ($asset instanceof \Pimcore\Model\Asset\Folder) {
				foreach ($asset->getChildren() as $child) {
					$queue[] = $child;
				}
			} elseif (in_array($asset->getType(), $allowedTypes)) {
				$assets[] = $asset;
			}
		}

		return $assets;
	}

	public static function loadMoreItems($listObjectOrClassName, $limit, $offsetId = 0, $orderKey = 'o_id', $order = 'asc', $additionalConditionsAsString = null)
	{
		$result = ['items' => [], 'offsetId' => null];

		//load the offset item if needed
		$itemClass = str_replace('\\Listing', '', (is_object($listObjectOrClassName)) ? get_class($listObjectOrClassName) : $listObjectOrClassName);
		if ($offsetId) {
			$offsetItem = $itemClass::getById($offsetId);
			if (!$offsetItem) {
				return $result;
			}
		}

		//just in case
		if ($orderKey == 'o_id' && substr($itemClass, 0, 6) != 'Object') {
			$orderKey = 'id';
		}

		//create the list
		$list = (is_object($listObjectOrClassName)) ? $listObjectOrClassName : new $listObjectOrClassName();
		if (!$list) {
			return $result;
		}

		//create the condition
		$conditions = [];
		if ($additionalConditionsAsString) {
			$conditions[] = $additionalConditionsAsString;
		}
		if ($list->getCondition()) {
			$conditions[] = $list->getCondition();
		}
		if ($offsetId) {
			$offsetValue = $offsetItem->{'get'.ucfirst($orderKey)}();
			if ($offsetValue instanceof \Pimcore\Date) {
				$offsetValue = $offsetValue->getTimestamp();
			}
			$conditions[] = sprintf('`%s` %s %s', $orderKey, ($order == 'asc') ? '>=' : '<=', $list->quote($offsetValue));
		}

		//load the list
		$list->setOrderKey($orderKey);
		$list->setOrder($order);
		$list->setLimit($limit + 1); // +1 because we want to check if there are more items
		$list->setCondition(implode(' AND ', $conditions));
		$list->load();

		//remove last item and set next offset ID
		$items = $list->getItems(0, 0);
		if (count($items) > $limit) {
			$tmp = array_pop($items);
			$result['offsetId'] = $tmp->getId();
		}
		$result['items'] = $items;

		return $result;
	}

	//Text stuff

	public static function webalize($path)
	{
		return trim(preg_replace('#-+#', '-', str_replace('.', '-', \Pimcore\File::getValidFilename($path))), '-');
	}

	public static function toCacheKey($text)
	{
		return str_replace(['-', '~'], '_', self::webalize($text));
	}

	public static function toObjectKey($text)
	{
		return preg_replace('#-+#', '-', str_replace(['_', '~'], '-', self::webalize($text)));
	}

	public static function cutStringRespectingWhitespace($string, $length, $suffix = '...')
	{
		if ($length < strlen($string)) {
			$text = substr($string, 0, $length);
			if (false !== ($length = strrpos($text, ' '))) {
				$text = substr($text, 0, $length);
			}
			$string = $text.$suffix;
		}

		return $string;
	}

	public static function mb_ucfirst($string, $encoding = 'utf-8')
	{
		$strlen = mb_strlen($string, $encoding);
		$firstChar = mb_substr($string, 0, 1, $encoding);
		$restOfString = mb_substr($string, 1, $strlen - 1, $encoding);

		return mb_strtoupper($firstChar, $encoding).$restOfString;
	}

	public static function mb_lcfirst($string, $encoding = 'utf-8')
	{
		$strlen = mb_strlen($string, $encoding);
		$firstChar = mb_substr($string, 0, 1, $encoding);
		$restOfString = mb_substr($string, 1, $strlen - 1, $encoding);

		return mb_strtolower($firstChar, $encoding).$restOfString;
	}

	public static function czechMonth($month)
	{
		static $arr = [1 => 'leden', 'únor', 'březen', 'duben', 'květen', 'červen', 'červenec', 'srpen', 'září', 'říjen', 'listopad', 'prosinec'];

		return $arr[$month];
	}

	public static function czechDay($day)
	{
		$days = ['neděle', 'pondělí', 'úterý', 'středa', 'čtvrtek', 'pátek', 'sobota'];

		return $days[$day];
	}

	public static function formatSizeUnits($size)
	{
		if ($size >= 1073741824) {
			$bytes = number_format($size / 1073741824, 2, ',', ' ').' GB';
		} elseif ($size >= 1048576) {
			$bytes = number_format($size / 1048576, 2, ',', ' ').' MB';
		} elseif ($size >= 1024) {
			$bytes = number_format($size / 1024, 2, ',', ' ').' KB';
		} elseif ($size > 1) {
			$bytes = $size.' B';
		} elseif ($size == 1) {
			$bytes = $size.' B';
		} else {
			$bytes = '0 B';
		}

		return $bytes;
	}

	public static function encrypt($text)
	{
		return trim(base64_encode(mcrypt_encrypt(MCRYPT_RIJNDAEL_256, self::SALT, $text, MCRYPT_MODE_ECB, mcrypt_create_iv(mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_ECB), MCRYPT_RAND))));
	}

	public static function decrypt($text)
	{
		return trim(mcrypt_decrypt(MCRYPT_RIJNDAEL_256, self::SALT, base64_decode($text), MCRYPT_MODE_ECB, mcrypt_create_iv(mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_ECB), MCRYPT_RAND)));
	}

	public static function serialize($data)
	{
		return base64_encode(serialize($data));
	}

	public static function unserialize($data)
	{
		return unserialize(base64_decode($data));
	}

	public static function detectCzechEncoding($s)
	{
		if (preg_match('#[\x80-\x{1FF}\x{2000}-\x{3FFF}]#u', $s)) {
			return 'UTF-8';
		}

		if (preg_match('#[\x7F-\x9F\xBC]#', $s)) {
			return 'WINDOWS-1250';
		}

		return 'ISO-8859-2';
	}

	/**
	 * Returns the total width in points of the string using the specified font and
	 * size.
	 *
	 * This is not the most efficient way to perform this calculation. I'm
	 * concentrating optimization efforts on the upcoming layout manager class.
	 * Similar calculations exist inside the layout manager class, but widths are
	 * generally calculated only after determining line fragments.
	 *
	 * @link http://devzone.zend.com/article/2525-Zend_Pdf-tutorial#comments-2535
	 *
	 * @param string                  $string
	 * @param \Zend_Pdf_Resource_Font $font
	 * @param float                   $fontSize Font size in points
	 *
	 * @return float
	 */
	public static function widthForStringUsingFontSize($string, $font, $fontSize)
	{
		$drawingString = iconv('UTF-8', 'UTF-16BE//IGNORE', $string);
		$characters = [];
		for ($i = 0; $i < strlen($drawingString); ++$i) {
			$characters[] = (ord($drawingString[$i++]) << 8) | ord($drawingString[$i]);
		}
		$glyphs = $font->glyphNumbersForCharacters($characters);
		$widths = $font->widthsForGlyphs($glyphs);
		$stringWidth = (array_sum($widths) / $font->getUnitsPerEm()) * $fontSize;

		return $stringWidth;
	}
}
