<?php

class ShopController extends \Website\Controller\BaseController 
{

	/* 						DOCUMENT ROUTED ACTIONS							 */

	public function productsAction()
	{
		$this->enableLayout();

		$this->setCanonicalUrl();

		$filterValues = $this->filterForm->getValidValues($this->getRequest()->getParams());

		$search = \Website\Model\ElasticSearch::getInstance($this->currencyModel, $this->language);

		$pagination = $search->getProducts(null, $filterValues);
		$pagination->ajax = true;

		$this->view->elasticSearch = $search;
		$this->view->pagination = $pagination;

		if ($this->isAjax()) {
			$this->ajaxResponseBuilder->addElement('#category-filter-wrapper', 'shop/snippets/category-filter.php')
				->addElement('#category-product-list-wrapper', 'shop/snippets/category-product-list.php')
				->sendResponse();
		}
	}

	public function cartAction()
	{
		$this->enableLayout();
		//mocking for administration
		if ($this->isEditOrAdminMode()) {
			$this->checkAuth();
		}

		$form = new \Website\Form\CartForm($this->cart);

		$ajaxRedirect = false;
		if ($this->getRequest()->isPost()) {
			if ($this->handleCartForm($form)) {
				if ($this->getRequest()->getPost('act') == 'recalculate') {
					if (!$this->isAjax()) {
						$this->gotoUrl($this->view->docUrl('/cs/kosik'));
						return;
					}
				} else {
					if (!$this->isAjax()) {
						$this->gotoUrl($this->view->docUrl('/cs/kosik/kontaktni-a-platebni-udaje'));
						return;
					} else {
						$ajaxRedirect = true;
					}
				}
			}
		}

		//set count values (because cart recalculation might change them)
		foreach ($this->cart->getOrder()->products as $key => $product) {
			$form->{'count_' . $key}->setValue($product->count);
		}

		$this->view->cartForm = $form;
		$this->view->products = $this->cart->getProductObjects();

		if ($this->isAjax()) {
			$this->ajaxResponseBuilder->addFlashMessages()
				->addElement('#cart-form-wrapper', 'shop/snippets/cart-form.php')
				->addElement('.cart-sum-wrapper', 'shop/snippets/cart-sum.php');
			if ($ajaxRedirect) {
				$this->ajaxResponseBuilder->addRedirect($this->view->docUrl('/cs/kosik/kontaktni-a-platebni-udaje'));
			}
			$this->ajaxResponseBuilder->sendResponse();
		}
	}

	public function cartInfoAction()
	{
		$this->enableLayout();
		//mocking for administration
		if ($this->isEditOrAdminMode()) {
			$this->checkAuth();
		} else {
			$tmpProducts = $this->cart->getOrder()->products;
			if (empty($tmpProducts)) {
				$this->gotoUrl($this->view->docUrl('/' . $this->defaultLanguage . '/kosik'));
			}
		}

		//@TODO cache this form
		$form = new \Website\Form\CartInfoForm(
			$this->shopModel->getShippingAndPaymentTypePrices(
				$this->cart->getOrder()->productsPrice,
				$this->cart->getProductsDiscount()->discountValue,
				($this->cart->getOrder()->discountCode && $this->cart->getProductsDiscount()->code->getApplyOnShipping()) ? true : false
			)
		);

		$order = $this->cart->getOrder();

		if ($this->getRequest()->isPost()) {
			if ($this->handleCartInfoForm($form)) {
				//if only applying discount, do not create order
				if ($this->getRequest()->getPost('act') == 'discount') {
					$this->gotoUrl($this->view->docUrl('/' . $this->defaultLanguage . '/kosik/kontaktni-a-platebni-udaje') . '#discountCode');
				}
				//to set shipping and payment prices
				$this->cart->recalculate();

				//create order
				$orderObject = $this->shopModel->createOrder($this->cart, $this->getCurrentUser($this->cart->getOrder()->contactInfo['fEmail']));
				if (!$orderObject instanceof \Website\Model\Order) {
					$this->addErrorMsg($this->translate('msg_unknown_error'));
					$this->gotoUrl($this->view->docUrl('/' . $this->defaultLanguage . '/kosik'));
					return;
				}
				//set orderId to cart so no double orders are created if for example credit card payment fails...
				$this->cart->setOrderId($orderObject->getId());

				if ($orderObject->getFinalPrice() > 0 && $orderObject->getPaymentType() == \Website\Model\OrderManager::PAYMENT_CREDIT_CARD) {
					$this->gotoUrl(\Website\Tool\Utils::url('gpe-webpay-gate', ['language' => $this->language]));
					return;
				}
				$this->gotoUrl($this->view->docUrl('/' . $this->defaultLanguage . '/kosik/kontaktni-a-platebni-udaje/rekapitulace'));
				return;
			}
		} else {
			//pre-select from cart session
			$form->populate($order->contactInfo);
			$form->country->setValue($order->country);
			$form->{'shippingType' . $order->country}->setValue($order->shippingType);
			$form->{'paymentType' . $order->country . $order->shippingType}->setValue($order->paymentType);
			$form->discountCode->setValue($this->cart->getOrder()->discountCode);
		}

		$user = $this->getCurrentUser();
		$this->view->userAddress = $this->_helper->getUserAddressJson($user);

		$this->view->cartInfoForm = $form;
		$this->view->backlink = $this->currentUrl;
	}

	public function cartSummaryAction()
	{
		$this->enableLayout();

		//clear cart
		$this->cart->clear();

		if ($this->isEditOrAdminMode()) {
			$order = $this->shopModel->mockOrder();
		} else {
			$order = \Website\Model\Order::getById($this->cart->getOrderId());
		}

		if (!$order instanceof \Website\Model\Order) {
			$this->addErrorMsg($this->translate('msg_unknown_error'));
			$this->gotoUrl($this->view->docUrl('/' . $this->defaultLanguage . '/kosik'));
			return;
		}

		$this->view->order = $order;

		if (!\Zend_Auth::getInstance()->hasIdentity() || ($this->isEditOrAdminMode())) {
			$form = new \Website\Form\QuickRegistrationForm();

			if ($this->getRequest()->isPost()) {
				if ($this->handleQuickRegistrationForm($form, $order)) {
					$this->gotoUrl($this->view->docUrl('/' . $this->defaultLanguage . '/kosik/kontaktni-a-platebni-udaje/rekapitulace'));
				} else {
					$form->email->setValue($order->getFEmail());
				}
			} else {
				$form->email->setValue($order->getFEmail());
			}
			$this->view->quickRegistrationForm = $form;
		}


		$this->view->products = $this->cart->getProductObjects();
	}

	public function compareProductsAction()
	{
		$this->enableLayout();

		$comparison = [];
		$parameters = [];
		foreach ($this->cart->getComparedProducts() as $key => $product) {
			$comparison[$key]['parameters'] = [];
			$comparison[$key]['product'] = $product;
			$params = $product->getParameters();
			if (!empty($params)) {
				foreach ($params as $row) {
					if (!empty($row) && count($row) > 1) {
						$comparison[$key]['parameters'][$row[0]] = $row[1];
					}
					if (!in_array($row[0], $parameters)) {
						$parameters[] = $row[0];
					}
				}
			}
		}
		$this->view->comparison = $comparison;
		$this->view->parameters = $parameters;
	}



	/* 					STATIC-ROUTE ROUTED ACTIONS						 */



	public function productAction()
	{
		$this->enableLayout();

		$this->setCanonicalUrl();

		$path = $this->getRequest()->getParam('translatedPath');

		//check if product exists
		//NOTICE have to use list, because the getByXY() getter magic doesn't work on localized fields
		$list = new \Website\Model\Product\Listing();
		$list->setObjectTypes([\Pimcore\Model\Object\AbstractObject::OBJECT_TYPE_OBJECT, \Pimcore\Model\Object\AbstractObject::OBJECT_TYPE_VARIANT]);
		$list->setCondition(\Pimcore\Db::get()->quoteInto('translatedPath = ?', $path));
		$list->setLimit(1);
		$list->load();
		$product = $list->current();
		//could also be category
		if (!$product) {
			$categories = new \Website\Model\Category\Listing();
			$categories->setCondition(\Pimcore\Db::get()->quoteInto('translatedPath = ?', $path));
			$product = $categories->current();
		}
		//and could also be in eshop_old_paths and require a 301 redirect
		if (!$product) {
			$db = \Pimcore\Db::get();
			$row = $db->select()->from('eshop_old_paths')
				->where('translatedPath = ?', $path)
				->where('language = ?', $this->language)
				->order('class ASC') # products are more important (their class ID is lower)
				->order('id DESC') # newer entries are more important
				->query()->fetch();
			if (!empty($row)) {
				if ($row['class'] == 1) {
					$redirectObject = \Website\Model\Product::getById($row['o_id']);
				} else {
					$redirectObject = \Website\Model\Category::getById($row['o_id']);
				}
				if ($redirectObject) {
					$this->gotoUrl($redirectObject->getUrl($this->language), 301);
					exit;
				} else {
					//@TODO if there is an entry in eshop_old_paths, but the products does not exists anymore, suggest a category
				}
			}
		}
		//and finaly, a 404
		if (!$product) {
			throw new \Zend_Controller_Router_Exception($this->translate('system_page_not_found'), 404);
			return;
		}

		//output cached stuff
		if (!$this->isOutputCached($this->cacheKeySuffix)) {
			//add parent categories to breadcrumbs
			$breadcrumbs = [];
			$parent = $product;
			//@TODO add filter object
			while ($parent instanceof \Website\Model\Category || $parent instanceof \Website\Model\Product) {
				$breadcrumbs[] = [
					'label' => ($parent->getType() instanceof \Website\Model\Product) ? $parent->getVariantName() : $parent->getName(),
					'url' => $parent->getUrl($this->language)
				];
				$parent = $parent->getParent();
			}
			//extend breadcrumbs
			foreach (array_reverse($breadcrumbs) as $breadcrumb) {
				$this->addToBreadcrumbs[] = $breadcrumb;
			}
			//explicit language switch
			$this->languageSwitch = $this->_helper->buildLanguageSwitchFromObject($product);
			//explicit seo stuff
			$title = ($product instanceof \Website\Model\Product)
				? $product->getVariantName()
				: $product->getName();
			$this->view->headTitle($product->getName());
			if ($product->getSeoKeywords()) {
				$this->view->headMeta()->setName('keywords', $product->getSeoKeywords());
			}
			if ($product->getSeoDescription()) {
				$this->view->headMeta()->setName('description', $product->getSeoDescription());
			}
		}

		$urlPrefix = $product->getUrl($this->language) . '?page=';
		$search = \Website\Model\ElasticSearch::getInstance($this->currencyModel, $this->language, $urlPrefix);

		if ($product instanceof \Website\Model\Category) {
			//category -> load products and render the category template
			$filterValues = $this->filterForm->getValidValues($this->getRequest()->getParams());
			$pagination = $search->getProducts($product->getTranslatedPath(), $filterValues);
			$pagination->ajax = true;

			$this->view->category = $product;
			$this->view->pagination = $pagination;
			$this->view->elasticSearch = $search;
			//handle ajax calls
			if ($this->isAjax()) {
				$this->ajaxResponseBuilder->addElement('#category-filter-wrapper', 'shop/snippets/category-filter.php')
					->addElement('#category-product-list-wrapper', 'shop/snippets/category-product-list.php')
					->sendResponse();
			}
			$this->_helper->viewRenderer('category');
		} else {
			//add to last viewed products
			$this->cart->addToLastViewed($product->getId());

			$this->view->product = $product;

			$category = $product->getCategory($this->language);
			$this->view->relatedProducts = $search->getRelatedProducts(($category) ? $category->getTranslatedPath($this->language) : null, $product->getId());
			if ($product->getIsMasterProduct()) {
				$this->view->variants = $search->getVariants($product->getId());
			}
		}
	}

	public function invoiceAction()
	{
		$this->checkAuth();
		$pdf = $this->getRequest()->getParam('pdf', false);

		$order = \Website\Model\Order::getById($this->getRequest()->getParam('id'));
		$settings = \Website\Tool\Utils::getEshopSettings();

		if (!$order instanceof \Website\Model\Order || !$order->getUser() instanceof \Pimcore\Model\Object\User || $order->getUser()->getEmail() != \Zend_Auth::getInstance()->getIdentity()->email) {
			throw new \Zend_Controller_Router_Exception($this->translate('system_page_not_found'), 404);
			return;
		}

		$this->view->order = $order;
		$this->view->settings = $settings;
		$this->view->doNotCacheLayout = true;

		if ($pdf) {
			$this->disableLayout();
			$this->disableViewAutoRender();

			$mpdf = new mPDF();
			$template = $this->view->render('shop/invoice-pdf.php');
			$mpdf->WriteHTML($template);
			$mpdf->Output($order->getOrderId() . '.pdf', 'D');
			return;
		} else {
			$this->enableLayout();
		}

		$this->addToBreadcrumbs[] = [
			'label' => sprintf($this->translate('title_order_detail'), $order->getOrderId()),
			'url' => \Website\Tool\Utils::url('order-' . $this->language, ['id' => $order->getId()])
		];
		$this->addToBreadcrumbs[] = [
			'label' => $this->translate('title_invoice_detail'),
			'url' => \Website\Tool\Utils::url('invoice-' . $this->language, ['id' => $order->getId()])
		];
		//explicit language switch
		$this->languageSwitch = $this->_helper->buildLanguageSwitchFromRoute('invoice-', true, ['id' => $order->getId()]);
	}



	/* 				VIEW-LESS ACTIONS (ALLWAYS REDIRECT)					 */



	public function addToCartAction()
	{
		$id = $this->getRequest()->getParam('id', 0);
		$count = $this->getRequest()->getParam('count', 1);

		try {
			$product = \Website\Model\Product::getById($id);
			if (!$product)
				throw new \Exception('product not found');
			$this->cart->addItem($id, $count);
		} catch (\Exception $e) {
			\Pimcore\Log\Simple::log('exceptions', $e->getMessage() . "\n" . $e->getTraceAsString());
			$this->addErrorMsg('msg_product_does_not_exist');
			$this->ajaxResponseBuilder->addFlashMessages()->sendResponse();
		}

		$this->view->cartProduct = $product;
		$refreshProductId = $product->getId();

		//for reloading(disabling) the add-to-cart buttons
		$this->view->product = $product;
		if ($this->getRequest()->getParam('rp')) {
			$search = \Website\Model\ElasticSearch::getInstance($this->currencyModel, $this->language, '');
			$refreshProductId = $product->getParentId();
			$this->view->variants = $search->getVariants($refreshProductId);
		}

		$this->ajaxResponseBuilder->addFlashMessages()
			->addPopup('shop/snippets/add-to-cart-popup.php')
			->addElement('.cart-sum-wrapper', 'shop/snippets/cart-sum.php')
			->addElement('#add-to-cart-' . $refreshProductId, 'shop/snippets/add-to-cart.php')
			->sendResponse();
	}

	public function modifyComparisonAction()
	{
		$productId = $this->getRequest()->getParam('id', 0);
		$remove = $this->getRequest()->getParam('remove', 0);
		$productContextId = $this->getRequest()->getParam('context', 0);

		if ($productId && $productContextId) {
			if ($remove) {
				$result = $this->cart->removeFromComparison((int) $productId);
			} else {
				$result = $this->cart->addToComparison((int) $productId);
				$this->_helper->handleModelResponse($result);
			}
		}

		$this->view->contextProductId = $productContextId;

		$this->ajaxResponseBuilder->addFlashMessages()
			->addPopup('shop/snippets/compare-popup.php')
			->sendResponse();
	}

	public function autocompleteAction()
	{
		$query = $this->getRequest()->getParam('query');

		$search = \Website\Model\ElasticSearch::getInstance($this->currencyModel, $this->language, null);
		$suggestions = $search->suggest($query);

		$this->_helper->json($suggestions);
	}



	/* 							FORM HANDLERS								 */



	private function handleCartForm(\Zend_Form &$form)
	{
		if ($form->isValid($this->getRequest()->getPost())) {
			$data = $form->getValues();
			//counts
			foreach ($data as $key => $value) {
				if (stristr($key, 'count')) {
					$parts = explode('_', $key);
					$id = (isset($parts[1])) ? intval($parts[1]) : 0;
					$this->cart->updateItem($id, $value);
				}
			}

			return true;
		} else {
			$this->addErrorMsg($this->translate('msg_form_invalid'));
		}

		return false;
	}

	private function handleCartInfoForm(\Zend_Form &$form)
	{
		if ($this->getRequest()->getPost('act') == 'discount') {//means the form was sent via the apply discount button
			$data = $this->getRequest()->getPost();
			//discount code
			if (isset($data['discountCode']) && !empty($data['discountCode'])) {
				$this->cart->setDiscountCode($data['discountCode']);
			}
			//persist other stuff to session
			if(isset($data['shippingType' . $data['country']])) {
				$this->cart->setShippingType($data['shippingType' . $data['country']]);
				if (isset($data['paymentType' . $data['country'] . $data['shippingType' . $data['country']]])) {
					$this->cart->setPaymentType($data['paymentType' . $data['country'] . $data['shippingType' . $data['country']]]);
				}
			}
			$this->cart->setContactInfo($data);

			$this->cart->recalculate();

			if (!$this->cart->getOrder()->discountCode) {
				$form->populate($data);
				$form->discountCode->addError($this->translate('msg_wrong_discount_code'));
				return false;
			}

			return true;
		} else {
			if ($form->isValid($this->getRequest()->getPost())) {
				$data = $form->getValues();

				if (!isset($data['shippingType' . $data['country']]) || !isset($data['paymentType' . $data['country'] . $data['shippingType' . $data['country']]])) {
					return false;
				}

				$result1 = $this->cart->setCountry($data['country']);
				$result2 = $this->cart->setShippingType($data['shippingType' . $data['country']]);
				$result3 = $this->cart->setPaymentType($data['paymentType' . $data['country'] . $data['shippingType' . $data['country']]]);

				if (!$result1 || !$result2 || !$result3) {
					return false;
				}

				$this->cart->setContactInfo($data);

				//do a quick registration if needed
				$userManager = new \Website\Model\UserManager($this->translator);
				$user = $userManager->quickRegistration($data, $this->language);
				if (!$user) {
					$this->addErrorMsg($this->translate('msg_unknown_error'));
					return false;
				}

				return true;
			} else {
				$this->addErrorMsg($this->translate('msg_form_invalid'));
			}

			return false;
		}
	}

	protected function handleQuickRegistrationForm(\Zend_Form &$form, \Website\Model\Order $order)
	{
		if ($form->isValid($this->getRequest()->getPost())) {
			$values = $form->getValues();

			//does not agree with terms
			if (!$values['agreed']) {
				$form->getElement('agreed')->addError('msg_agree_with_terms');
			}
			//error sum-up
			$errors = $form->getMessages();
			if (!empty($errors)) {
				$this->addErrorMsg($this->translate('msg_form_invalid'));
				return false;
			}

			$data = get_object_vars($order);

			$userManager = new \Website\Model\UserManager($this->translator);
			$user = $userManager->quickRegistration($data, $this->language, $values['password']);

			if (!$user instanceof \Pimcore\Model\Object\User) {
				$this->addErrorMsg($this->translate('msg_unknown_error'));
				return false;
			}

			//log the user
			$identity = new \stdClass();
			$identity->email = $data['fEmail'];
			\Zend_Auth::getInstance()->getStorage()->write($identity);

			$this->addSuccessMsg(sprintf($this->translate('msg_quick_registration_successfull'), $data['fEmail']));

			return true;
		} else {
			$this->addErrorMsg($this->translate('msg_form_invalid'));
		}

		return false;
	}

}
