<?php

namespace Website\Model;

class FeedManager
{
	protected $language = null;
	/**
	 * @var \Pimcore\Translate\Website
	 */
	protected $translator = null;
	protected $serverUrl = null;
	protected $heurekaShippingCodeMap = [];

	public function __construct($language)
	{
		$this->language = $language;
		if (\Zend_Registry::isRegistered('Zend_Translate')
				&& \Zend_Registry::get('Zend_Translate')->getLocale() == $language) {
			$this->translator = \Zend_Registry::get('Zend_Translate');
		} else {
			$locale = (\Zend_Registry::isRegistered('Zend_Locale')
					&& \Zend_Registry::get('Zend_Locale')->getLanguage() == $language)
				? \Zend_Registry::get('Zend_Locale')
				: new \Zend_Locale($language);
			$this->translator = new \Pimcore\Translate\Website($locale);
		}

		$serverUrlHelper = new \Zend_View_Helper_ServerUrl();
		$this->serverUrl = $serverUrlHelper->serverUrl();

		$this->heurekaShippingCodeMap[OrderManager::SHIPPING_CZECH_POST] = 'CESKA_POSTA';
		$this->heurekaShippingCodeMap[OrderManager::SHIPPING_CZECH_PPL] = 'PPL';
		$this->heurekaShippingCodeMap[OrderManager::SHIPPING_ULOZENKA] = 'HEUREKAPOINT';
	}

	public function getHeurekaXml(Currency $currencyModel, $isForZbozi = false)
	{
		$orderManager = new OrderManager($this->translator, $currencyModel);
		$country = $currencyModel->getCountry();

		$productList = new Product\Listing();
		$productList->setObjectTypes([\Pimcore\Model\Object\Concrete::OBJECT_TYPE_OBJECT]);
		$productList->setUnpublished(true);
		$productList->load();

		$variantList = new Product\Listing();
		$variantList->setObjectTypes([\Pimcore\Model\Object\Concrete::OBJECT_TYPE_VARIANT]);

		$xml = '<?xml version="1.0" encoding="UTF-8"?>';
		$xmlns = ($isForZbozi) ? ' xmlns="http://www.zbozi.cz/ns/offer/1.0"' : '';
		$xml .= sprintf('<SHOP%s>', $xmlns);
		foreach ($productList->getItems(0, 0) as $product) {
			$variantList->setCondition('o_parentId = '.$product->getId().' AND o_published = 1');
			$variantList->load();
			if ($variantList->count()) {
				foreach ($variantList->getItems(0, 0) as $variant) {
					//remove variants in the middle of the variant tree, take only the deepest ones
					$variantVariants = $variant->getChildren([\Pimcore\Model\Object\Concrete::OBJECT_TYPE_VARIANT]);
					if (empty($variantVariants)) {
						$xml .= $this->getHeurekaItem($variant, $country, $orderManager, $isForZbozi, $product->getId());
					} else {
						foreach ($variantVariants as $variantVariant) {
							if ($variantVariant->isPublished()) {
								$xml .= $this->getHeurekaItem($variantVariant, $country, $orderManager, $isForZbozi, $product->getId());
							}
						}
					}
				}
			} elseif ($product->getPublished()) {
				$xml .= $this->getHeurekaItem($product, $country, $orderManager, $isForZbozi);
			}
		}
		$xml .= sprintf('</SHOP>');

		return $xml;
	}

	public function getZboziXml(Currency $currencyModel)
	{
		return $this->getHeurekaXml($currencyModel, true);
	}

	private function getHeurekaItem(Product $product, $country, OrderManager $orderManager, $isForZbozi, $parentId = null)
	{
		$xml = '';

		$xml .= sprintf('<SHOPITEM>');
		if ($parentId) {
			$xml .= sprintf('<ITEMGROUP_ID>%s</ITEMGROUP_ID>', $parentId);
			$vas = $product->getVariantAttributes()->getSize();
			if ($vas) {
				$size = $vas->getSize();
				if ($size instanceof \Pimcore\Model\Object\ProductSize) {
					$xml .= sprintf('<PARAM><PARAM_NAME>Velikost</PARAM_NAME><VAL>%s</VAL></PARAM>', $size->getName($this->language));
				}
			}
			$vac = $product->getVariantAttributes()->getColor();
			if ($vac) {
				$color = $vac->getColor();
				if ($color instanceof \Pimcore\Model\Object\ProductColor) {
					$xml .= sprintf('<PARAM><PARAM_NAME>Barva</PARAM_NAME><VAL>%s</VAL></PARAM>', $color->getName($this->language));
				}
			}
		}
		$xml .= sprintf('<ITEM_ID>%s</ITEM_ID>', $product->getId());
		$xml .= sprintf('<PRODUCT><![CDATA[%s]]></PRODUCT>', $product->getName());
		$xml .= sprintf('<PRODUCTNAME><![CDATA[%s]]></PRODUCTNAME>', $product->getName());
		$xml .= sprintf('<DESCRIPTION><![CDATA[%s]]></DESCRIPTION>', strip_tags(html_entity_decode($product->getDescription())));
		$xml .= sprintf('<URL>%s</URL>', $this->serverUrl.$product->getUrl($this->language));
		$previewImage = $product->getPreviewImage();
		if ($previewImage instanceof \Pimcore\Model\Asset\Image) {
			$xml .= sprintf('<IMGURL>%s</IMGURL>', $this->serverUrl.$previewImage->getRelativeFileSystemPath());
		}
		if (!$isForZbozi) {
			foreach ($product->getGallery() as $image) {
				if ($image instanceof \Pimcore\Model\Asset\Image) {
					if ($previewImage && $previewImage->getId() == $image->getId()) {
						continue;
					}
					$xml .= sprintf(
						'<IMGURL_ALTERNATIVE>%s</IMGURL_ALTERNATIVE>',
						$this->serverUrl.$image->getRelativeFileSystemPath()
					);
				}
			}
		}
		$priceGetter = 'getDiscountedPrice'.strtoupper($country);
		$price = $product->$priceGetter();
		$xml .= sprintf('<PRICE_VAT>%d</PRICE_VAT>', $price);
		if ($product->getInStock()) {
			$xml .= sprintf('<DELIVERY_DATE>%d</DELIVERY_DATE>', 0);
		} elseif ($isForZbozi) {
			$xml .= '<DELIVERY_DATE>-1</DELIVERY_DATE>';
		}
		if (!$isForZbozi) {
			$shippingAndPaymentPrices = $orderManager->getShippingAndPaymentTypePrices($price)[$country];
			foreach ($this->heurekaShippingCodeMap as $shippingCode => $heurekaShippingCode) {
				if (isset($shippingAndPaymentPrices[$shippingCode])) {
					$deliveryPrice = $shippingAndPaymentPrices[$shippingCode]['price'];
					if (isset($shippingAndPaymentPrices[$shippingCode]['paymentPrices'][OrderManager::PAYMENT_COD])) {
						$deliveryPriceCod = $shippingAndPaymentPrices[$shippingCode]['paymentPrices'][OrderManager::PAYMENT_COD]['price'];
						$xml .= sprintf(
							'<DELIVERY><DELIVERY_ID>%s</DELIVERY_ID><DELIVERY_PRICE>%s</DELIVERY_PRICE><DELIVERY_PRICE_COD>%s</DELIVERY_PRICE_COD></DELIVERY>',
								$heurekaShippingCode,
								$deliveryPrice,
								$deliveryPrice + $deliveryPriceCod
						);
					} else {
						$xml .= sprintf(
							'<DELIVERY><DELIVERY_ID>%s</DELIVERY_ID><DELIVERY_PRICE>%s</DELIVERY_PRICE></DELIVERY>',
							$heurekaShippingCode,
							$deliveryPrice
						);
					}
				}
			}

			$category = $product->getCategory();
			if ($category) {
				$categoryName = $category->getName($this->language);
				$xml .= sprintf('<categorytext><![CDATA[%s]]></categorytext>', $categoryName);
			}
		}

		$xml .= sprintf('</SHOPITEM>');

		return $xml;
	}
}
