<?php

/**
 * class that handles communication with the GPE WebPay ecommerce system
 * part of the solution are also the two request and response forms.
 *
 * @see \Website\Form\GpeWebPayRequestForm
 * @see \Website\Form\GpeWebPayResponseForm
 */
namespace Website\Model;

class GpeWebPay
{
	const LOG_TYPE_LENGTH = 30;
	const FROM_ESHOP = 1;
	const FROM_ONE_TIME_SUPPORT = 2;

	protected $language;
	protected $url;
	protected $merchant;
	protected $pass;
	protected $publicKeyPath;
	protected $privateKeyPath;
	protected $muzoPublicKeyPath;
	protected $translator;

	public function __construct($language, $url = null, $publicKeyPath = null, $privateKeyPath = null, $muzoPublicKeyPath = null, $merchant = null, $pass = null)
	{
		$this->language = $language;
		$this->translator = \Zend_Registry::get('Zend_Translate');
		if (!$publicKeyPath) {
			$this->publicKeyPath = PIMCORE_WEBSITE_PATH.'/data/test_cert.pem';
		} else {
			$this->publicKeyPath = $publicKeyPath;
		}
		if (!$privateKeyPath) {
			$this->privateKeyPath = PIMCORE_WEBSITE_PATH.'/data/test_key.pem';
		} else {
			$this->privateKeyPath = $privateKeyPath;
		}
		if (!$muzoPublicKeyPath) {
			$this->muzoPublicKeyPath = PIMCORE_WEBSITE_PATH.'/data/muzo.signing_test.pem';
		} else {
			$this->muzoPublicKeyPath = $muzoPublicKeyPath;
		}
		if (!$merchant) {
			$this->merchant = 123456789;
		} else {
			$this->merchant = $merchant;
		}
		if (!$pass) {
			$this->pass = 'changeit';
		} else {
			$this->pass = $pass;
		}
		if (!$url) {
			$this->url = 'https://test.3dsecure.gpwebpay.com/csob/order.do';
		} else {
			$this->url = $url;
		}
	}

	public function getUrl()
	{
		return $this->url;
	}

	public function genRequestForm(Order $order)
	{
		$orderId = $order->getOrderId();
		$totalPrice = $order->getFinalPrice();
		$currency = $order->getCurrency();

		$form = new \Website\Form\GpeWebPayRequestForm();
		$form->setAction($this->url);

		$form->MERCHANTNUMBER->setValue($this->merchant);
		$form->OPERATION->setValue('CREATE_ORDER');
		$form->ORDERNUMBER->setValue($orderId.substr(time(), -5, 5)); //this should be unique for every request (gpe transaction) to allow payment re-try
		$form->AMOUNT->setValue(round($totalPrice * 100));
		//@TODO extend list
		switch ($currency) {
			case 'cs_CZ':
				$currencyCode = 203;
				break;
			case 'sk_SK':
			case 'de_DE':
			case 'de_AT':
				$currencyCode = 978;
				break;
			case 'en_US':
				$currencyCode = 840;
				break;
			default:
				$currencyCode = 203;
				break;
		}
		$form->CURRENCY->setValue($currencyCode); //ISO ISO 4217 num
		$form->DEPOSITFLAG->setValue(1);
		$form->MERORDERNUM->setValue($orderId);
		$form->URL->setValue(\Zend_Controller_Front::getInstance()->getRequest()->getScheme().'://'.\Zend_Controller_Front::getInstance()->getRequest()->getHttpHost().\Zend_Controller_Front::getInstance()->getBaseUrl().\Website\Tool\Utils::url('gpe-webpay', ['language' => $this->language]));
		$form->DESCRIPTION->setValue($this->translator->translate('msg_gpe_webpay_order_info'));
		$form->MD->setValue(time());

		//calculate the signature
		$CSignature = new \Website\Model\CSignature($this->privateKeyPath, $this->pass, $this->publicKeyPath);
		$data = [];
		foreach ($form->getElements() as $element) {
			if ($element instanceof \Zend_Form_Element_Hidden) {
				$data[$element->getName()] = $element->getValue();
			}
		}
		unset($data['DIGEST']);

		$signature = $CSignature->sign(implode('|', $data));

		$form->DIGEST->setValue($signature);

		//log
		$data['DIGEST'] = $signature;
		$this->log('NEW FORM', $data);

		return $form;
	}

	public function handleResponse(\Zend_Form $form, $params, Order $orderObject)
	{
		//invalid params
		if (!$form->isValid($params)) {
			//log
			$data = [];
			foreach ($params as $key => $value) {
				if (is_string($value)) {
					$data[$key] = $value;
				}
			}
			$this->log('CORRUPTED RESPONSE', $data);

			return ['status' => -1, 'key' => 'msg_gpe_webpay_response_corrupted_data'];
		}
		$data = $form->getValues();

		//assemble signature string
		$signature = $data['DIGEST'];
		unset($data['DIGEST']);
		$dataString = implode('|', $data);
		$data['DIGEST'] = $signature;

		//private and pass are redundand here
		$CSignature = new \Website\Model\CSignature($this->privateKeyPath, $this->pass, $this->muzoPublicKeyPath);

		//invalid signature
		if (!$CSignature->verify($dataString, $signature)) {
			$this->log('INVALID SIGNATURE', $data);

			return ['status' => -1, 'key' => 'msg_gpe_webpay_response_corrupted_data'];
		}

		//we do only the CREATE_ORDER
		if ($data['OPERATION'] != 'CREATE_ORDER') {
			$this->log('UNSUPPORTED OPERATION', $data);

			return ['status' => -1, 'key' => 'msg_gpe_webpay_response_corrupted_data'];
		}

		//get order
		$order = Order::getByOrderId($data['MERORDERNUM'])->current();
		if (!$order instanceof Order) {
			$this->log('NO ORDER MATCHED', $data);

			return ['status' => -1, 'key' => 'msg_gpe_webpay_response_order_does_not_exist'];
		}
		if ($order->getOrderId() != $orderObject->getOrderId()) {
			$this->log('ORDER MISMATCH', $data + ['orderId' => $order->getOrderId(), 'orderObjectId' => $orderObject->getOrderId()]);

			return ['status' => -1, 'key' => 'msg_gpe_webpay_response_order_does_not_exist'];
		}

		//update order info text
		$PRCode = isset($data['PRCODE']) ? (int) $data['PRCODE'] : 9999;
		$SRCode = isset($data['SRCODE']) ? (int) $data['SRCODE'] : 9999;
		$oldTimestamp = $order->getGpeWebPayTimestamp();
		$responseTimestamp = (int) $data['MD'];
		$saved = true;
		if ($oldTimestamp != $responseTimestamp) {
			$order->setGpeWebPayTimestamp($responseTimestamp);
			$order->setGpeWebPayNote(sprintf('GPE Webpay: PRCODE=%s, SRCODE=%s, TEXT=%s (%s)', $PRCode, $SRCode, $this->PRCodeText($PRCode), $this->SRCodeText($SRCode)));
			if ($PRCode == 0) {
				$order->setPaid(true);
			}
			try {
				$order->save();
			} catch (\Exception $e) {
				$saved = false;
				$this->log('ORDER NOT SAVED', $data);
			}
		}

		//inform user about result status and log it
		if ($PRCode != 0) {
			$this->log('WEBPAY ERROR', $data);
			if ($PRCode >= 11 && $PRCode <= 35) {
				return ['status' => -1, 'key' => 'msg_gpe_webpay_response_refused_payment_reason', 'args' => [$this->PRCodeText($PRCode), $this->SRCodeText($SRCode)]];
			}

			return ['status' => -1, 'key' => 'msg_gpe_webpay_response_refused_payment'];
		} else {
			$this->log('WEBPAY SUCCESS', $data);
			if ($saved) {
				return ['status' => 1, 'key' => 'msg_gpe_webpay_response_paid_saved'];
			} else {
				return ['status' => 0, 'key' => 'msg_gpe_webpay_response_paid_not_saved'];
			}
		}
	}

	public function log($type, $data)
	{
		$parts = [];
		foreach ($data as $key => $value) {
			$parts[] = $key.':'.$value;
		}
		\Pimcore\Log\Simple::log('webpay', str_pad($type, self::LOG_TYPE_LENGTH, ' ', STR_PAD_LEFT).' - '.implode('|', $parts));
	}

	public function PRCodeText($code)
	{
		$codes = [
			0 => 'OK',
			1 => 'Pole příliš dlouhé',
			2 => 'Pole příliš krátké',
			3 => 'Chybný obsah pole',
			4 => 'Pole je prázdné',
			5 => 'Chybí povinné pole',
			11 => 'Neznámý obchodník',
			14 => 'Duplikátní číslo objednávky',
			15 => 'Objekt nenalezen',
			17 => 'Částka k úhradě překročila autorizovanou částku',
			18 => 'Součet kreditovaných částek překročil uhrazenou částku',
			20 => 'Objekt není ve stavu odpovídajícím této operaci',
			25 => 'Uživatel není oprávněn k provedení operace',
			26 => 'Technický problém při spojení s autorizačním centrem',
			27 => 'Chybný typ objednávky',
			28 => 'Zamítnuto v 3D',
			30 => 'Zamítnuto v autorizačním centru',
			31 => 'Chybný podpis',
			35 => 'Expirovaná session',
			1000 => 'Technický problém',
		];

		if (array_key_exists($code, $codes)) {
			return $codes[$code];
		} else {
			return 'Neznámá chyba';
		}
	}

	public function SRCodeText($code)
	{
		$codes = [
			0 => '-',
			1 => 'ORDERNUMBER',
			2 => 'MERCHANTNUMBER',
			6 => 'AMOUNT',
			7 => 'CURRENCY',
			8 => 'DEPOSITFLAG',
			10 => 'MERORDERNUM',
			11 => 'CREDITNUMBER',
			12 => 'OPERATION',
			18 => 'BATCH',
			22 => 'ORDER',
			24 => 'URL',
			25 => 'MD',
			26 => 'DESC',
			34 => 'DIGEST',
			3000 => 'Neověřeno v 3D. Vydavatel karty není zapojen do 3D nebo karta nebyla aktivována.',
			3001 => 'Držitel karty ověřen.',
			3002 => 'Neověřeno v 3D. Vydavatel karty nebo karta není zapojena do 3D.',
			3004 => 'Neověřeno v 3D. Vydavatel karty není zapojen do 3D nebo karta nebyla aktivována.',
			3005 => 'Zamítnuto v 3D.Technický problém při ověření držitele karty.',
			3006 => 'Zamítnuto v 3D. Technický problém při ověření držitele karty.',
			3007 => 'Zamítnuto v 3D. Technický problém v systému zúčtující banky. Kontaktujte obchodníka.',
			3008 => 'Zamítnuto v 3D. Použit nepodporovaný karetní produkt.',
			1001 => 'Zamitnuto v autorizacnim centru, karta blokovana.',
			1002 => 'Zamitnuto v autorizacnim centru, autorizace zamítnuta.',
			1003 => 'Zamitnuto v autorizacnim centru, problem karty.',
			1004 => 'Zamitnuto v autorizacnim centru, technicky problem.',
			1005 => 'Zamitnuto v autorizacnim centru, problem uctu.',
		];
		if (array_key_exists($code, $codes)) {
			return $codes[$code];
		} else {
			return 'Neznámá chyba';
		}
	}
}
