<?php

/**
 * simple class to help assemble the where and order clauses for Pimcore lists
 * 
 * @author Martin Kuric <martin.kuric@portadesign.cz>
 */

namespace Website\Model;

class Filter
{

	protected $conditions = [];
	protected $orderings = [];
	protected $orderConstants = [1 => 'ASC', 2 => 'DESC'];

	/**
	 * adds a formula to order clause
	 * 
	 * @param srring $key column name
	 * @param int $sort 1 for ascending order, 2 for descending order
	 * @return \Website\Model\Filter
	 */
	public function order($key, $sort = 1)
	{
		if ($sort != 1 && $sort != 2)
			$sort = 1;

		$this->orderings[$key] = $this->orderConstants[$sort];

		return $this;
	}

	/**
	 * adds a formula to where clause, formulas are logically connected with AND operator
	 * 
	 * @param type $formula the formula string, one ? placeholder is allowed for the value
	 * @param type $value replaces the ? placeholder in formula, is quoted
	 * @return \Website\Model\Filter
	 */
	public function where($formula, $value = null)
	{
		$this->conditions[] = [$formula => $value];

		return $this;
	}

	/**
	 * order params should start with "order" string
	 * 
	 * @param \Zend_Form $form should be in state after isValid() so it has prefilled values...
	 * @param array $map assoc array, allows to set element name mapping (if it doesn't match the DB column name) or to explicitly disable element (set value to false)
	 * @return \Website\Model\Filter
	 */
	public function setFromFormValues(\Zend_Form $form, $map = [])
	{
		$values = $form->getValues();

		$elements = $form->getElements();

		foreach ($elements as $element) {
			$key = $element->getName();
			//filter out unwanted elements
			if (!$element instanceof \Zend_Form_Element_Submit
					&& !$element instanceof \Zend_Form_Element_Button
					&& !empty($values[$key])
					&& (!isset($map[$key]) || $map[$key] !== false)) {
				//try to get mapped key
				$mappedKey = (isset($map[$key])) ? $map[$key] : $key;
				//order param
				if (substr($key, 0, 5) == 'order') {
					$this->order($mappedKey, $values[$key]);
					//where param
				} else {
					$this->where(sprintf('%s = ?', $mappedKey), $values[$key]);
				}
			}
		}

		return $this;
	}

	/**
	 * 
	 * @return string orderKey for Pimcore list
	 */
	public function getOrderKey()
	{
		return array_keys($this->orderings);
	}

	/**
	 * 
	 * @return string order for Pimcore list
	 */
	public function getOrder()
	{
		return array_values($this->orderings);
	}

	/**
	 * 
	 * @return string condition for Pimcore list
	 */
	public function getCondition()
	{
		$conditions = [];

		foreach ($this->conditions as $stc) {
			$clause = key($stc);
			$value = current($stc);

			if ($value === null) {
				$conditions[] = $clause;
			} else {
				$conditions[] = \Pimcore\Db::get()->quoteInto($clause, $value);
			}
		}

		return implode(' AND ', $conditions);
	}

}
