<?php

/**
 * simple class to help with elastic searching
 * 
 * @author Martin Kuric <martin.kuric@portadesign.cz>
 */

namespace Website\Model;

class ElasticSearch
{

	/**
	 * @var \Elasticsearch\Client
	 */
	protected $elasticClient = null;
	/**
	 * @var \Website\Model\Currency
	 */
	protected $currencyModel = null;
	protected $index = null;
	protected $language = 'cs';
	protected $urlPrefix = null;
	protected $translator = null;
	protected $allowedFilterParams = array('colors', 'sizes', 'brands', 'new', 'prices', 'sale');

	public function __construct(\Website\Model\Currency $currencyModel, $language, $urlPrefix = null)
	{
		$this->currencyModel = $currencyModel;
		$this->language = $language;
		$this->elasticClient = \Website\Tool\ElasticSearch::getClient();
		$this->index = \Website\Tool\ElasticSearch::getMainIndexName(false);
		$this->urlPrefix = $urlPrefix;
		$this->translator = \Zend_Registry::get('Zend_Translate');

		if (!$this->elasticClient->indices()->existsAlias(array('name' => $this->index))) {
			throw new \Exception('Elastic search main index not found.');
		}
	}

	public function getProducts($path, $filter = array())
	{
		$country = $this->currencyModel->getCountry();
		$showVariantsInList = (\Website\Tool\Utils::getEshopSettings()->getShowVariantsInLists());


		/*	PREPARE QUERY BASE	*/


		$filteredQuery = $this->queryBase(\Website\Tool\ElasticSearch::PRODUCT_TYPE_KEY . '_' . $this->language, true, $showVariantsInList);
		//category path restriction
		if (!empty($path)) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['searchPaths'] = $path;
		}
		//fulltext query param
		$paramQuery = $this->translator->translate('param_search_query');
		if (isset($filter[$paramQuery]) && !empty($filter[$paramQuery])) {
			$query = array();
			$query['multi_match']['query'] = $filter[$paramQuery];
			$query['multi_match']['fields'] = array('name^3', 'description');
			$filteredQuery['body']['query']['filtered']['query'] = $query;
		} else {
			$filteredQuery['body']['query']['filtered']['query']['match_all'] = array();
		}


		/*	PREPARE FILTER PARTS	*/


		$filterParts = array();
		//brands
		$paramBrand = $this->translator->translate('param_brand');
		if (isset($filter[$paramBrand]) && !empty($filter[$paramBrand])) {
			$filterParts['brands'] = array('terms' => array('brand' => $filter[$paramBrand]));
		}
		//colors
		$paramColor = $this->translator->translate('param_color');
		if (isset($filter[$paramColor]) && !empty($filter[$paramColor])) {
			if ($showVariantsInList) {
				$filterParts['colors'] = array('terms' => array('color' => $filter[$paramColor]));
			} else {
				$filterParts['colors'] = array('terms' => array('colors' => $filter[$paramColor]));
			}
		}
		//sizes
		$paramSize = $this->translator->translate('param_size');
		if (isset($filter[$paramSize]) && !empty($filter[$paramSize])) {
			if ($showVariantsInList) {
				$filterParts['sizes'] = array('terms' => array('sizes' => $filter[$paramSize]));
			} else {
				//@TODO complete this after multi-level variant attributes propagation finished
			}
		}
		//price ranges
		$paramPrice = $this->translator->translate('param_price');
		if (isset($filter[$paramPrice]) && !empty($filter[$paramPrice])) {
			$priceFilter = array();
			$priceBuckets = \Website\Tool\Utils::getFilterPriceBuckets($country, true);
			foreach ($filter[$paramPrice] as $key) if (isset($priceBuckets[$key])) {
				$priceFilter['or'][]['range']['discountedPrice'.$country] = $priceBuckets[$key];
			}
			if (!empty($priceFilter['or'])) {
				$filterParts['prices'] = $priceFilter;
			}
		}
		//new
		$paramNew = $this->translator->translate('param_new');
		if (isset($filter[$paramNew]) && !empty($filter[$paramNew])) {
			$filterParts['new'] = array('term' => array('isNew' => true));
		}
		//on sale
		$paramSale = $this->translator->translate('param_sale');
		if (isset($filter[$paramSale]) && !empty($filter[$paramSale])) {
			$filterParts['sale'] = array('term' => array('isOnSale' => true));
		}

		/* AGGREGATIONS */

		$aggregations = array();
		list($aggregations['priceMin'], $aggregations['priceMax']) = $this->priceRangeAggregation($filteredQuery);
		$aggregations[$paramPrice] = $this->priceBucketsAggregation($filteredQuery, $filterParts, $country);
		$aggregations[$paramColor] = $this->colorsAggregation($filteredQuery, $filterParts, $showVariantsInList);
		$aggregations[$paramSize] = $this->sizesAggregation($filteredQuery, $filterParts, $showVariantsInList);
		$aggregations[$paramBrand] = $this->brandsAggregation($filteredQuery, $filterParts);
		$aggregations += $this->singleFieldAggregations($filteredQuery, $filterParts);


		/*	PAGINATION, SORT, FINAL QUERY	*/


		//add filter parts to base query
		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'] = array_merge($filteredQuery['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		//get total count
		$countQuery = array('index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => array('query' => $filteredQuery['body']['query']));
		$countResult = $this->elasticClient->count($countQuery);
		$totalCount = $countResult['count'];
		//switch to folded fulltext analyzer if no results
		if (isset($filter[$paramQuery]) && !empty($filter[$paramQuery]) && !$totalCount) {
			$filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'] = array('name.folded^3', 'description.folded');
			$countQuery = array('index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => array('query' => $filteredQuery['body']['query']));
			$countResult = $this->elasticClient->count($countQuery);
			$totalCount = $countResult['count'];
		}
		//pagination
		$page = isset($filter[$this->translator->translate('param_page')]) ? intval($filter[$this->translator->translate('param_page')]) : 1;
		$limit = isset($filter[$this->translator->translate('param_limit')]) ? intval($filter[$this->translator->translate('param_limit')]) : \Website\Tool\Utils::getEshopSettings()->getProductPageLimit();
		if ((int) $page > ceil($totalCount / $limit))
			$page = ceil($totalCount / $limit);
		if ($page < 1) $page = 1;
		$filteredQuery['body']['from'] = ($page - 1) * $limit;
		$filteredQuery['body']['size'] = $limit;
		//only sort when no searchQuery is set and some results matched...
		if ($totalCount && (!isset($filter[$paramQuery]) || empty($filter[$paramQuery]))) { # 
			$filteredQuery['body']['sort'][]['ordering'] = 'asc';
		}
		//query
		$resultSet = $this->elasticClient->search($filteredQuery);
		$resultSet['aggregations'] = $aggregations;
		return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
	}

	public function suggest($query, $size = 10)
	{
		$suggestQuery = array(
			'index' => $this->index,
			'body' => array(
				'suggestions' => array(
					'text' => $query,
					'completion' => array('field' => 'name_suggest', 'size' => $size),
				)
			)
		);
		//$this->elasticClient
		$suggestions = array();
		$resultSet = $this->elasticClient->suggest($suggestQuery);
		foreach ($resultSet['suggestions'][0]['options'] as $result) {
			$suggestions[] = $result['text'];
		}
		return $suggestions;
	}

	public function getMostSoldProducts($limit = 3)
	{
		$filteredQuery = $this->queryBase(\Website\Tool\ElasticSearch::PRODUCT_TYPE_KEY . '_' . $this->language);

		$filteredQuery['body']['query']['filtered']['query']['match_all'] = array();

		$filteredQuery['body']['from'] = 0;
		$filteredQuery['body']['size'] = $limit;
		$filteredQuery['body']['sort'][]['soldCount'] = 'desc';

		$resultSet = $this->elasticClient->search($filteredQuery);

		return $this->constructSimplePagination($resultSet, $limit, 1, 0);
	}

	//@TODO multi level variants fix
	public function getRelatedProducts($path, $id, $limit = 3)
	{
		$functionScore = $this->queryBase(\Website\Tool\ElasticSearch::PRODUCT_TYPE_KEY . '_' . $this->language);

		$functionScore['body']['query']['filtered']['query']['match_all'] = array();
		$functionScore['body']['query']['function_score']['query'] = $functionScore['body']['query'];
		$functionScore['body']['query']['function_score']['functions'][0]['random_score'] = new \stdClass();
		unset($functionScore['body']['query']['filtered']);

		$functionScore['body']['query']['function_score']['query']['filtered']['filter']['bool']['must_not'][]['term']['id'] = $id;
		if ($path) {
			$functionScore['body']['query']['function_score']['query']['filtered']['filter']['bool']['must'][]['term']['searchPaths'] = $path;
		}

		$functionScore['body']['from'] = 0;
		$functionScore['body']['size'] = $limit;

		$resultSet = $this->elasticClient->search($functionScore);

		return $this->constructSimplePagination($resultSet, $limit, 1, 0);
	}

	public function getLastViewedProducts($idList)
	{
		if (empty($idList)) {
			return array();
		}

		$filteredQuery = $this->queryBase(\Website\Tool\ElasticSearch::PRODUCT_TYPE_KEY . '_' . $this->language);

		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['ids'] = array(
			'type' => $filteredQuery['type'],
			'values' => $idList
		);

		$resultSet = $this->elasticClient->search($filteredQuery);

		return $resultSet['hits']['hits'];
		//return $this->constructSimplePagination($resultSet['hits']['hits'], 999, 1, 0);
	}

	public function getVariants($productId)
	{
		$filteredQuery['index'] = $this->index;
		$filteredQuery['type'] = \Website\Tool\ElasticSearch::PRODUCT_TYPE_KEY . '_' . $this->language;

		$filter = array();
		$filter['bool']['must'][0]['term']['published'] = true;
		$filter['bool']['must'][1]['term']['isVariant'] = true;
		$filter['bool']['must'][2]['term']['parentId'] = (int) $productId;

		$filteredQuery['body']['query']['filtered']['filter'] = $filter;
		$filteredQuery['body']['from'] = 0;
		$filteredQuery['body']['size'] = 100;

		$resultSet = $this->elasticClient->search($filteredQuery);

		//@TODO multi-level support
		$hits = $resultSet['hits']['hits'];
		foreach($hits as $key => $hit) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][2]['term']['parentId'] = $hit['_source']['id'];
			$resultSet = $this->elasticClient->search($filteredQuery);
			if (!empty($resultSet['hits']['hits'])) {
				unset($hits[$key]);
				foreach ($resultSet['hits']['hits'] as $tmpHit) {
					$hits[] = $tmpHit;
				}

			}
		}

		return $hits;
		//return $this->constructSimplePagination($resultSet['hits']['hits'], 999, 1, 0);
	}

	public function fulltextSearch($query, $page = 1, $limit = 10, $onlyPublished = true)
	{
		if (empty($query)) {
			return $this->constructSimplePagination(array(), $limit, $page, 0);
		}

		$types = array(
			\Website\Tool\ElasticSearch::DOCUMENT_TYPE_KEY . '_' . $this->language,
			\Website\Tool\ElasticSearch::OBJECT_TYPE_KEY . '_' . $this->language
		);
		$filteredQuery['index'] = $this->index;
		$filteredQuery['type'] = $types;

		$filter = array();
		if ($onlyPublished) {
			$filter['bool']['must'][]['term']['published'] = true;
		}
		$filteredQuery['body']['query']['filtered']['filter'] = $filter;
		if (!empty($query)) {
			$multiMatchQuery = array();
			$multiMatchQuery['multi_match']['query'] = $query;
			$multiMatchQuery['multi_match']['fields'] = array('title^4', 'keywords^3', 'description^2', 'content');
			$filteredQuery['body']['query']['filtered']['query'] = $multiMatchQuery;
		} else {
			$filteredQuery['body']['query']['filtered']['query']['match_all'] = array();
		}

		//count
		$countQuery = array('index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => array('query' => $filteredQuery['body']['query']));
		$countResult = $this->elasticClient->count($countQuery);
		$totalCount = $countResult['count'];
		//switch to folded fulltext analyzer if no results
		if (!$totalCount) {
			$filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'] = array('title.folded^4', 'keywords.folded^3', 'description.folded^2', 'content.folded');
			$countQuery = array('index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => array('query' => $filteredQuery['body']['query']));
			$countResult = $this->elasticClient->count($countQuery);
			$totalCount = $countResult['count'];
		}
		//pagination
		if ((int) $page > ceil($totalCount / $limit))
			$page = ceil($totalCount / $limit);
		$page = (int) max(array(1, $page));
		$filteredQuery['body']['from'] = ($page - 1) * $limit;
		$filteredQuery['body']['size'] = $limit;

		$resultSet = $this->elasticClient->search($filteredQuery);
		return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
	}

	/**
	 * 
	 * @param type $published
	 * @param type $isMaserProduct
	 * @return \Elastica\Query\Bool
	 */
	private function queryBase($type, $published = true, $showVariantsInList = false)
	{
		$query['index'] = $this->index;
		$query['type'] = $type;

		$filter = array();
		$filter['bool']['must'][]['term']['published'] = $published;
		if ($showVariantsInList) {
			$tmpCount = count($filter['bool']['must']);
			//is variant and is not hidden (like size)
			$filter['bool']['must'][$tmpCount]['or'][0]['and'][]['term']['isVariant'] = true;
			$filter['bool']['must'][$tmpCount]['or'][0]['and'][]['term']['size'] = 0;
			//is master product and has no visible variants (like color)
			$filter['bool']['must'][$tmpCount]['or'][1]['and'][]['term']['isMasterProduct'] = true;
			$filter['bool']['must'][$tmpCount]['or'][1]['and'][]['missing']['field'] = 'colors';
			//is no master product nor variant
			$filter['bool']['must'][$tmpCount]['or'][2]['and'][]['term']['isVariant'] = false;
			$filter['bool']['must'][$tmpCount]['or'][2]['and'][]['term']['isMasterProduct'] = false;
		} else {
			$filter['bool']['must'][]['term']['isVariant'] = false;
		}

		$query['body']['query']['filtered']['filter'] = $filter;

		return $query;
	}

	private function priceRangeAggregation($query)
	{
		$aggQuery = array();
		$aggQuery['min_price']['min']['field'] = 'discountedPrice';
		$aggQuery['max_price']['max']['field'] = 'discountedPrice';
		$query['body']['aggs'] = $aggQuery;
		$result = $this->elasticClient->search($query);

		return array($result['aggregations']['min_price']['value'], $result['aggregations']['max_price']['value']);

	}

	private function priceBucketsAggregation($query, $filterParts, $country)
	{
		unset($filterParts['prices']);

		$priceBuckets = \Website\Tool\Utils::getFilterPriceBuckets($country);
		$aggQuery = array();
		$aggQuery['price']['range']['field'] = 'discountedPrice'.$country;
		$aggQuery['price']['range']['ranges'] = $priceBuckets;

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$prices = array();
		foreach ($result['aggregations']['price']['buckets'] as $key => $bucket) {
			$prices[$key] = $bucket['doc_count'];
		}

		return $prices;
	}

	private function colorsAggregation($query, $filterParts, $showVariantsInList)
	{
		unset($filterParts['colors']);

		//colors
		$aggQuery = array();
		$aggQuery['color']['terms']['size'] = 100;
		if ($showVariantsInList) {
			$aggQuery['color']['terms']['field'] = "color";
		} else {
			$aggQuery['color']['terms']['field'] = "colors";
		}

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$colors = array();
		foreach ($result['aggregations']['color']['buckets'] as $bucket) if ($bucket['key']) {
			$colors[$bucket['key']] = $bucket['doc_count'];
		}

		return $colors;
	}

	private function sizesAggregation($query, $filterParts, $showVariantsInList)
	{
		$tmp = isset($filterParts['sizes']) ? $filterParts['sizes'] : null;
		unset($filterParts['sizes']);

		//sizes
		$aggQuery = array();
		if ($showVariantsInList) {
			$aggQuery['size']['terms']['size'] = 100;
			$aggQuery['size']['terms']['field'] = 'sizes';
		} else {
			//@TODO complete this after multi-level variant attributes propagation finished
		}

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		if ($tmp) {
			$query['body']['query']['filtered']['filter']['bool']['must_not'][] = $tmp;
		}
		$result = $this->elasticClient->search($query);

		$sizes = array();
		if (isset($result['aggregations']['size'])) {
			foreach ($result['aggregations']['size']['buckets'] as $bucket) if ($bucket['key']) {
				$sizes[$bucket['key']] = $bucket['doc_count'];
			}
		}
		return $sizes;
	}

	private function brandsAggregation($query, $filterParts)
	{
		unset($filterParts['brands']);
		//brands
		$aggQuery = array();
		$aggQuery['brand']['terms']['size'] = 100;
		$aggQuery['brand']['terms']['field'] = 'brand';

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		$brands = array();
		foreach ($result['aggregations']['brand']['buckets'] as $bucket) {
			$brands[$bucket['key']] = $bucket['doc_count'];
		}

		return $brands;
	}

	private function singleFieldAggregations($query, $filterParts)
	{
		$aggQuery = array();
		//news
		$aggQuery['new']['filter']['term']['isNew'] = true;
		//on sale
		$aggQuery['sale']['filter']['term']['isOnSale'] = true;

		$query['body']['aggs'] = $aggQuery;
		$query['body']['query']['filtered']['filter']['bool']['must'] = array_merge($query['body']['query']['filtered']['filter']['bool']['must'], array_values($filterParts));
		$result = $this->elasticClient->search($query);

		return array(
			$this->translator->translate('param_new') => $result['aggregations']['new']['doc_count'],
			$this->translator->translate('param_sale') => $result['aggregations']['sale']['doc_count']
		);
	}

	private function constructSimplePagination($resultSet, $limit, $page, $totalCount)
	{
		$pagination = new \stdClass();
		$pagination->current = $page;
		$pagination->pageCount = (int) ceil($totalCount / $limit);
		if ($page > 1) {
			$pagination->previous = $page - 1;
		}
		if ($page < $pagination->pageCount) {
			$pagination->next = $page + 1;
		}
		$pagination->totalCount = $totalCount;
		$pagination->first = 1;
		$pagination->last = $pagination->pageCount;
		$pagination->items = empty($resultSet) ? array() : $resultSet['hits']['hits'];
		$pagination->aggregations = isset($resultSet['aggregations']) ? $resultSet['aggregations'] : array();
		$pagination->urlprefix = $this->urlPrefix;

		return $pagination;
	}

}
