#!/bin/bash
#NOTICE: watch for line endings if you are using windows or mac, use LF only
#NOTICE: this script uses --login-path=deployment for mysql access
#NOTICE: this script uses Git for file sync and expects credentials set in ./git/config

#load remote conf
source ./website/lib/Website/Tool/Deployment/remote.conf
if [ -z "${HOSTNAME}" ]; then
	echo 'Hostname not set - check your remote.conf'
	exit;
fi
if [ -z "${HOSTUSERNAME}" ]; then
	echo 'Host username not set - check your remote.conf'
	exit;
fi
if [ -z "${HOSTROOT}" ]; then
	echo 'Host root not set - check your remote.conf'
	exit;
fi
if [ -z "${WEBNAME}" ]; then
	echo 'Web name not set - check your remote.conf'
	exit;
fi

WEBROOT="$HOSTROOT/$WEBNAME/html"


###Check remote mysql access
MYSQL_ERR=$(ssh $HOSTUSERNAME@$HOSTNAME "mysql --login-path=deployment -e \"GRANT USAGE ON *.* TO 'deploymentuser'@'localhost'; DROP USER 'deploymentuser'@'localhost';\"" 2>&1 >/dev/null);
if [ ! -z "${MYSQL_ERR}" ]; then
	echo -e "\nRemote mysql access denied, exiting...\n"
	exit;
fi
###Check local mysql access
MYSQL_ERR=$(mysql --login-path=deployment -e "GRANT USAGE ON *.* TO 'deploymentuser'@'localhost'; DROP USER 'deploymentuser'@'localhost';" 2>&1 >/dev/null);
if [ ! -z "${MYSQL_ERR}" ]; then
	echo -e "\nLocal mysql access denied, exiting...\n"
	exit;
fi


###Check git
echo "Checking local Git remote..."
GIT_ERR=$(git ls-remote 2>&1 >/dev/null);
if [[ $GIT_ERR == *fatal:* ]]; then
	echo $GIT_ERR;
	exit;
fi
echo "It's OK"
echo "Checking remote Git remote..."
GIT_ERR=$(ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && git ls-remote" 2>&1 >/dev/null);
if [[ $GIT_ERR == *fatal:* ]]; then
	echo $GIT_ERR;
	exit;
fi
echo "It's OK"


###sync source with git
echo -e "\n\nPushing server changes into Git..."
ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && git add -A"
ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && git commit -u -m \"increment\""
ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && git push -u origin --all"
ssh $HOSTUSERNAME@$HOSTNAME "cd $WEBROOT && git push -u origin --tags"
echo "Push complete"
echo -e "\n\nPulling Git changes..."
git pull;
echo "Pull complete"

###create temp dir
mkdir ./deployment_tmp 2>/dev/null


###extract configuration stuff
#copy system.xml and cache.xml if they do not exist (in case of new installations after cloning the git repo)
echo -e "\n\nExtracting configuration..."
if [[ ! -f ./website/var/config/system.xml ]]; then
	scp $HOSTUSERNAME@$HOSTNAME:$WEBROOT/website/var/config/system.xml ./website/var/config/system.xml
fi
if [[ ! -f ./website/var/config/cache.xml ]]; then
	scp $HOSTUSERNAME@$HOSTNAME:$WEBROOT/website/var/config/cache.xml ./website/var/config/cache.xml
fi
#extract local DB configuration 
MYSQLUSER=$(cat ./website/var/config/system.xml | tr -d '[\n\t ]' | sed -e "s/.*host><username>\(.*\)<\/username><password>.*<\/password><dbname>.*/\1/")
MYSQLPASSWORD=$(cat ./website/var/config/system.xml | tr -d '[\n\t ]' | sed -e "s/.*<\/username><password>\(.*\)<\/password><dbname>.*/\1/")
DBNAME=$(cat ./website/var/config/system.xml | tr -d '[\n\t ]' | sed -e "s/.*<dbname>\(.*\)<\/dbname>.*/\1/")
#extract remote DB configuration
scp $HOSTUSERNAME@$HOSTNAME:$WEBROOT/website/var/config/system.xml ./deployment_tmp/system.xml
REMOTEMYSQLUSER=$(cat ./deployment_tmp/system.xml | tr -d '[\n\t ]' | sed -e "s/.*host><username>\(.*\)<\/username><password>.*<\/password><dbname>.*/\1/")
REMOTEDBNAME=$(cat ./deployment_tmp/system.xml | tr -d '[\n\t ]' | sed -e "s/.*<dbname>\(.*\)<\/dbname>.*/\1/")


###dump the DB on remote machine and fetch it
echo -e "\n\nRequesting DB dump..."
ssh $HOSTUSERNAME@$HOSTNAME "mysqldump --login-path=deployment --routines $REMOTEDBNAME > ~/$REMOTEDBNAME.sql"
scp $HOSTUSERNAME@$HOSTNAME:~/$REMOTEDBNAME.sql ./deployment_tmp/$REMOTEDBNAME-tmp.sql


###sed replace the MySQL DEFINER to avoid errors
cat ./deployment_tmp/$REMOTEDBNAME-tmp.sql | sed "s/\`$REMOTEMYSQLUSER\`/\`$MYSQLUSER\`/g" > ./deployment_tmp/$REMOTEDBNAME.sql


###drop and create new DB user, if needed (no root user)
if [ "$MYSQLUSER" != "root" ]; then
	echo -e "\n\nCreating new DB user and granting privileges."
	MYSQL_ERR=$(mysql --login-path=deployment -e "GRANT USAGE ON *.* TO '$MYSQLUSER'@'localhost'; DROP USER '$MYSQLUSER'@'localhost'; CREATE USER '$MYSQLUSER'@'localhost' IDENTIFIED BY '$MYSQLPASSWORD';GRANT ALL PRIVILEGES ON *.* TO '$MYSQLUSER'@'localhost';" 2>&1 >/dev/null);
	if [[ $MYSQL_ERR == *ERROR* ]]; then
		echo 'Error granting MySQL user permissions. Check the script.'
		exit;
	fi
fi


###import the remote DB on local machine
mysqladmin --login-path=deployment -f drop $DBNAME 2>/dev/null
echo -e "\n\nImporting DB dump..."
mysql --login-path=deployment -e "CREATE DATABASE $DBNAME CHARACTER SET utf8 COLLATE utf8_general_ci" 2>/dev/null
mysql --login-path=deployment $DBNAME < ./deployment_tmp/$REMOTEDBNAME.sql 2>/dev/null


###sync files not in Git
echo -e "\n\nSynchronizing FS..."
mkdir ./website/var/assets 2>/dev/null
mkdir ./website/var/classes/Object 2>/dev/null
mkdir ./website/var/email 2>/dev/null
mkdir ./website/var/tmp 2>/dev/null
mkdir ./website/var/versions 2>/dev/null
mkdir ./website/var/versions/asset 2>/dev/null
mkdir ./website/var/versions/document 2>/dev/null
mkdir ./website/var/versions/object 2>/dev/null
mkdir ./website/var/search 2>/dev/null
mkdir ./website/var/cache 2>/dev/null
#copy .htaccess only if it does not exist
if [[ ! -f ./.htaccess ]]; then
	scp $HOSTUSERNAME@$HOSTNAME:$WEBROOT/.htaccess ./
fi
#use tar on multiple files
ssh $HOSTUSERNAME@$HOSTNAME "tar -czf ~/classes.tar.gz -C $WEBROOT/website/var/classes/Object ."
scp $HOSTUSERNAME@$HOSTNAME:~/classes.tar.gz ./deployment_tmp/classes.tar.gz
tar -xzf ./deployment_tmp/classes.tar.gz -C ./website/var/classes/Object/
if [ "$1" = "-a" ]; then
	ssh $HOSTUSERNAME@$HOSTNAME "tar -czf ~/emails.tar.gz -C $WEBROOT/website/var/email ."
	scp $HOSTUSERNAME@$HOSTNAME:~/emails.tar.gz ./deployment_tmp/emails.tar.gz
	tar -xzf ./deployment_tmp/emails.tar.gz -C ./website/var/email/
	ssh $HOSTUSERNAME@$HOSTNAME "tar -czf ~/assets.tar.gz -C $WEBROOT/website/var/assets ."
	scp $HOSTUSERNAME@$HOSTNAME:~/assets.tar.gz ./deployment_tmp/assets.tar.gz
	tar -xzf ./deployment_tmp/assets.tar.gz -C ./website/var/assets/
fi


###cleanup (remote and locally)
echo -e "\n\nCleaning up..."
ssh $HOSTUSERNAME@$HOSTNAME "rm ~/$REMOTEDBNAME.sql"
ssh $HOSTUSERNAME@$HOSTNAME "rm ~/classes.tar.gz"
if [ "$1" = "-a" ]; then
	ssh $HOSTUSERNAME@$HOSTNAME "rm ~/emails.tar.gz"
	ssh $HOSTUSERNAME@$HOSTNAME "rm ~/assets.tar.gz"
fi
rm -rf ./deployment_tmp
echo -e "\n\nDone."
