<?php

namespace AppBundle\Controller;

use AppBundle\Form\ContactFormType;
use AppBundle\Form\NewsletterFormType;
use AppBundle\Model\News;
use AppBundle\Service\MailManager;
use AppBundle\Service\NewsletterManager;
use Pimcore\File;
use Pimcore\Logger;
use Pimcore\Model\DataObject;
use Pimcore\Model\DataObject\ContactForm;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Form\FormInterface;
use Zend\Paginator\Paginator;

class CmsController extends BaseController
{
	/*						DOCUMENT ROUTED ACTIONS							*/

	/**
	 * @return Response
	 */
	public function templateSimpleAction()
	{
	}

	/**
	 * @return Response
	 */
	public function simpleMailAction()
	{
	}

	/**
	 * @return Response
	 */
	public function sitemapAction()
	{
	}

	/**
	 * @param Request     $request
	 * @param MailManager $mailManager
	 *
	 * @return Response
	 */
	public function contactUsAction(Request $request, MailManager $mailManager)
	{
		$form = $this->createForm(ContactFormType::class);
		$form->handleRequest($request);

		if ($form->isSubmitted()) {
			if ($this->handleContactForm($form, $mailManager)) {
				return $this->gotoUrl($this->docUrl('/kontakt'));
			}
		}

		$this->view->contactForm = $form->createView();
	}

	/**
	 * @param string $page
	 *
	 * @return Response
	 */
	public function newsAction($page = 1)
	{
		$list = new News\Listing();
		$list->setOrder('desc');
		$list->setOrderKey('createdOn');
		$list->setCondition('createdOn < ?', [time()]);

		$paginator = new Paginator($list);
		$paginator->setCurrentPageNumber($page);
		$paginator->setItemCountPerPage(6);

		$this->view->news = $paginator;
	}

	/* 						STATIC-ROUTE ROUTED ACTIONS						 */

	/**
	 * @param string $id
	 *
	 * @return Response
	 */
	public function newsDetailAction($id)
	{
		//output cached stuff
		if (!$this->isOutputCached($this->cacheKeySuffix)) {
			$news = News::getById($id);

			if (!$news) {
				return $this->createNotFoundException($this->translate('system_page_not_found'));
			}

			//get sibblings (next and previous news)
			$list = new News\Listing();
			$list->setOrder('desc');
			$list->setOrderKey('createdOn');
			$list->setCondition('createdOn < ?', [time()]);
			$list->load();
			list($this->view->previousNews, $this->view->nextNews) = $this->getSiblings($list, $news, 'createdOn');

			//extend breadcrumbs
			$this->addToBreadcrumbs[] = [
				'label' => $news->getName(),
				'url' => $news->getUrl(),
			];

			//explicit language switch
			$this->languageSwitch = $this->buildLanguageSwitchFromObject($news);

			//explicit seo stuff
			$this->headTitle->set($news->getName());
			if ($news->getSeoDescription()) {
				$this->headMeta->setName('description', $news->getSeoDescription());
			}

			$this->view->news = $news;
		}
	}

	/*				VIEW-LESS ACTIONS (ALLWAYS REDIRECT)					*/

	/**
	 * @param Request           $request
	 * @param NewsletterManager $newsletterManager
	 *
	 * @return Response
	 */
	public function newsletterSubscribeAction(Request $request, NewsletterManager $newsletterManager)
	{
		$this->disableViewAutoRender();

		$this->newsletterForm->handleRequest($request);
		if ($this->newsletterForm->isSubmitted()) {
			if ($this->handleNewsletterForm($this->newsletterForm, $newsletterManager)) {
				// TODO reset the form
				//$this->newsletterForm = $this->createForm(NewsletterFormType::class);
				//$this->view->newsletterForm = $this->newsletterForm->createView();
				//$this->ajaxResponseBuilder->addElement('#newsletter-form-wrapper', 'Form/newsletter.html.php');
			}
		}

		return $this->ajaxResponseBuilder->addFlashMessages()
			->getResponse();
	}

	/**
	 * @param NewsletterManager $newsletterManager
	 * @param string            $code
	 *
	 * @return Response
	 */
	public function newsletterUnsubscribeAction(NewsletterManager $newsletterManager, $code)
	{
		$this->disableViewAutoRender();

		$email = $newsletterManager->unsubscribe($code);
		if ($email) {
			$this->addSuccessMsg(sprintf($this->translate('msg_newsletter_unsubscribed'), $email));
		} elseif (false === $email) {
			$this->addInfoMsg($this->translate('msg_newsletter_code_invalid'));
		} else {
			$this->addErrorMsg($this->translate('msg_unknown_error'));
		}

		return $this->gotoUrl($this->getPreviousUrl());
	}

	/*							FORM HANDLERS								*/

	/**
	 * @param FormInterface     $form
	 * @param NewsletterManager $newsletterManager
	 *
	 * @return bool
	 */
	private function handleNewsletterForm(FormInterface $form, NewsletterManager $newsletterManager)
	{
		if ($form->isValid()) {
			$data = $form->getData();
			$data['lang'] = $this->language;

			$result = $newsletterManager->subscribe($data);

			if ($result) {
				$this->addSuccessMsg(sprintf($this->translate('msg_newsletter_registered'), $data['email']));
			} else {
				$this->addErrorMsg($this->translate('msg_unknown_error'));
			}

			return $result;
		} else {
			$this->addErrorMsg($this->translate('msg_form_invalid'));
		}

		return false;
	}

	/**
	 * @param FormInterface $form
	 * @param MailManager   $mailManager
	 *
	 * @return bool
	 */
	private function handleContactForm(FormInterface $form, MailManager $mailManager)
	{
		if ($form->isValid()) {
			$data = $form->getData();
			$data['lang'] = $this->language;

			$contactForm = ContactForm::create($data);
			$contactForm->setCreationDate(time());
			$key = File::getValidFilename($data['fullName'].'_'.date('Y-m-d-H:i:s', time()));
			$contactForm->setKey($key);
			$contactForm->setParentId(DataObject\Folder::getByPath('/kontakt/kontaktni-formulare')->getId());
			$contactForm->setPublished(true);

			try {
				$contactForm->save();

				//send email to admin
				$mailManager->sendMail('/notifikace/pro-administratora-novy-kontaktni-formular', $this->language, null, null, [
					'deeplink' => $this->getRequest()->getSchemeAndHttpHost().'/admin/login/deeplink?object_'.$contactForm->getId().'_object',
					'form' => $contactForm,
				]);

				$this->addSuccessMsg($this->translate('msg_contact_form_success'));

				return true;
			} catch (\Exception $e) {
				Logger::error('CONTACT FORM - '.$e->getMessage()."\n".$e->getTraceAsString());
				$this->addErrorMsg($this->translate('msg_unknown_error'));
			}
		} else {
			$this->addErrorMsg($this->translate('msg_form_invalid'));
		}

		return false;
	}
}
