<?php

namespace AppBundle\Tool\Templator;

use Sunra\PhpSimple\HtmlDomParser;
use Symfony\Component\Finder\Finder;

class Parser
{
	const ACTION_COPY_UNTIL_SPECIFIC_OR_FIRST_DIFF = 1;
	const ACTION_COPY_UNTIL_SPECIFIC_OR_END_AND_BREAK_ON_DIFF = 2;
	const ACTION_COPY_AND_BREAK_ON_FORM = 3;
	const ACTION_COPY_TREE = 4;

	/** @var array */
	public $templates = [];
	/** @var string */
	public $indent = "\t";
	/**
	 * switch for inline/non-inline writing mode.
	 *
	 * @var bool
	 */
	protected $inlineMode = false;

	/**
	 * @var Finder
	 */
	protected $finder = null;

	/**
	 * @param string $templatesDir
	 */
	public function __construct($templatesDir)
	{
		$this->finder = new Finder();
		$htmls = $this->finder->in($templatesDir)->depth('== 0')->name('/\.html$/');
		foreach ($htmls as $file) {
			$crawlerWithComments = HtmlDomParser::str_get_html(file_get_contents($file->getRealPath()));
			foreach ($crawlerWithComments->find('comment') as $e) {
				$e->outertext = '';
			}
			$crawler = HtmlDomParser::str_get_html($crawlerWithComments->save());
			$name = basename(current(explode('.', $file->getRealPath())));
			$this->templates[] = ['crawler' => $crawler, 'name' => $name];
		}
	}

	/**
	 * meta config:
	 *  selector - array with a root selector on 0th position and then just numbers for child positions down the tree
	 *  visited -
	 *  skip - skips the node (doesn't skip the children)
	 *  break -
	 *  root -
	 *  selfClosing -.
	 *
	 * @TODO try to unify skip, break and maybe root
	 *
	 * @param int    $crawlerIndex
	 * @param array  $stack
	 * @param string $action
	 * @param int    $indentationLevel
	 *
	 * @return array
	 */
	public function DFS($crawlerIndex, $stack, $action, $indentationLevel = 0)
	{
		$output = '';
		$crawler = $this->templates[$crawlerIndex]['crawler'];

		$indentationOffset = $indentationLevel - count($stack);

		while (!empty($stack)) {
			end($stack);
			$node = $this->find($crawler, $stack);
			$meta = $this->getNodeMetadata($stack, $node);
			if (1 == $meta['visited']) {
				//add a closing tag if necessary
				if (!$meta['skip'] && !$meta['selfClosing']) {
					$output .= $this->addClosingTag($node, count($stack) + $indentationOffset);
				}
				//unset last node from stack
				array_pop($stack);
				//add next sibbling to stack or backtrack to parent node by marking it as visited
				if ($node->next_sibling() && (!$meta['root'] || !$meta['skip'])) {
					$stack[] = ['selector' => $meta['selector'] + 1, 'visited' => 0];
					$actionResult = $this->DFSAction($crawlerIndex, $action, $stack, $output);
					if (isset($actionResult['return'])) {
						return $actionResult['return'];
					}
				} elseif (!$meta['root']) {
					end($stack);
					$stack[key($stack)]['visited'] = 1;
				}
			} else {
				//add the current node content to output
				if (!$meta['skip']) {
					$this->normalizeNode($node); // modifies the $node
					$output .= $this->addOpeningTag($node, count($stack) + $indentationOffset);
				}
				//update the stack and continue
				if (!$node->has_child() || ($meta['skip'] && !$meta['root'])) {
					$stack[key($stack)]['visited'] = 1;
				} else { // add first child to the stack
					$stack[] = ['selector' => 0, 'visited' => 0];
					$actionResult = $this->DFSAction($crawlerIndex, $action, $stack, $output);
					if (isset($actionResult['return'])) {
						return $actionResult['return'];
					}
				}
			}

			//add closing tag if necessary
			//TODO remove this after unifying skip and break
			end($stack);
			if (isset($stack[key($stack)]['break']) && $stack[key($stack)]['break']) {
				if (!isset($stack[key($stack)]['skip']) || !$stack[key($stack)]['skip']) {
					$node = $this->find($crawler, $stack);
					$output .= $this->addClosingTag($node, count($stack) + $indentationOffset);
				}
				break;
			}
		}

		$result = $this->DFSAction($crawlerIndex, $action, $stack, $output, true);

		return $result['return'];
	}

	/**
	 * @param int    $crawlerIndex
	 * @param string $action
	 * @param array  $stack
	 * @param array  $output
	 * @param bool   $finished
	 *
	 * @return array
	 *
	 * @throws TemplatorException
	 */
	protected function DFSAction($crawlerIndex, $action, $stack, &$output, $finished = false)
	{
		$specificElementsRules = [];
		switch ($action) {
			case self::ACTION_COPY_UNTIL_SPECIFIC_OR_FIRST_DIFF:
				$specificElementsRules['class'] = ['header', 'header-wrapper', 'navigation', 'mobile-nav'];
				$specificElementsRules['id'] = ['header', 'header-wrapper', 'navigation', 'mobile-nav'];
				$specificElementsRules['tag'] = ['form'];
				// no break
			case self::ACTION_COPY_UNTIL_SPECIFIC_OR_END_AND_BREAK_ON_DIFF:
				if (self::ACTION_COPY_UNTIL_SPECIFIC_OR_END_AND_BREAK_ON_DIFF == $action) {
					if (empty($stack)) {
						return ['return' => ['stack' => $stack, 'output' => $output, 'finished' => $finished]];
					}
					$specificElementsRules['class'] = ['footer', 'footer-wrapper', 'navigation', 'mobile-nav'];
					$specificElementsRules['id'] = ['footer', 'footer-wrapper', 'navigation', 'mobile-nav'];
					$specificElementsRules['tag'] = ['form'];
				}
				if (self::ACTION_COPY_UNTIL_SPECIFIC_OR_FIRST_DIFF == $action && $finished) {
					throw new TemplatorException('There is no difference in layout.');
				}
				$equal = $this->isEqual($stack);
				$result = ['return' => ['stack' => $stack, 'output' => $output, 'finished' => $finished]];
				if (!$equal && self::ACTION_COPY_UNTIL_SPECIFIC_OR_END_AND_BREAK_ON_DIFF == $action && !$finished) {
					if (count($stack > 1)) {
						$result['return']['deadEnd'] = true;
					} else {
						throw new TemplatorException("There is more than one difference in layout. Next diff at: \n\n".$this->printStackTrace($stack, $this->templates[$crawlerIndex]['crawler']));
					}
				}
				$specificFound = $this->isSpecificElement($stack, $specificElementsRules);
				if ($specificFound) {
					$result['return']['specific'] = $specificFound;

					return $result;
				}
				if (!$equal || $finished) {
					return $result;
				}
				break;
			case self::ACTION_COPY_TREE:
				if ($finished) {
					return ['return' => ['stack' => $stack, 'output' => $output, 'finished' => $finished]];
				}
				break;
			case self::ACTION_COPY_AND_BREAK_ON_FORM:
				$node = $this->find($this->templates[$crawlerIndex]['crawler'], $stack);
				if ('form' == $node->tag || $finished) {
					return ['return' => ['stack' => $stack, 'output' => $output, 'finished' => $finished]];
				}
				break;
			default:
				throw new TemplatorException('Not implemented yet.');
		}

		return [];
	}

	/**
	 * @param array $stack
	 * @param array $rules
	 *
	 * @return bool
	 */
	protected function isSpecificElement(&$stack, &$rules)
	{
		$crawler = $this->templates[0]['crawler'];
		$node = $this->find($crawler, $stack);

		foreach ($rules as $rule => $keywords) {
			switch ($rule) {
				case 'class':
				case 'id':
					$attr = $node->$rule;
					foreach ($keywords as $keyword) {
						if (!empty($attr) && false !== stristr($attr, $keyword)) {
							return $keyword;
						}
					}
					break;
				case 'tag':
					foreach ($keywords as $keyword) {
						if ($node->tag == $keyword) {
							return $keyword;
						}
					}
					break;
			}
		}

		return false;
	}

	/**
	 * @param array $stack
	 *
	 * @return bool
	 */
	protected function isEqual($stack)
	{
		$equal = true;
		$ignoreWords = ['active', 'first', 'last'];

		$toCompare = [];
		foreach ($this->templates as $template) {
			$node = $this->find($template['crawler'], $stack);
			$toCompare[] = [
				'html' => ($node) ? $node->makeup() : '',
				'text' => ($node) ? (($node->has_child()) ? '' : trim($node->innertext)) : '',
			];
		}
		$first = $toCompare[0];
		foreach ($toCompare as $next) {
			if ($first !== $next) {
				$equal = false;
				foreach ($ignoreWords as $iw) {
					if (false !== stristr($first['html'], $iw) || false !== stristr($next['html'], $iw)) {
						$equal = true;
						break;
					}
				}
			}
			if (!$equal) {
				break;
			}
		}

		return $equal;
	}

	/**
	 * @param array                 $stack
	 * @param \simple_html_dom_node $node
	 *
	 * @return type
	 */
	protected function getNodeMetadata(&$stack, $node)
	{
		$meta = current($stack);
		if (!isset($meta['skip'])) {
			$meta['skip'] = false;
		}
		if (!isset($meta['break'])) {
			$meta['break'] = false;
		}
		if (!isset($meta['visited'])) {
			$meta['visited'] = false;
		}
		if (!isset($meta['root'])) {
			$meta['root'] = false;
		}
		$meta['selfClosing'] = in_array($node->tag, [
			'area', 'base', 'br', 'col', 'command', 'embed', 'hr', 'img', 'input',
			'keygen', 'link', 'meta', 'param', 'source', 'spacer', 'track', 'wbr',
			//for svg temporary workaround, TODO just copy the whole svg block
			'circle', 'line', 'path', 'polygon', 'rect', 'use',
		]);

		//$meta['skip'] = ($meta['skip'] || in_array($node->tag, ['script']));

		return $meta;
	}

	/**
	 * @param \simple_html_dom_node $node
	 */
	protected function normalizeNode(&$node)
	{
		if ($node->src && 'http' != substr($node->src, 0, 4) && '/' != $node->src[0]) {
			$node->src = '/'.$node->src;
		}
		if ($node->src && 'http' != substr($node->src, 0, 4) && '/static' != substr($node->src, 0, 7)) {
			$node->src = '/static'.$node->src;
		}
		if ($node->{'xlink:href'} && 'http' != substr($node->{'xlink:href'}, 0, 4) && $node->{'xlink:href'}[0] != '/' && $node->{'xlink:href'}[0] != '#') {
			$node->{'xlink:href'} = '/'.$node->{'xlink:href'};
		}
		if ($node->{'xlink:href'} && 'http' != substr($node->{'xlink:href'}, 0, 4) && $node->{'xlink:href'}[0] != '#' && '/static' != substr($node->{'xlink:href'}, 0, 7)) {
			$node->{'xlink:href'} = '/static'.$node->{'xlink:href'};
		}
		foreach ($node->getAllAttributes() as $key => $value) {
			if ($value && stristr($value, 'url(') && !stristr($value, 'static/')) {
				$node->$key = preg_replace("#^(.*)url\('([0-9a-z/-]+\.[a-z]+)'\)(.*)$#", '${1}url(\'/static/${2}\')${3}', $value);
			}
		}
		if (isset($node->href) && empty($node->href) || '/' == $node->href) {
			$node->href = 'javascript:void(0);';
		}
		if (isset($node->alt) && empty($node->alt)) {
			unset($node->alt);
		}
	}

	/**
	 * @param \simple_html_dom_node $node
	 * @param int                   $indentationMultiplier
	 *
	 * @return string
	 */
	protected function addOpeningTag($node, $indentationMultiplier)
	{
		$output = '';
		//add text in the middle of sibbling tags
		$sibling = $node->prev_sibling();
		$parentNodes = $node->parent()->nodes;
		if ($sibling && count($parentNodes) > count($node->parent()->children)) {
			$idx = 0;
			while (isset($parentNodes[$idx])) {
				if ($parentNodes[$idx]->tag_start == $sibling->tag_start) {
					break;
				}
				$idx += 1;
			}
			if (isset($parentNodes[$idx + 1]) && 'text' == $parentNodes[$idx + 1]->tag) {
				$text = trim($parentNodes[$idx + 1]->innertext);
				if (!empty($text)) {
					if (!$this->inlineMode) {
						$output .= str_repeat($this->indent, $indentationMultiplier);
					}
					$output .= $text;
					if (!$this->inlineMode) {
						$output .= "\n";
					}
				}
			}
		}
		//add opening tag
		$tagHtml = $node->makeup();
		if (!$this->inlineMode) {
			$output .= str_repeat($this->indent, $indentationMultiplier);
		}
		$tit = trim($node->innertext);
		$hasText = (!$tit || ('<' != $tit[0] && '>' != $tit[mb_strlen($tit) - 1]));
		if (!$this->inlineMode && $hasText
			&& in_array($node->tag, ['a', 'i', 'p', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'li', 'tr', 'td', 'span'])) {
			$this->inlineMode = $node->tag;
		}
		$output .= $tagHtml;
		if (!$this->inlineMode) {
			$output .= "\n";
		}
		$text = substr(trim($node->innertext), 0, strspn(trim($node->innertext) ^ trim($node->plaintext), "\0"));
		if (!empty($text)) {
			if (!$this->inlineMode) {
				$output .= str_repeat($this->indent, $indentationMultiplier + 1);
			}
			$output .= $text;
			if (!$this->inlineMode) {
				$output .= "\n";
			}
		}

		return $output;
	}

	/**
	 * @param \simple_html_dom_node $node
	 * @param int                   $indentationMultiplier
	 *
	 * @return string
	 */
	protected function addClosingTag($node, $indentationMultiplier)
	{
		$output = '';
		if (trim($node->innertext) != trim($node->plaintext)) {
			$pos = strspn(strrev(trim($node->innertext)) ^ strrev(trim($node->plaintext)), "\0");
			if ($pos) {
				$text = trim(substr(trim($node->innertext), -$pos));
				if (!$this->inlineMode) {
					$output .= str_repeat($this->indent, $indentationMultiplier + 1);
				}
				$output .= $text;
				if (!$this->inlineMode) {
					$output .= "\n";
				}
			}
		}
		if (!$this->inlineMode) {
			$output .= str_repeat($this->indent, $indentationMultiplier);
		}
		$output .= sprintf('</%s>', $node->tag);
		if ($this->inlineMode && $node->tag == $this->inlineMode) {
			$this->inlineMode = false;
		}
		if (!$this->inlineMode) {
			$output .= "\n";
		}

		return $output;
	}

	/**
	 * @param array                              $stack
	 * @param \simplehtmldom_1_5\simple_html_dom $crawler
	 *
	 * @return string
	 */
	public function printStackTrace($stack, $crawler)
	{
		$output = '';
		for ($i = 0; $i < count($stack); ++$i) {
			$output .= str_repeat($this->indent, $i);
			if (is_int($stack[$i]['selector']) && $stack[$i]['selector'] > 0) {
				for ($j = 1; $j <= $stack[$i]['selector']; ++$j) {
					$output .= "...\n".str_repeat($this->indent, $i);
				}
			}
			$output .= $this->find($crawler, array_slice($stack, 0, $i + 1))->makeup()."\n";
		}

		return $output;
	}

	/**
	 * @param \simplehtmldom_1_5\simple_html_dom $crawler
	 * @param array                              $stack
	 *
	 * @return \simple_html_dom_node
	 */
	public function find($crawler, $stack)
	{
		if (empty($stack)) {
			return;
		}

		$stackArray = array_values($stack);

		if (!is_array($stackArray) || empty($stackArray)) {
			return;
		}

		$tmp = $crawler->find($stackArray[0]['selector'], 0);

		if (!$tmp) {
			return;
		}

		for ($i = 1; $i < count($stackArray); ++$i) {
			$tmp = $tmp->children($stackArray[$i]['selector']);
			if (!$tmp) {
				break;
			}
		}

		return $tmp;
	}

	/**
	 * @param array $htmlAttribs
	 * @param array $bodyAttribs
	 *
	 * @return array
	 */
	public function getHtmlBodyAttribs($htmlAttribs = [], $bodyAttribs = [])
	{
		foreach ($this->templates as $template) {
			$crawler = $template['crawler'];
			foreach ((array) $crawler->find('html', 0)->attr as $name => $value) {
				if ('lang' != $name) {
					if (!isset($htmlAttribs[$name])) {
						$htmlAttribs[$name] = [];
					}
					if (!in_array($value, $htmlAttribs[$name])) {
						$htmlAttribs[$name][] = $value;
					}
				}
			}
			foreach ((array) $crawler->find('body', 0)->attr as $name => $value) {
				if (!isset($bodyAttribs[$name])) {
					$bodyAttribs[$name] = [];
				}
				if (!in_array($value, $bodyAttribs[$name])) {
					$bodyAttribs[$name][] = $value;
				}
			}
		}
		// concat attrib sets
		foreach ($bodyAttribs as $name => $set) {
			$bodyAttribs[$name] = sprintf('%s="%s"', $name, implode(' ', $set));
		}
		foreach ($htmlAttribs as $name => $set) {
			$htmlAttribs[$name] = sprintf('%s="%s"', $name, implode(' ', $set));
		}

		return [$htmlAttribs, $bodyAttribs];
	}

	/**
	 * @param array $css
	 * @param array $js
	 * @param array $inlineScripts
	 * @param array $externalCssAndJs
	 *
	 * @return array
	 */
	public function getCssJsIncludes($css = [], $js = [], $inlineScripts = [], $externalCssAndJs = [])
	{
		foreach ($this->templates as $template) {
			$crawler = $template['crawler'];
			foreach ($crawler->find('link[rel="stylesheet"],script[src]') as $element) {
				$url = ('link' == $element->tag) ? $element->href : $element->src;
				// ignore external links
				if ((strlen($url) >= 4 && 'http' == substr($url, 0, 4)) || (strlen($url) >= 2 && '//' == substr($url, 0, 2))) {
					$externalCssAndJs[] = $element->outertext;
					continue;
				}
				// normalize url
				if ($url && '/' != $url[0]) {
					$url = '/'.$url;
				}
				if ('/static' != substr($url, 0, 7)) {
					$url = '/static'.$url;
				}
				if ('link' == $element->tag) {
					$css[] = $url;
				} else {
					$js[] = $url;
				}
			}
			foreach ($crawler->find('script[!src]') as $element) {
				$scriptContent = trim($element->outertext);
				if (!in_array($scriptContent, $inlineScripts)) {
					$inlineScripts[] = $scriptContent;
				}
			}
		}
		// return only uniqueu includes
		$cssIncludes = array_values(array_unique($css));
		$jsIncludes = array_values(array_unique($js));
		$externalCssAndJsIncludes = array_values(array_unique($externalCssAndJs));

		return [$cssIncludes, $jsIncludes, $inlineScripts, $externalCssAndJsIncludes];
	}

	public function removeScriptTags()
	{
		foreach ($this->templates as $key => $template) {
			foreach ($template['crawler']->find('script') as $e) {
				$e->outertext = '';
			}
			$crawler = HtmlDomParser::str_get_html($template['crawler']->save());
			$this->templates[$key]['crawler'] = $crawler;
		}
	}
}

class TemplatorException extends \Exception
{
}
