<?php declare(strict_types=1);

namespace App\Controller\Helper;

use App\Tool\Utils;
use Pimcore\Bundle\StaticRoutesBundle\Model\Staticroute;
use Pimcore\Cache\RuntimeCache;
use Pimcore\Model\DataObject;
use Pimcore\Model\DataObject\Concrete;
use Pimcore\Model\DataObject\Listing;
use Pimcore\Model\Site;
use Pimcore\Tool;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\User\UserInterface;

trait ControllerTrait
{
    protected array $languageSwitch = [];
    protected array $addToBreadcrumbs = [];

    protected function getRequest(): Request
    {
        return $this->request;
    }

    protected function getLanguage(): string
    {
        return $this->language;
    }

    protected function docUrl(string $realFullPath, string $language = null): string
    {
        if (null === $language) {
            $language = $this->getLanguage();
        }

        return Utils::docUrl($realFullPath, $language);
    }

    protected function docUrlById(int $documentId, string $language = null): string
    {
        if (null === $language) {
            $language = $this->getLanguage();
        }

        return Utils::docUrlById($documentId, $language);
    }

    protected function addSuccessMsg(string $msg, bool $forCurrentRequest = false): void
    {
        if ($forCurrentRequest) {
            $this->addFlashForCurrentRequest('success', $msg);
        } else {
            $this->addFlash('success', $msg);
        }
    }

    protected function addErrorMsg(string $msg, bool $forCurrentRequest = false): void
    {
        if ($forCurrentRequest) {
            $this->addFlashForCurrentRequest('danger', $msg);
        } else {
            $this->addFlash('danger', $msg);
        }
    }

    protected function addInfoMsg(string $msg, bool $forCurrentRequest = false): void
    {
        if ($forCurrentRequest) {
            $this->addFlashForCurrentRequest('info', $msg);
        } else {
            $this->addFlash('info', $msg);
        }
    }

    private function addFlashForCurrentRequest(string $type, string $message): void
    {
        $flashMessages = [];

        if (RuntimeCache::isRegistered('currentRequestFlashMessages')) {
            $flashMessages = RuntimeCache::get('currentRequestFlashMessages');
        }

        $flashMessages[$type][] = $message;

        RuntimeCache::save($flashMessages, 'currentRequestFlashMessages');
    }

    protected function gotoUrl(string $url, int $code = 302): ?RedirectResponse
    {
        if ($this->getRequest()->get('elastic_document_indexing')) {
            return null;
        }

        return $this->redirect($url, $code);
    }

    /**
     * @param array<string|int> $params
     */
    protected function gotoRoute(string $route, array $params = [], int $code = 302): ?RedirectResponse
    {
        return $this->gotoUrl(Utils::url($route, $params), $code);
    }

    protected function translate(string $keyword): string
    {
        return $this->translator->trans($keyword);
    }

    protected function isAjax(): bool
    {
        return $this->getRequest()->isXmlHttpRequest() &&
            !\App\Tool\ElasticSearch::isUpdateRequest() &&
            !$this->getRequest()->get('elastic_document_indexing')
        ;
    }

    protected function isAdminFrontendRequest(): bool
    {
        return Tool::isFrontendRequestByAdmin();
    }

    protected function isSiteRequest(): bool
    {
        return Site::isSiteRequest();
    }

    protected function isIndexingRequest(): bool
    {
        return (bool)$this->getRequest()->get('elastic_document_indexing');
    }

    /**
     * checks if there is a cache hit for given key.
     */
    protected function isOutputCached(string $cacheKey): bool
    {
        $viewCacheKeyPrefix = 'pimcore_viewcache_';

        if (\Pimcore\Tool::isFrontendRequestByAdmin() ||
                false === \Pimcore\Cache::load($viewCacheKeyPrefix.$cacheKey) ||
                $this->getRequest()->get('elastic_document_indexing')) {
            return false;
        } else {
            return true;
        }
    }

    protected function isEditOrAdminMode(): bool
    {
        if ($this->editmode) {
            return true;
        }
        if (\Pimcore\Tool::isFrontendRequestByAdmin()) {
            $admin = \Pimcore\Tool\Authentication::authenticateSession();
            if ($admin instanceof \Pimcore\Model\User) {
                return true;
            }
        }

        return false;
    }

    protected function getPreviousUrl(string $defaultUri = null): string
    {
        $referer = $this->getRequest()->cookies->get('referer');
        $httpReferer = $this->getHttpReferer();

        if ($referer) {
            setcookie('referer', '', time() - 10, '/');

            return (string)$referer;
        }

        if ($defaultUri) {
            return $defaultUri;
        }

        if ($httpReferer) {
            return $httpReferer;
        }

        return Utils::docUrl('/', $this->language);
    }

    protected function setPreviousUrl(
        string $relativeUri,
        bool $force = false,
        int $lifetime = 600,
    ): void {
        $isRefererNotSet = !$this->getRequest()->cookies->get('referer', null);

        if (($isRefererNotSet || $force) && $relativeUri) {
            setcookie('referer', $relativeUri, time() + $lifetime, '/');
        }
    }

    protected function getHttpReferer(): ?string
    {
        $parts1 = parse_url($this->getRequest()->getSchemeAndHttpHost());
        $parts2 = parse_url($this->getRequest()->server->get('HTTP_REFERER'));

        if (empty($parts1['scheme']) || empty($parts1['host']) || empty($parts2['scheme']) || empty($parts2['host'])) {
            return null;
        }

        if ($parts1['scheme'] === $parts2['scheme'] && $parts1['host'] === $parts2['host'] && isset($parts2['path'])) {
            $query = '';

            if (!empty($parts2['query'])) {
                $query = '?' . $parts2['query'];
            }

            return ('' !== $parts2['path'] && '/' !== $parts2['path'])
                ? $parts2['path'] . $query
                : null;
        }

        return null;
    }

    protected function setCanonicalUrl(?string $url = null): void
    {
        if (!$url) {
            $url = preg_replace(
                '/\?.*/',
                '',
                $this->getRequest()->getSchemeAndHttpHost().$this->currentUrl
            );
        }
        if ($url && 'http' != substr($url, 0, 4)) {
            $url = $this->getRequest()->getSchemeAndHttpHost().$url;
        }

        $this->headLink->appendAlternate([
            'rel' => 'canonical',
            'href' => $url,
        ], null, null, null);
    }

    protected function checkAuth(): ?Response
    {
        // mock user in editmode
        if ($this->isEditOrAdminMode()) {
            //$user = new User();
            //$user->mockForEditmode();
        } else {
            if (!$this->isGranted('ROLE_USER')) {
                $this->addErrorMsg($this->translate('msg_authentication_required'));

                return $this->gotoUrl($this->docUrl('/prihlaseni'));
            }
        }

        return null;
    }

    protected function getCurrentUser(): ?UserInterface
    {
        return $this->getUser();
    }

    protected function buildLanguageSwitchFromObject(Concrete $object, ?string $hpFallbackPath = null): array
    {
        $switcher = [];
        $languages = Utils::getValidFrontendLanguages();

        if (method_exists($object, 'getUrl')) {
            foreach ($languages as $language) {
                $url = $object->getUrl($language);
                if (!$url && $hpFallbackPath) {
                    $url = $this->docUrl($hpFallbackPath, $language);
                    if ('/' == $url) {
                        $url = null;
                    }
                }
                if ($url) {
                    $switcher[$language] = $url;
                }
            }
        }

        return $switcher;
    }

    /**
     * returns associative array with localized URLs for each language, e.g. ['en' => 'en_url', 'cs' => 'cs_url']
     */
    protected function buildLanguageSwitchFromRoute(string $routeName, bool $routeForEachLanguage = true, array $params = []): array
    {
        $switcher = [];
        $languages = Utils::getValidFrontendLanguages();
        $route = null;

        if (!$routeForEachLanguage) {
            $route = Staticroute::getByName($routeName, (\Pimcore\Model\Site::isSiteRequest()) ? \Pimcore\Model\Site::getCurrentSite()->getId() : 0);
        }

        foreach ($languages as $language) {
            $baseUrl = '';
            $siteId = 0;
            if ($routeForEachLanguage) {
                if (\Pimcore\Model\Site::isSiteRequest()) {
                    $siteRootId = \Pimcore\Model\Document\Page::getByPath('/'.$language)?->getId();
                    if ($siteRootId && $site = \Pimcore\Model\Site::getByRootId($siteRootId)) {
                        $siteId = $site->getId();
                        $baseUrl = $this->getRequest()->getScheme().'://'.$site->getMainDomain();
                    }
                }
                $route = Staticroute::getByName($routeName.$language, $siteId);
            } else {
                $params['language'] = $language;
            }

            if ($route) {
                $switcher[$language] = $baseUrl . $route->assemble($params, true);
            }
        }

        return $switcher;
    }

    /**
     * @return array{0: DataObject\Concrete|null, 1: DataObject\Concrete|null}
     */
    protected function getSiblings(Listing\Concrete $list, Concrete $currentObject, string $param)
    {
        $previous = null;
        $tmpList = $list->getObjects();
        end($tmpList);
        $tmpPrevious = current($tmpList); //last
        reset($tmpList);
        $next = current($tmpList); //first
        $loadNext = false;
        foreach ($list as $tmpCurrent) {
            if (true == $loadNext) {
                $next = $tmpCurrent;
                $loadNext = false;
            }
            if ($currentObject->{'get' . ucfirst($param)}() == $tmpCurrent->{'get' . ucfirst($param)}()) {
                $previous = $tmpPrevious;
                $loadNext = true;
            }
            $tmpPrevious = $tmpCurrent;
        }

        return [$previous, $next];
    }
}
