<?php declare(strict_types=1);

namespace App\Service;

use Pimcore\Bundle\ApplicationLoggerBundle\ApplicationLogger;
use Pimcore\Mail;
use Pimcore\Model\Asset;
use Pimcore\Model\Document;
use Pimcore\Model\Document\Email;
use Psr\Log\LogLevel;

class MailService
{
    public function __construct(protected Document\Service $documentService, protected ApplicationLogger $applicationLogger)
    {
    }

    public function sendMail(
        Email|string $pathOrTemplate,
        string $language,
        string $to = null,
        string $from = null,
        array $params = null,
        Asset $attachment = null,
        string $replyTo = null
    ): bool {
        if ($pathOrTemplate instanceof Email) {
            $document = $pathOrTemplate;
        } else {
            $document = Email::getByPath($pathOrTemplate);

            if (!$document) {
                return false;
            }

            if ($document->getProperty('language') != $language) {
                $translations = $this->documentService->getTranslations($document);
                if (isset($translations[$language])) {
                    $translatedDocument = Email::getById($translations[$language]);
                    if ($translatedDocument) {
                        $document = $translatedDocument;
                    }
                }
            }
        }

        $mail = new Mail();
        $mail->setDocument($document);

        if ($to) {
            $mail->to($to);
        }
        if ($from) {
            $mail->from($from);
        }

        if ($replyTo) {
            $mail->addReplyTo($replyTo);
        }
        if ($params) {
            $mail->setParams($params);
        }

        if ($attachment instanceof Asset) {
            $mail->attach($attachment->getData(), $attachment->getFilename(), $attachment->getMimeType());
        }

        try {
            $mail->send();

            return true;
        } catch (\Exception $e) {
            $this->applicationLogger->logException(
                sprintf('Send mail %s to %s', $document->getFullPath(), $to),
                $e,
                LogLevel::ERROR,
                null,
                __METHOD__ . ':' . __LINE__,
            );

            return false;
        }
    }
}
