<?php declare(strict_types=1);

namespace App\Service;

use App\Model\DataObject\User;
use App\Tool\Utils;
use Pimcore\Bundle\ApplicationLoggerBundle\ApplicationLogger;
use Pimcore\File;
use Pimcore\Translation\Translator;
use Psr\Log\LogLevel;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

class UserService
{
    const ROOT_FOLDER_ID = 3;

    protected Request $request;

    public function __construct(
        protected Translator $translator,
        protected MailService $mailService,
        protected NewsletterService $newsletterService,
        protected ApplicationLogger $applicationLogger,
        protected RequestStack $requestStack
    ) {
        $this->request = $requestStack->getCurrentRequest();
    }

    public function createUser(array $data): bool
    {
        $data['activationCode'] = uniqid();

        try {
            $user = User::create($data);
            $user->setKey(File::getValidFilename($data['email']));
            $user->setParentId(self::ROOT_FOLDER_ID);
            $user->setPublished(false);
            $user->save();

            if (!empty($data['newsletter'])) {
                $this->newsletterService->subscribe($data);
            }

            $this->mailService->sendMail('/notifikace/registrace', $data['lang'], $data['email'], null, [
                'siteUrl' => $this->request->getSchemeAndHttpHost(),
                'activationUrl' => $this->request->getSchemeAndHttpHost() . Utils::url('activate-account', [
                    'language' => $data['lang'],
                    'code' => $data['activationCode'],
                ]),
                'user' => $user,
            ]);

            return true;
        } catch (\Exception $e) {
            $this->applicationLogger->logException(
                sprintf('Create user %s', $data['email']),
                $e,
                LogLevel::ERROR,
                null,
                __METHOD__ . ':' . __LINE__,
            );

            return false;
        }
    }
}
