<?php

namespace App\Tool;

use Pimcore\Db;
use Pimcore\File;
use Pimcore\Logger;
use Pimcore\Model\DataObject\ClassDefinition;
use Symfony\Component\Yaml\Yaml;

class Generator
{
    const CONTROLLER_SECTION_DOCUMENT_ACTIONS = 'DOCUMENT ROUTED ACTIONS';
    const CONTROLLER_SECTION_FORM_ACTIONS = 'FORM HANDLERS';

    public function createClassDefinition(string $name, bool $url = false, bool $localized = false,
        string $icon = null, int|string $id = null, string $definitionPath = null): bool|int|string
    {
        if ($id && $definitionPath) {
            if (!file_exists($definitionPath)) {
                Logger::error(
                    'CREATING CLASS - definition file '.$definitionPath.' not found',
                    ['component' => 'generator']
                );

                return false;
            }
            //we have to insert a row to `classes` table manually before saving the definition
            $db = Db::get();
            try {
                $saved = $db->insert('classes', [
                    'name' => $name,
                    'id' => $id,
                ]);
            } catch (\Exception $e) {
                $saved = false;
            }
            if (!$saved) {
                Logger::error(
                    'CREATING CLASS - Could not insert row into `classes` table',
                    ['component' => 'generator']
                );

                return false;
            }

            $classDefinition = include $definitionPath;
            $classDefinition->setId($id);
        } else {
            if (!$id) {
                $id = strtolower(preg_replace('/(?<!^)[A-Z]/', '_$0', $name));
            }
            $classDefinition = new ClassDefinition();
            $classDefinition->setName($name);
            $classDefinition->setId($id);
            $classDefinition->setCreationDate(time());
            $classDefinition->setModificationDate(time());
            $classDefinition->setUserOwner(2);
            $classDefinition->setUserModification(2);
            $classDefinition->setAllowInherit(false);
            $classDefinition->setAllowVariants(false);
            if ($icon) {
                $classDefinition->setIcon($this->normalizePimcoreIconPath($icon));
            }
            $classDefinition->setShowVariants(false);
            $classDefinition->setPropertyVisibility([
                'grid' => [
                    'id' => true,
                    'path' => false,
                    'published' => true,
                    'modificationDate' => false,
                    'creationDate' => true,
                ],
                'search' => [
                    'id' => true,
                    'path' => true,
                    'published' => true,
                    'modificationDate' => true,
                    'creationDate' => true,
                ],
            ]);
            $fieldDefinitions = [];
            $nameInput = new ClassDefinition\Data\Input();
            $nameInput->setWidth(($localized) ? 550 : 600);
            $nameInput->setName('name');
            $nameInput->setTitle('Name');
            $nameInput->setMandatory(false);
            $nameInput->setVisibleGridView(true);
            $nameInput->setVisibleSearch(true);
            $urlInput = null;
            if ($url) {
                $urlInput = new ClassDefinition\Data\CalculatedValue();
                $urlInput->setCalculatorClass('\App\Service\ValueCalculator');
                $urlInput->setWidth(($localized) ? 550 : 600);
                $urlInput->setName('url');
                $urlInput->setTitle('Url');
                $urlInput->setMandatory(false);
                $urlInput->setNoteditable(true);
                $urlInput->setVisibleGridView(true);
                $urlInput->setVisibleSearch(false);
            }
            if ($localized) {
                $lf = new ClassDefinition\Data\Localizedfields();
                $lf->setName('localizedfields');
                $lf->setWidth(700);
                $lf->addChild($nameInput);
                if ($urlInput) {
                    $lf->addChild($urlInput);
                }
                $fieldDefinitions['localizedfields'] = $lf;
            } else {
                $fieldDefinitions['name'] = $nameInput;
                if ($urlInput) {
                    $fieldDefinitions['url'] = $urlInput;
                }
            }
            $classDefinition->setFieldDefinitions($fieldDefinitions);

            $panel = new ClassDefinition\Layout\Panel();
            $panel->setLabelWidth(100);
            $panel->setName($name);
            $panel->setChildren(array_values($fieldDefinitions));
            $layout = new ClassDefinition\Layout\Panel();
            $layout->setLabelWidth(100);
            $layout->setName('pimcore_root');
            $layout->addChild($panel);

            $classDefinition->setLayoutDefinitions($layout);
        }

        try {
            $classDefinition->save();

            // check if the ID checks out
            if ($id && $classDefinition->getId() != $id) {
                throw new \Exception(sprintf('ID mismatch [%d]', $classDefinition->getId()));
            }

            return $classDefinition->getId();
        } catch (\Exception $e) {
            \Pimcore\Logger::error(
                'CREATING CLASS - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }
    }

    public function deleteClassDefinition(string $name): bool
    {
        try {
            $classDefinition = ClassDefinition::getByName($name);
        } catch (\Exception $e) {
            $classDefinition = null;
        }
        if (!$classDefinition) {
            Logger::error(
                'DELETING CLASS - '.$name.'not found',
                ['component' => 'generator']
            );

            return false;
        }

        try {
            $classDefinition->delete();
        } catch (\Exception $e) {
            Logger::error(
                'DELETING CLASS - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }

        return true;
    }

    public function createClassMapping(string|int $classNameOrId, bool $mapList = true): bool
    {
        $className = $this->getClassName($classNameOrId);
        if (!$className) {
            return false;
        }
        $configPath = PIMCORE_PROJECT_ROOT.'/config/config.yaml';
        try {
            $classMap = Yaml::parseFile($configPath);
        } catch (\Exception $e) {
            Logger::error(
                'CREATING CLASS MAPPING - '.$e->getMessage(),
                ['component' => 'generator']
            );

            return false;
        }

        $parent = sprintf('Pimcore\Model\DataObject\%s', $className);
        $parentList = sprintf('Pimcore\Model\DataObject\%s\Listing', $className);
        $child = sprintf('App\Model\DataObject\%s', $className);
        $childList = sprintf('App\Model\DataObject\%s\Listing', $className);
        if (!isset($classMap['pimcore'])) {
            $classMap['pimcore'] = [];
        }
        if (!isset($classMap['pimcore']['models'])) {
            $classMap['pimcore']['models'] = [];
        }
        if (!isset($classMap['pimcore']['models']['class_overrides'])) {
            $classMap['pimcore']['models']['class_overrides'] = [];
        }
        if (isset($classMap['pimcore']['models']['class_overrides'][$parent]) || ($mapList && isset($classMap['pimcore']['models']['class_overrides'][$parentList]))) {
            Logger::error(
                sprintf('CREATING CLASS MAPPING - mapping for class %s already exists', $className),
                ['component' => 'generator']
            );

            return false;
        }

        $classMap['pimcore']['models']['class_overrides'][$parent] = $child;
        if (!$this->createMappedClass($className)) {
            return false;
        }
        if ($mapList) {
            $classMap['pimcore']['models']['class_overrides'][$parentList] = $childList;
            if (!$this->createMappedClassList($className)) {
                return false;
            }
        }

        return false !== file_put_contents($configPath, Yaml::dump($classMap, 99));
    }

    public function deleteClassMapping(string|int $classNameOrId): bool
    {
        $className = $this->getClassName($classNameOrId);
        if (!$className) {
            return false;
        }
        $configPath = PIMCORE_PROJECT_ROOT.'/config/config.yaml';
        try {
            $classMap = Yaml::parseFile($configPath);
        } catch (\Exception $e) {
            Logger::error(
                'CREATING CLASS MAPPING - '.$e->getMessage(),
                ['component' => 'generator']
            );

            return false;
        }

        $parent = sprintf('Pimcore\Model\DataObject\%s', $className);
        $parentList = sprintf('Pimcore\Model\DataObject\%s\Listing', $className);
        if (isset($classMap['pimcore']['models']['class_overrides'][$parent])) {
            unset($classMap['pimcore']['models']['class_overrides'][$parent]);
            if (!$this->deleteMappedClass($className)) {
                return false;
            }
        }
        if (isset($classMap['pimcore']['models']['class_overrides'][$parentList])) {
            unset($classMap['pimcore']['models']['class_overrides'][$parentList]);
            if (!$this->deleteMappedClassList($className)) {
                return false;
            }
        }

        return false !== file_put_contents($configPath, Yaml::dump($classMap, 99));
    }

    public function createMappedClass(string $className): bool
    {
        $template = '<?php

namespace App\Model\DataObject;

use Pimcore\Model\DataObject;

class %s extends DataObject\%s
{

	/*								Helpers										*/

	/*								Mapped										*/

	/* 								For elastic search							*/

}
';

        $modelPath = sprintf('%s/src/Model/DataObject/%s.php', PIMCORE_PROJECT_ROOT, $className);
        try {
            if (!file_exists($modelPath)) {
                $result = $this->writePhpFile($modelPath, vsprintf($template, [
                    $className,
                    $className,
                ]));
                if (!$result) {
                    return false;
                }
            }
        } catch (\Exception $e) {
            Logger::error(
                'CREATING MAPPED CLASS - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }

        return true;
    }

    public function createMappedClassList(string $className): bool
    {
        $template = '<?php

namespace App\Model\DataObject\%s;

use Pimcore\Model\DataObject;

class Listing extends DataObject\%s\Listing
{
}
';

        $modelPath = sprintf('%s/src/Model/DataObject/%s/Listing.php', PIMCORE_PROJECT_ROOT, $className);
        try {
            if (!file_exists($modelPath)) {
                $result = $this->writePhpFile($modelPath, vsprintf($template, [
                    $className,
                    $className,
                ]));
                if (!$result) {
                    return false;
                }
            }
        } catch (\Exception $e) {
            Logger::error(
                'CREATING MAPPED CLASS LIST - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }

        return true;
    }

    public function deleteMappedClass(string $className): bool
    {
        $modelPath = PIMCORE_PROJECT_ROOT.'/src/Model/DataObject/'.$className.'.php';
        dump($modelPath);
        exit;
        try {
            if (file_exists($modelPath)) {
                unlink($modelPath);
            }
        } catch (\Exception $e) {
            Logger::error(
                'DELETING MAPPED CLASS - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }

        return true;
    }

    public function deleteMappedClassList(string $className): bool
    {
        $modelsDir = PIMCORE_PROJECT_ROOT.'/src/Model/DataObject';
        $modelPath = sprintf('%s/%s/Listing.php', $modelsDir, $className);
        try {
            if (file_exists($modelPath)) {
                unlink($modelPath);
            }
            if (file_exists($modelsDir.'/'.$className)) {
                rmdir($modelsDir.'/'.$className);
            }
        } catch (\Exception $e) {
            Logger::error(
                'DELETING MAPPED CLASS LIST - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }

        return true;
    }

    public function createArea(string $name, string $icon): bool
    {
        $className = preg_replace('@[-_ ]+@', '', ucwords($name, '-_ '));
        $fullClassName = sprintf('App\Document\Areabrick\%s', $className);
        $areaFolderPath = sprintf('%s/src/Document/Areabrick', PIMCORE_PROJECT_ROOT);
        $areaViewFolderPath = sprintf('%s/templates/areas/%s', PIMCORE_PROJECT_ROOT, $name);

        // create folder
        if (file_exists($areaFolderPath.'/'.$className.'.php') || file_exists($areaViewFolderPath)) {
            Logger::error(
                sprintf(
                    'CREATING AREA BLOCK - area [%s] already exists (either view or class)',
                    $name
                ),
                ['component' => 'generator']
            );

            return false;
        }

        // enable area in extensions.php
        $extensionsPath = sprintf('%s/config/bricks.yaml', PIMCORE_PROJECT_ROOT);
        $exConfig = Yaml::parseFile($extensionsPath);
        if (!isset($exConfig['services'][$fullClassName])) {
            $exConfig['services'][$fullClassName] = [
                'tags' => [
                    [
                        'name' => 'pimcore.area.brick',
                        'id' => $name
                    ]
                ]
            ];

            file_put_contents($extensionsPath, Yaml::dump($exConfig, 2, 4));
        }

        // create action.php and view.php files
        $actionTemplate = '<?php declare(strict_types=1);

namespace App\Document\Areabrick;

use Pimcore\Extension\Document\Areabrick\AbstractTemplateAreabrick;
use Pimcore\Model\Document\Editable\Area\Info;

class %s extends AbstractTemplateAreabrick
{
    public function getName(): string
    {
        return \'areabrick_%s\';
    }

    public function getDescription(): string
    {
        return \'areabrick_%s_description\';
    }

    public function getIcon(): string
    {
		return \'%s\';
    }

    public function getHtmlTagOpen(Info $info): string
    {
        return \'\';
    }

    public function getHtmlTagClose(Info $info): string
    {
        return \'\';
    }
}';
        try {
            $result2 = $this->writePhpFile(sprintf($areaFolderPath.'/%s.php', $className), vsprintf($actionTemplate, [
                $className,
                $name,
                $name,
                $icon ? $this->normalizePimcoreIconPath($icon) : '/bundles/pimcoreadmin/img/flat-color-icons/html.svg',
            ]));
            if (!$result2) {
                return false;
            }
            $result3 = $this->writePhpFile($areaViewFolderPath.'/view.html.twig', '');
            if (!$result3) {
                return false;
            }
        } catch (\Exception $e) {
            Logger::error(
                'CREATING AREA BLOCK - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }

        return true;
    }

    public function deleteArea(string $name): bool
    {
        $areaFolderPath = sprintf('%s/src/Document/Areabrick', PIMCORE_PROJECT_ROOT);
        $areaViewFolderPath = sprintf('%s/templates/areas/%s', PIMCORE_PROJECT_ROOT, $name);
        $className = preg_replace('@[-_ ]+@', '', ucwords($name, '-_ '));
        $fullClassName = sprintf('App\Document\Areabrick\%s', $className);

        // remove files
        try {
            if (file_exists($areaViewFolderPath)) {
                foreach (scandir($areaViewFolderPath) as $fileOrFolder) {
                    if (!in_array($fileOrFolder, ['.', '..'])) {
                        unlink($areaViewFolderPath.'/'.$fileOrFolder);
                    }
                }
                rmdir($areaViewFolderPath);
            }
            if (file_exists($areaFolderPath.'/'.ucfirst($name).'.php')) {
                unlink($areaFolderPath.'/'.ucfirst($name).'.php');
            }
        } catch (\Exception $e) {
            Logger::error(
                'DELETING AREA BLOCK - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }

        // disable area in extensions.php
        $extensionsPath = sprintf('%s/config/bricks.yaml', PIMCORE_PROJECT_ROOT);
        $exConfig = Yaml::parseFile($extensionsPath);
        if (isset($exConfig['services'][$fullClassName])) {
            unset($exConfig['services'][$fullClassName]);

            file_put_contents($extensionsPath, Yaml::dump($exConfig, 2, 4));
        }
    }

    public function normalizeScriptFileName(string $action): string
    {
        return strtolower(preg_replace('/([A-Z]+)/', '-$1', lcfirst($action)));
    }

    public function normalizePathParam(string $path, bool $startWithSlash = true): string
    {
        if (stristr($path, 'Program Files') && stristr($path, '/Git/')) {
            $pos = stripos($path, '/Git/');
            $path = substr($path, $pos + 4);
        }
        if ($startWithSlash && '/' != $path[0]) {
            $path = '/'.$path;
        }

        return $path;
    }

    public function normalizePimcoreIconPath(string $path, bool $startWithSlash = true): string
    {
        if ($startWithSlash && '/' != $path[0]) {
            $path = '/'.$path;
        }
        if ('/web/pimcore/' === substr($path, 0, 13)) {
            $path = substr($path, 4);
        }

        return $path;
    }

    private function getClassName(string|int $classNameOrId): false|string
    {
        $className = null;
        try {
            $class = ClassDefinition::getById($classNameOrId);
            if (!$class) {
                $class = ClassDefinition::getByName($classNameOrId);
            }
        } catch (\Exception $e) {
        }

        if ($class) {
            $className = $class->getName();
        }

        if (!$className) {
            Logger::error(
                sprintf(
                    'GETTING CLASS NAME - class [%s] does not exist',
                    $classNameOrId
                ),
                ['component' => 'generator']
            );

            return false;
        }

        return $className;
    }

    private function writePhpFile(string $path, string $data, bool $toPhpDataFileFormat = false): bool
    {
        if ($toPhpDataFileFormat) {
            $data = to_php_data_file_format($data);
        }
        // replace spaces with tabs
        $dataWithTabs = $this->replaceSpacesWithTabs($data);
        try {
            File::putPhpFile($path, $dataWithTabs);
        } catch (\Exception $e) {
            Logger::error(
                'WRITING PHP FILE - '.$e->getMessage()."\n".$e->getTraceAsString(),
                ['component' => 'generator']
            );

            return false;
        }

        return true;
    }

    private function replaceSpacesWithTabs(string $content): string
    {
        $rows = explode("\n", $content);

        foreach ($rows as $i => $row) {
            $tmp = preg_replace("/\G\t/", '    ', $row);
            $rows[$i] = preg_replace("/\G {4}/", "\t", $tmp);
        }

        return implode("\n", $rows);
    }
}
