<?php declare(strict_types=1);

namespace App\Twig\Extension;

use App\Tool\Utils;
use Pimcore\Model\Document\Editable\EditableInterface;
use Pimcore\Model\Document\Page;
use Pimcore\Templating\Renderer\EditableRenderer;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class DocExtension extends AbstractExtension
{
    public function __construct(protected EditableRenderer $editableRenderer)
    {
    }

    public function getFunctions(): array
    {
        return [
            new TwigFunction('doc_url', [$this, 'docUrl'], ['needs_context' => true]),
            new TwigFunction('doc_url_by_id', [$this, 'docUrlById'], ['needs_context' => true]),
            new TwigFunction('doc_is_active', [$this, 'docIsActive'], ['needs_context' => true]),
            new TwigFunction('doc_editable', [$this, 'docEditable'], ['needs_context' => true, 'is_safe' => ['html']]),
        ];
    }

    public function docUrl(array $context, string $realFullPath): string
    {
        return Utils::docUrl($realFullPath, $context['language']);
    }

    public function docUrlById(array $context, int $id): string
    {
        return Utils::docUrlById($id, $context['language']);
    }

    public function docIsActive(array $context, int $id): bool
    {
        $document = $context['document'] ?? null;

        return $id && $document instanceof Page &&
            ($document->getId() == $id || $document->getContentMainDocumentId() == $id);
    }

    public function docEditable(array $context, string $name, string $type, array $options = []): EditableInterface
    {
        $defaultConfigurations = [
            'input' => ['htmlspecialchars' => true],
            'numeric' => ['minValue' => 0],
            'textarea' => ['height' => 100, 'nl2br' => true, 'htmlspecialchars' => true],
            'wysiwyg' => ['height' => 100, 'inline' => true, 'enterMode' => 1, 'customConfig' => '/admin/js/cke_config_document_loader.js'],
            'image' => ['hidetext' => true, 'reload' => false, 'thumbnail' => 'AdminFallback', 'uploadPath' => '/dokumenty'],
            'multihref' => ['uploadPath' => '/dokumenty', 'thumbnail' => null],
            'relations' => ['uploadPath' => '/dokumenty', 'thumbnail' => null],
            'href' => [],
            'relation' => [],
            'checkbox' => ['reload' => false],
            'date' => ['format' => 'd.m.Y', 'startDay' => 1],
            'select' => ['reload' => false, 'store' => [['dummy', 'dummy']]],
            'table' => ['defaults' => ['cols' => 1, 'rows' => 1, 'data' => [['dummy']]]],
            'multiselect' => ['store' => [['dummy', 'dummy']]],
        ];

        if ($type === 'image' && $context['editmode']) {
            $defaultConfigurations['image']['width'] = 0;
        }

        $options = array_merge($defaultConfigurations[$type] ?? [], $options);

        return $this->editableRenderer->render($context['document'], $type, $name, $options, $context['editmode']);
    }
}
