<?php declare(strict_types=1);

namespace App\Command;

use App\Elasticsearch\Service\IndexService;
use Elastic\Elasticsearch\Exception\AuthenticationException;
use Elastic\Elasticsearch\Exception\ClientResponseException;
use Elastic\Elasticsearch\Exception\MissingParameterException;
use Elastic\Elasticsearch\Exception\ServerResponseException;
use Pimcore\Console\AbstractCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class ElasticReindexCommand extends AbstractCommand
{
    public function __construct(
        private readonly IndexService $indexService,
        ?string                       $name = null,
    ) {
        parent::__construct($name);
    }

    protected function configure(): void
    {
        $this
            ->setName('elasticsearch:reindex')
            ->setAliases(['elastic:reindex'])
            ->setDescription('Reindex Elasticsearch')
            ->addOption(
                'dataObjects',
                'o',
                InputOption::VALUE_OPTIONAL,
                'Select specific data object indices with a comma separated class names',
            )
            ->addOption(
                'fulltext',
                'f',
                InputOption::VALUE_OPTIONAL,
                'Select specific fulltext indices with a comma separated class names',
            );
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws ServerResponseException
     * @throws MissingParameterException
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->indexService->setOutput($this->io)->updateIndicesThenReindex(
            $this->getOptionAsArray($input, 'dataObjects'),
            $this->getOptionAsArray($input, 'fulltext'),
        );

        return static::SUCCESS;
    }

    /**
     * @return array<string>
     */
    private function getOptionAsArray(InputInterface $input, string $optionName): array
    {
        $option = $input->getOption($optionName);

        return !empty($option) ? explode(',', $option) : [];
    }
}
