<?php declare(strict_types=1);

namespace App\Controller;

use App\Elasticsearch\Service\IndexService;
use App\Form\NewsletterFormType;
use App\Service\AjaxResponseService;
use App\Service\LanguageService;
use App\Tool\Utils;
use Pimcore\Bundle\ApplicationLoggerBundle\ApplicationLogger;
use Pimcore\Cache\RuntimeCache;
use Pimcore\Controller\FrontendController;
use Pimcore\Controller\KernelControllerEventInterface;
use Pimcore\Model\Document;
use Pimcore\Model\Site;
use Pimcore\Translation\Translator;
use Pimcore\Twig\Extension\Templating\HeadLink;
use Pimcore\Twig\Extension\Templating\HeadMeta;
use Pimcore\Twig\Extension\Templating\HeadScript;
use Pimcore\Twig\Extension\Templating\HeadTitle;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpKernel\Event\ControllerEvent;
use Twig\Environment;

class BaseController extends FrontendController implements KernelControllerEventInterface
{
    use Helper\ControllerTrait;

    protected FormInterface $newsletterForm;

    protected Request $request;
    protected string $language;
    protected string $currentUrl;
    protected string $cacheKeySuffix;

    public function __construct(
        protected Environment         $twig,
        protected Translator          $translator,
        protected LanguageService     $languageService,
        protected AjaxResponseService $ajaxResponseService,
        protected HeadScript          $headScript,
        protected HeadLink            $headLink,
        protected HeadMeta            $headMeta,
        protected HeadTitle           $headTitle,
        protected ApplicationLogger   $applicationLogger,
        protected IndexService        $indexService,
    ) {
    }

    /**
     * @throws \Exception
     */
    public function onKernelControllerEvent(ControllerEvent $event): void
    {
        \Pimcore\Cache::setForceImmediateWrite(true);

        $this->init($event->getRequest());

        $this->twig->addGlobal('document', $this->document);
        $this->twig->addGlobal('language', $this->language);
        $this->twig->addGlobal('currentUrl', $this->currentUrl);

        $this->twig->addGlobal('cacheKey', $this->cacheKeySuffix);
        $this->twig->addGlobal('breadcrumbsCacheKey', '_breadcrumbs_'.$this->cacheKeySuffix);
        $this->twig->addGlobal('metaCacheKey', '_meta_'.$this->cacheKeySuffix);
        $this->twig->addGlobal('languageSwitchCacheKey', '_language_switch_'.$this->cacheKeySuffix);
        $this->twig->addGlobal('mainNavCacheKey', '_main_nav_'.$this->cacheKeySuffix);
        $this->twig->addGlobal('socialNavCacheKey', '_social_nav_' . $this->cacheKeySuffix);

        $addToBreadcrumbs = new \stdClass();
        $addToBreadcrumbs->pages = &$this->addToBreadcrumbs;
        $this->twig->addGlobal('addToBreadcrumbs', $addToBreadcrumbs);

        $languageSwitch = new \stdClass();
        $languageSwitch->pages = &$this->languageSwitch;
        $this->twig->addGlobal('languageSwitch', $languageSwitch);

        // Other variables.
        $this->twig->addGlobal('isFrontendRequestByAdmin', \Pimcore\Tool::isFrontendRequestByAdmin());

        //global forms
        if ($this->document instanceof Document\Page) {
            $this->newsletterForm = $this->createForm(NewsletterFormType::class);
            $this->twig->addGlobal('newsletterForm', $this->newsletterForm->createView());
        }

        // Flash messages.
        if ($this->request->hasSession()) {
            $session = $this->request->getSession();
            $sessionFlashMessages = $session instanceof Session ? $session->getFlashBag()->all() : [];
            $currentRequestFlashMessages = RuntimeCache::isRegistered('currentRequestFlashMessages') ? RuntimeCache::get('currentRequestFlashMessages') : [];
            $this->twig->addGlobal('flashMessages', array_merge($sessionFlashMessages, $currentRequestFlashMessages));
        }

        $rootDoc = Utils::getTranslatedDocumentByPath('/' . $this->languageService->getDefaultLanguage(), $this->language)
                ?: Utils::getTranslatedDocumentByPath('/', $this->language);
        $this->twig->addGlobal('rootDocument', $rootDoc);
    }

    /**
     * @throws \Exception
     */
    private function init(Request $request): void
    {
        $this->request = $request;

        try {
            $this->language = $this->languageService->getLanguageFromLocale($this->request->getLocale());
        } catch (\Throwable) {
            $this->language = $this->languageService->getDefaultLanguage();
        }

        $this->currentUrl = $this->request->getPathInfo();

        $cacheKeySuffix = $this->language . '_' . $this->currentUrl;

        if ($this->isSiteRequest()) {
            $cacheKeySuffix = Site::getCurrentSite()->getId() . '_' . $cacheKeySuffix;
        }

        $this->cacheKeySuffix = Utils::toCacheKey($cacheKeySuffix);
    }
}
