<?php declare(strict_types=1);

namespace App\Controller;

use App\Form\ContactFormType;
use App\Service\MailService;
use Pimcore\File;
use Pimcore\Logger;
use Pimcore\Model\DataObject;
use Pimcore\Model\DataObject\ContactForm;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class ContentController extends BaseController
{
    /*						DOCUMENT ROUTED ACTIONS							*/

    public function templateSimpleAction(): Response
    {
        return $this->render('content/templateSimple.html.twig');
    }

    public function simpleMailAction(): Response
    {
        return $this->render('content/simpleMail.html.twig');
    }

    public function sitemapAction(): Response
    {
        return $this->render('content/sitemap.html.twig');
    }

    public function contactUsAction(Request $request, MailService $mailService): ?Response
    {
        $form = $this->createForm(ContactFormType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            if ($this->handleContactForm($form, $mailService, ContactFormType::FIELD_HONEYPOT)) {
                return $this->gotoUrl($this->docUrl('/kontakt'));
            }
        }

        return $this->render('content/contactUs.html.twig', [
            'contactForm' => $form->createView(),
        ]);
    }

    /* 						STATIC-ROUTE ROUTED ACTIONS						 */

    /*				VIEW-LESS ACTIONS (ALWAYS REDIRECT)					*/

    /*							FORM HANDLERS								*/

    private function handleContactForm(FormInterface $form, MailService $mailService, string $honeypot): bool
    {
        if ($form->isValid()) {
            $data = $form->getData();

            // test honeypot field to not be empty
            if ($honeypot && !empty($data[$honeypot])) {
                $this->addErrorMsg($this->translate('msg_form_invalid'), true);

                return false;
            }

            $data['lang'] = $this->language;

            $contactForm = ContactForm::create($data);
            $contactForm->setCreationDate(time());
            $key = File::getValidFilename($data['fullName'].'_'.date('Y-m-d-H:i:s', time()));
            $contactForm->setKey($key);
            $contactForm->setParentId(DataObject\Folder::getByPath('/kontakt/kontaktni-formulare')?->getId());
            $contactForm->setPublished(true);

            try {
                $contactForm->save();

                //send email to admin
                $mailService->sendMail('/notifikace/pro-administratora-novy-kontaktni-formular', $this->language, null, null, [
                    'deeplink' => $this->getRequest()->getSchemeAndHttpHost().'/admin/login/deeplink?object_'.$contactForm->getId().'_object',
                    'form' => $contactForm,
                ]);

                $this->addSuccessMsg($this->translate('msg_contact_form_success'));

                return true;
            } catch (\Exception $e) {
                Logger::error('CONTACT FORM - '.$e->getMessage()."\n".$e->getTraceAsString());
                $this->addErrorMsg($this->translate('msg_unknown_error'));
            }
        } else {
            $this->addErrorMsg($this->translate('msg_form_invalid'));
        }

        return false;
    }
}
