<?php declare(strict_types=1);

namespace App\Controller;

use App\Service\ElasticSearch;
use App\Service\FeedService;
use App\Tool;
use App\Tool\Utils;
use Elastic\Elasticsearch\Exception\AuthenticationException;
use Elastic\Elasticsearch\Exception\ClientResponseException;
use Elastic\Elasticsearch\Exception\MissingParameterException;
use Elastic\Elasticsearch\Exception\ServerResponseException;
use Pimcore\Model\Asset;
use Pimcore\Model\Document;
use Pimcore\Tool\Admin;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;

class DefaultController extends BaseController
{
    public function emptyPageAction(): Response
    {
        $cannotRedirect = $this->editmode || $this->isAdminFrontendRequest() || $this->isIndexingRequest();

        if ($cannotRedirect) {
            return $this->render('default/emptyPage.html.twig');
        }

        if (!$this->document instanceof Document) {
            throw $this->createNotFoundException($this->translate('system_page_not_found'));
        }

        $documentList = $this->document->getChildren();
        $documents = $documentList->getDocuments();

        if (empty($documents)) {
            throw $this->createNotFoundException($this->translate('system_page_not_found'));
        }

        $nextDocument = current($documents);

        return $this->redirect($nextDocument->getFullPath());
    }

    public function indexAction(): Response
    {
        return $this->render('default/index.html.twig');
    }

    public function defaultAction(): Response
    {
        return $this->render('default/default.html.twig');
    }

    public function errorPageAction(): Response
    {
        return $this->render('default/errorPage.html.twig');
    }

    /* 						STATIC-ROUTE ROUTED ACTIONS						 */

    /*							VIEW-LESS ACTIONS							*/

    #[Route('/admin/plugin/backend/is-elastic-search-enabled')]
    public function isElasticSearchEnabledAction(): JsonResponse
    {
        return new JsonResponse([
            'status' => true,
            'ese' => (Utils::elasticSearchEnabled() && Admin::getCurrentUser()?->isAdmin()),
        ]);
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws ServerResponseException
     * @throws MissingParameterException
     */
    #[Route('/plugin/backend/reindex-elastic-search')]
    public function reindexElasticAction(): JsonResponse
    {
        $responseData = ['status' => false];

        if (Utils::elasticSearchEnabled() && Admin::getCurrentUser()?->isAdmin()) {

            $this->indexService->updateIndicesThenReindex();
            $responseData = ['status' => true];
        }

        return new JsonResponse($responseData);
    }

    #[Route('/api/search/{query}')]
    public function searchAction(ElasticSearch $elastic, string $query): JsonResponse
    {
        $responseData = [];
        if (Utils::elasticSearchEnabled()) {
            $responseData = $elastic->fulltextSearch(['query' => $query]);
            //$responseData = $elastic->getNews(['query' => $query]);
        }

        $jsonResponse = new JsonResponse($responseData);

        return $jsonResponse;
    }

    public function downloadAssetAction(int $id): RedirectResponse|BinaryFileResponse|null
    {
        $asset = Asset::getById($id);
        if (!$asset) {
            $this->addErrorMsg($this->translate('msg_invalid_request'));

            return $this->gotoUrl($this->docUrl('/'));
        }

        $assetFilename = $asset->getFilename() ?: $asset->getFullPath();
        $filename = substr(basename($assetFilename), (stripos($assetFilename, '_')) ? stripos($assetFilename, '_') + 1 : 0);

        $fileResponse = new BinaryFileResponse($asset->getRealFullPath());
        $fileResponse->setContentDisposition('attachment', $filename);

        return $fileResponse;
    }

    public function sitemapXmlAction(FeedService $feedService): Response
    {
        // since v10.5 use of @PrestaSitemapBundle is possible
        $xml = $feedService->generateSitemapXml();

        $xmlResponse = new Response($xml);
        $xmlResponse->headers->set('Content-Type', 'application/xml');

        return $xmlResponse;
    }
}
