<?php declare(strict_types=1);

namespace App\Controller;

use App\Form\NewsletterFormType;
use App\Service\NewsletterService;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

class NewsletterController extends BaseController
{
    /*						DOCUMENT ROUTED ACTIONS							*/

    /*				VIEW-LESS ACTIONS (ALWAYS REDIRECT)					*/

    public function newsletterSubscribeAction(Request $request, NewsletterService $newsletterService): Response
    {
        $this->newsletterForm->handleRequest($request);
        if ($this->newsletterForm->isSubmitted()) {
            if ($this->handleNewsletterForm($this->newsletterForm, $newsletterService, true)) {
                $this->newsletterForm = $this->createForm(NewsletterFormType::class);

                $this->ajaxResponseService->addElement('#newsletter', 'snippet/newsletter.html.twig', [
                    'newsletterSent' => true,
                    'newsletterForm' => $this->newsletterForm->createView()
                ]);
            }
        }

        return $this->ajaxResponseService->addFlashMessages()->getResponse();
    }

    public function newsletterConfirmAction(NewsletterService $newsletterService, string $code): ?Response
    {
        if ($email = $newsletterService->confirm($code)) {
            $this->addSuccessMsg(sprintf($this->translate('msg_newsletter_subscribed'), $email));
        } elseif (false === $email) {
            $this->addInfoMsg($this->translate('msg_newsletter_code_invalid'));
        } else {
            $this->addErrorMsg($this->translate('msg_unknown_error'));
        }

        return $this->gotoUrl($this->getPreviousUrl());
    }

    public function newsletterUnsubscribeAction(NewsletterService $newsletterService, string $code): ?Response
    {
        if ($email  = $newsletterService->unsubscribe($code)) {
            $this->addSuccessMsg(sprintf($this->translate('msg_newsletter_unsubscribed'), $email));
        } elseif (false === $email) {
            $this->addInfoMsg($this->translate('msg_newsletter_code_invalid'));
        } else {
            $this->addErrorMsg($this->translate('msg_unknown_error'));
        }

        return $this->gotoUrl($this->getPreviousUrl());
    }

    /*							FORM HANDLERS								*/

    protected function handleNewsletterForm(FormInterface $form, NewsletterService $newsletterService, bool $flashForCurrentRequest = false): bool
    {
        if (!$form->isValid()) {
            $this->addErrorMsg($this->translate('msg_form_invalid'), $flashForCurrentRequest);

            return false;
        }

        $data = $form->getData();

        // antispam hidden field, should be empty
        if (!empty($data[NewsletterFormType::FIELD_HONEYPOT])) {
            $this->addErrorMsg($this->translate('msg_unknown_error'), $flashForCurrentRequest);

            return false;
        }

        $data['lang'] = $this->language;

        if ($result = $newsletterService->subscribe($data)) {
            $this->addSuccessMsg(sprintf($this->translate('msg_newsletter_registered'), $data['email']), $flashForCurrentRequest);
        } else {
            $this->addErrorMsg($this->translate('msg_unknown_error'), $flashForCurrentRequest);
        }

        return $result;
    }
}
