<?php declare(strict_types=1);

namespace App\Elasticsearch\EventSubscriber;

use App\Elasticsearch\Contract\ElasticsearchDataObjectInterface;
use App\Elasticsearch\Message\EraseDataObject;
use App\Elasticsearch\Message\IndexDataObject;
use App\Service\MessageBufferService;
use Pimcore\Event\DataObjectEvents;
use Pimcore\Event\Model\DataObjectEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

readonly class ElasticsearchDataObjectSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private MessageBufferService $messageBufferService,
    ) {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            DataObjectEvents::POST_UPDATE => [
                ['index'],
            ],
            DataObjectEvents::POST_UPDATE_FAILURE => [
                ['dropBufferedIndex'],
            ],
            DataObjectEvents::PRE_DELETE => [
                ['erase'],
            ],
            DataObjectEvents::POST_DELETE_FAILURE => [
                ['dropBufferedErase'],
            ],
        ];
    }

    public function index(DataObjectEvent $event): void
    {
        $dataObject = $event->getObject();
        $id = $dataObject->getId();

        if (empty($id)) {
            return;
        }

        if ($dataObject instanceof ElasticsearchDataObjectInterface) {
            $this->messageBufferService->push(new IndexDataObject($id));
        }
    }

    public function dropBufferedIndex(DataObjectEvent $event): void
    {
        $this->messageBufferService->drop(IndexDataObject::class);
    }
    
    public function erase(DataObjectEvent $event): void
    {
        $dataObject = $event->getObject();
        $id = $dataObject->getId();

        if (empty($id)) {
            return;
        }

        if ($dataObject instanceof ElasticsearchDataObjectInterface) {
            $this->messageBufferService->push(new EraseDataObject($id));
        }
    }

    public function dropBufferedErase(DataObjectEvent $event): void
    {
        $this->messageBufferService->drop(EraseDataObject::class);
    }
}
