<?php declare(strict_types=1);

namespace App\Elasticsearch\EventSubscriber;

use App\Elasticsearch\Contract\ElasticsearchFulltextInterface;
use App\Elasticsearch\Message\EraseFulltext;
use App\Elasticsearch\Message\IndexFulltext;
use App\Enum\Element\Type;
use App\Service\MessageBufferService;
use Pimcore\Event\DataObjectEvents;
use Pimcore\Event\DocumentEvents;
use Pimcore\Event\Model\DataObjectEvent;
use Pimcore\Event\Model\DocumentEvent;
use Pimcore\Model\Element\ElementInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

readonly class ElasticsearchFulltextSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private MessageBufferService $messageBufferService,
    ) {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            DocumentEvents::POST_UPDATE => [
                ['indexDocument'],
            ],
            DataObjectEvents::POST_UPDATE => [
                ['indexDataObject'],
            ],
            DocumentEvents::POST_UPDATE_FAILURE => [
                ['dropBufferedIndex'],
            ],
            DataObjectEvents::POST_UPDATE_FAILURE => [
                ['dropBufferedIndex'],
            ],
            DocumentEvents::PRE_DELETE => [
                ['eraseDocument'],
            ],
            DataObjectEvents::PRE_DELETE => [
                ['eraseDataObject'],
            ],
            DocumentEvents::POST_DELETE_FAILURE => [
                ['dropBufferedErase'],
            ],
            DataObjectEvents::POST_DELETE_FAILURE => [
                ['dropBufferedErase'],
            ],
        ];
    }

    public function indexDocument(DocumentEvent $event): void
    {
        $this->index($event->getDocument(), Type::DOCUMENT);
    }

    public function indexDataObject(DataObjectEvent $event): void
    {
        $this->index($event->getObject(), Type::DATA_OBJECT);
    }

    public function dropIndexMessage(DocumentEvent $event): void
    {
        $this->messageBufferService->drop(IndexFulltext::class);
    }

    public function eraseDocument(DocumentEvent $event): void
    {
        $this->erase($event->getDocument(), Type::DOCUMENT);
    }

    public function eraseDataObject(DataObjectEvent $event): void
    {
        $this->erase($event->getObject(), Type::DATA_OBJECT);
    }

    public function dropBufferedErase(DataObjectEvent $event): void
    {
        $this->messageBufferService->drop(EraseFulltext::class);
    }

    private function index(ElementInterface $element, Type $type): void
    {
        $id = $element->getId();

        if (empty($id)) {
            return;
        }

        if ($element instanceof ElasticsearchFulltextInterface) {
            $this->messageBufferService->push(new IndexFulltext($id, $type));
        }
    }

    private function erase(ElementInterface $element, Type $type): void
    {
        $id = $element->getId();

        if (empty($id)) {
            return;
        }

        if ($element instanceof ElasticsearchFulltextInterface) {
            $this->messageBufferService->push(new EraseFulltext($id, $type));
        }
    }
}
