<?php declare(strict_types=1);

namespace App\Elasticsearch\MessageHandler;

use App\Elasticsearch\Contract\ElasticsearchFulltextInterface;
use App\Elasticsearch\Message\IndexFulltext;
use App\Elasticsearch\Service\IndexService;
use App\Enum\Element\Type;
use Pimcore\Bundle\ApplicationLoggerBundle\ApplicationLogger;
use Pimcore\Model\DataObject;
use Pimcore\Model\Document;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;

#[AsMessageHandler]
readonly class IndexFulltextHandler
{
    public function __construct(
        private IndexService      $indexService,
        private ApplicationLogger $applicationLogger,
    ) {
    }

    public function __invoke(IndexFulltext $message): void
    {
        $type = $message->getType();

        if (!in_array($type, [Type::DOCUMENT, Type::DATA_OBJECT])) {
            return;
        }

        $element = null;
        $id = $message->getId();

        if (Type::DOCUMENT === $type) {
            $element = Document::getById($id);
        }

        if (Type::DATA_OBJECT === $type) {
            $element = DataObject::getById($id);
        }

        if (!$element instanceof ElasticsearchFulltextInterface) {
            return;
        }

        try {
            $this->indexService->indexFulltext($element);
        } catch (\Throwable $e) {
            $this->applicationLogger->logException(
                sprintf('Cannot index fulltext for %s ID %d', $type->value, $id),
                $e,
                component: __METHOD__ . ':' . __LINE__,
            );
        }
    }
}
