<?php declare(strict_types=1);

namespace App\Elasticsearch\Service;

use App\Elasticsearch\Contract\ElasticsearchDataObjectInterface;
use App\Elasticsearch\Contract\ElasticsearchElementInterface;
use App\Elasticsearch\Trait\OutputAwareTrait;
use App\Model\Document;
use Elastic\Elasticsearch\Client;
use Elastic\Elasticsearch\ClientBuilder;
use Elastic\Elasticsearch\Exception\AuthenticationException;
use Elastic\Elasticsearch\Exception\ClientResponseException;
use Elastic\Elasticsearch\Exception\MissingParameterException;
use Elastic\Elasticsearch\Exception\ServerResponseException;
use Pimcore\Cache\RuntimeCache;
use Pimcore\Model\DataObject\ClassDefinition;

class ClientService
{
    use OutputAwareTrait;

    private ?Client $client = null;

    /**
     * @var ?array<string, array<string, mixed>>
     */
    private ?array $configuration = null;

    public function __construct(
        private readonly ConfigurationService $configurationService,
        private readonly string               $host,
        private readonly string               $aliasPrefix,
    ) {
    }

    public function getElementMapping(ElasticsearchElementInterface $element): array
    {
        $elementMapping = $element->getElasticsearchMapping();

        if (!empty($elementMapping)) {
            return $elementMapping;
        }

        return $this->getElementMappingByClassName($element->getClassNameShort());
    }

    /**
     * @return array<string, array<string, mixed>>
     *
     * @throws \RuntimeException
     */
    public function getElementMappingByClassName(string $className): array
    {
        $configuration = $this->getConfiguration();

        if (is_null($configuration)) {
            $defaultConfiguration = $this->configurationService->getDefaultConfiguration();

            $this->setConfiguration($defaultConfiguration);
        }

        $mappingKey = strtolower($className);

        if (empty($configuration[$mappingKey])) {
            $mappingKey = $this->configurationService->getKeyDefaultMapping();
        }

        if (empty($configuration[$mappingKey])) {
            throw new \RuntimeException(sprintf('No mapping for "%s"', $mappingKey));
        }

        return $configuration[$mappingKey];
    }

    /**
     * @throws AuthenticationException
     */
    public function getClient(): Client
    {
        if (!$this->client instanceof Client) {
            $hosts = [$this->getHost()];

            $this->client = ClientBuilder::create()
                ->setHosts($hosts)
                ->build();
        }

        return $this->client;
    }

    public function setIsUpdateRequest(?bool $isUpdateRequest): void
    {
        RuntimeCache::set($this->getIsUpdateRequestCacheKey(), $isUpdateRequest);
    }

    public function isUpdateRequest(): bool
    {
        return RuntimeCache::isRegistered($this->getIsUpdateRequestCacheKey());
    }

    public function isNotUpdateRequest(): bool
    {
        return !$this->isUpdateRequest();
    }

    public function setNewIndexPrefix(?string $prefix): void
    {
        RuntimeCache::set($this->getNewIndexPrefixCacheKey(), $prefix);
    }

    public function getIndexPrefixCacheKey(): string
    {
        return 'elasticsearch_new_index_prefix';
    }

    public function getDocumentFulltextIndexKey(): string
    {
        return 'document';
    }

    public function getDataObjectFulltextIndexKey(): string
    {
        return 'object';
    }

    public function getHost(): string
    {
        return $this->host;
    }

    public function getAliasPrefix(): string
    {
        return $this->aliasPrefix;
    }

    /**
     * @return ?array<string, array<string, mixed>>
     */
    public function getConfiguration(): ?array
    {
        return $this->configuration;
    }

    /**
     * @param ?array<string, array<string, mixed>> $configuration
     */
    public function setConfiguration(?array $configuration): void
    {
        $this->configuration = $configuration;
    }

    /**
     * @return array<string, string>
     *
     * @throws ClientResponseException
     * @throws ServerResponseException
     * @throws MissingParameterException
     * @throws AuthenticationException
     */
    public function createObjectIndices(): array
    {
        $indices = [];
        $classList = new ClassDefinition\Listing();

        foreach ($classList->getClasses() as $class) {
            $className = '\\App\\Model\\' . $class->getName();

            if (!class_exists($className) || !in_array(ElasticsearchDataObjectInterface::class, class_implements($className))) {
                continue;
            }

            $objectMapping = $this->getElementMappingByClassName((string)$class->getName());

            foreach ($objectMapping['meta']['languages'] as $language) {
                $mapping = [
                    '_source' => ['enabled' => true],
                    'dynamic' => false,
                    'properties' => $objectMapping['properties'],
                ];

                if (!empty($objectMapping['meta']['fulltextFields'])) {
                    $this->addFulltextFieldsToMapping($mapping, $objectMapping['meta']['fulltextFields'], $language);
                }

                $suffix = strtolower((string)$class->getName()) . '_' . $language;
                $indexName = $this->getIndexName($suffix);

                $this->getClient()->indices()->create([
                    'index' => $indexName,
                    'body' => [
                        'settings' => ['analysis' => $this->configurationService->getAnalysisConfiguration()],
                        'mappings' => $mapping,
                    ]
                ]);

                $indices[$suffix] = $indexName;
                $this->output?->success(sprintf('Create "%s"', $indexName));
            }
        }

        return $indices;
    }

    /**
     * @return array<string, string>
     *
     * @throws ClientResponseException
     * @throws ServerResponseException
     * @throws MissingParameterException
     * @throws AuthenticationException
     */
    public function createFulltextIndices(): array
    {
        $indices = [];
        $fulltextMapping = $this->getElementMappingByClassName($this->configurationService->getKeyFulltextMapping());

        foreach ($fulltextMapping['meta']['languages'] as $languages) {
            $mapping = [
                '_source' => ['enabled' => true],
                'dynamic' => false,
                'properties' => $fulltextMapping['properties'],
            ];

            $this->addFulltextFieldsToMapping($mapping, $fulltextMapping['meta']['fulltextFields'], $languages);

            foreach ([$this->getDocumentFulltextIndexKey(), $this->getDataObjectFulltextIndexKey()] as $key) {
                $suffix = $key . '_' . $languages;
                $indexName = $this->getIndexName($suffix);

                $this->getClient()->indices()->create([
                    'index' => $indexName,
                    'body' => [
                        'settings' => ['analysis' => $this->configurationService->getAnalysisConfiguration()],
                        'mappings' => $mapping,
                    ]
                ]);

                $indices[$suffix] = $indexName;
                $this->output?->success(sprintf('[%s] ✔', $indexName));
            }
        }

        return $indices;
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws ServerResponseException
     * @throws \Exception
     */
    public function getIndexName(string $suffix, bool $forceCurrent = false): string
    {
        $aliasPrefix = $this->getAliasPrefix();
        $cacheKey = $this->getIndexPrefixCacheKey();

        if (!$forceCurrent && RuntimeCache::isRegistered($cacheKey)) {
            $indexPrefix = RuntimeCache::get($cacheKey);

            return $aliasPrefix . '_' . $indexPrefix . '_' . $suffix;
        }

        $aliasName = $aliasPrefix . '_' . $suffix;
        $client = $this->getClient();

        $alias = $client->indices()->getAlias(['name' => $aliasName]);

        // TODO: Test & fix!
        return key($alias);
    }

    public function getIndexSuffix(
        ElasticsearchElementInterface $element,
        string $language,
        bool $isForGroup = false,
    ): string {
        $separator = $isForGroup ? '_group_' : '_';

        return strtolower($element->getClassNameShort() . $separator . $language);
    }


    public function getIndexFulltextSuffix(
        ElasticsearchElementInterface $element,
        string $language,
    ): string {
        $isDocument = $element instanceof Document\Page;

        $key = $isDocument ?
            $this->getDocumentFulltextIndexKey() :
            $this->getDataObjectFulltextIndexKey();

        return $key . '_' . $language;
    }

    /**
     * @throws ServerResponseException
     * @throws ClientResponseException
     * @throws AuthenticationException
     */
    public function copyIndices(string $type, string $indexKey): void
    {
        $this->output?->info(sprintf('Copy "%s"', $type));

        $fulltextMapping = $this->getElementMappingByClassName('fulltext_mapping');

        foreach ($fulltextMapping['meta']['languages'] as $language) {
            $suffix = $indexKey . '_' . $language;
            $indexName = $this->getIndexName($suffix, true);

            $indexCopyName = $this->getIndexName($suffix);
            $this->copyIndex($indexName, $indexCopyName);
        }

        $this->output?->success('Done');
    }

    private function getIsUpdateRequestCacheKey(): string
    {
        return 'elasticsearch_is_update_request';
    }

    private function getNewIndexPrefixCacheKey(): string
    {
        return 'elasticsearch_new_index_prefix';
    }

    /**
     * @param array<string, array<string, mixed>> $mapping
     * @param array<string> $fieldNames
     */
    private function addFulltextFieldsToMapping(
        array  &$mapping,
        array  $fieldNames,
        string $language,
    ): void {
        foreach ($fieldNames as $fieldName) {
            $mapping['properties'][$fieldName] = $mapping['properties'][$fieldName] ?? [
                'type' => 'text',
                'term_vector' => 'with_positions_offsets',
            ];

            $mapping['properties'][$fieldName]['fields']['folded'] = $mapping['properties'][$fieldName]['fields']['folded'] ?? [
                'type' => 'text',
                'term_vector' => 'with_positions_offsets',
            ];

            $analyzer = in_array($language, ['cs', 'sk']) ? $language . '_hunspell' : 'standard';

            $mapping['properties'][$fieldName]['analyzer'] = $analyzer;
            $mapping['properties'][$fieldName]['search_analyzer'] = $analyzer;

            $icuAnalyzer = in_array($language, ['cs', 'sk']) ? $language . '_icu_analyzer' : 'standard';

            $mapping['properties'][$fieldName]['fields']['folded']['analyzer'] = $icuAnalyzer;
            $mapping['properties'][$fieldName]['fields']['folded']['search_analyzer'] = $icuAnalyzer;
        }
    }

    /**
     * @throws ServerResponseException
     * @throws ClientResponseException
     * @throws AuthenticationException
     */
    private function copyIndex(string $indexName, string $indexCopyName, ?string $type = null): void
    {
        $body = [
            'source' => ['index' => $indexName],
            'dest' => ['index' => $indexCopyName],
        ];

        if (is_null($type)) {
            $body['source']['query'] = ['match' => ['type' => $type]];
        }

        $this->getClient()->reindex(['body' => $body, 'refresh' => true]);
    }
}
