<?php declare(strict_types=1);

namespace App\Elasticsearch\Service;

use App\Elasticsearch\Trait\OutputAwareTrait;
use App\Service\LanguageService;
use Elastic\Elasticsearch\Client;
use Elastic\Elasticsearch\ClientBuilder;
use Elastic\Elasticsearch\Exception\AuthenticationException;
use Elastic\Elasticsearch\Exception\ClientResponseException;
use Elastic\Elasticsearch\Exception\MissingParameterException;
use Elastic\Elasticsearch\Exception\ServerResponseException;
use Pimcore\Cache\RuntimeCache;
use Pimcore\Db;
use Pimcore\Tool;

readonly class ConfigurationService
{
    public function __construct(
        private LanguageService $languageService,
    ) {
    }

    public function getDefaultConfiguration(): array
    {
        $baseMapping = $this->getBaseConfiguration();
        $analysisMapping = $this->getAnalysisConfiguration();
        $validLanguages = $this->languageService->getValidLanguages();

        return [
            'analysis' => $analysisMapping,
            $this->getKeyDefaultMapping() => [
                'meta' => [
                    'languages' => $validLanguages,
                ],
                'properties' => $baseMapping,
            ],
            $this->getKeyFulltextMapping() => [
                'meta' => [
                    'languages' => $validLanguages,
                    'fulltextFields' => ['title', 'description', 'content'],
                ],
                'properties' => array_merge($baseMapping, [
                    'path' => ['type' => 'text'],
                ]),
            ],
            'news' => [
                'meta' => [
                    'languages' => $validLanguages,
                    'fulltextFields' => ['name'],
                ],
                'properties' => array_merge($baseMapping, [
                    'url' => ['type' => 'keyword'],
                ]),
            ],
        ];
    }

    public function getKeyDefaultMapping(): string
    {
        return 'default_mapping';
    }

    public function getKeyFulltextMapping(): string
    {
        return 'fulltext_mapping';
    }

    /**
     * @return array<string, array<string, mixed>>
     */
    public function getAnalysisConfiguration(): array
    {
        return [
            'analyzer' => [
                'pathIndexAnalyzer' => [
                    'type' => 'custom',
                    'tokenizer' => 'pathIndexTokenizer',
                ],
                'cs_hunspell' => [
                    'type' => 'custom',
                    'tokenizer' => 'standard',
                    'filter' => ['lowercase', 'stopwords_CZ', 'cs_CZ', 'remove_duplicities'],
                ],
                'sk_hunspell' => [
                    'type' => 'custom',
                    'tokenizer' => 'standard',
                    'filter' => ['lowercase', 'stopwords_SK', 'sk_SK', 'remove_duplicities'],
                ],
                'cs_icu_analyzer' => [
                    'type' => 'custom',
                    'tokenizer' => 'standard',
                    'filter' => ['stopwords_CZ', 'icu_folding', 'remove_duplicities'],
                ],
                'cs_icu_analyzer_sort' => [
                    'type' => 'custom',
                    'tokenizer' => 'keyword',
                    'filter' => ['lowercase', 'cs_icu_collation'],
                ],
                'sk_icu_analyzer' => [
                    'type' => 'custom',
                    'tokenizer' => 'standard',
                    'filter' => ['stopwords_SK', 'icu_folding', 'remove_duplicities'],
                ],
            ],
            'filter' => [
                'stopwords_CZ' => [
                    'type' => 'stop',
                    'stopwords' => ['právě', 'že', '_czech_'],
                    'ignore_case' => true,
                ],
                'stopwords_SK' => [
                    'type' => 'stop',
                    'stopwords' => ['a', 'aby', 'aj', 'ako', 'ale', 'alebo', 'ani', 'áno', 'asi', 'až', 'bez', 'buď', 'by', 'cez', 'či', 'čo', 'ešte', 'ho', 'i', 'iba', 'ich', 'ja', 'je', 'jeho', 'jej', 'ju', 'k', 'kam', 'kde', 'keď', 'kto', 'ku', 'menej', 'mi', 'moja', 'moje', 'môj', 'my', 'nad', 'nám', 'než', 'nič', 'nie', 'o', 'od', 'on', 'on', 'ona', 'ona', 'oni', 'ono', 'po', 'pod', 'podľa', 'pokiaľ', 'potom', 'práve', 'prečo', 'pred', 'preto', 'pretože', 'pri', 's', 'sa', 'si', 'sme', 'so', 'som', 'späť', 'ste', 'sú', 'sú', 'ta', 'tá', 'tak', 'tak', 'takže', 'tam', 'tam', 'táto', 'teda', 'ten', 'tento', 'tieto', 'tiež', 'to', 'to', 'toho', 'tom', 'tomto', 'toto', 'tu', 'túto', 'ty', 'tým', 'týmto', 'už', 'v', 'vám', 'viac', 'vo', 'však', 'vy', 'z', 'za', 'zo'],
                    'ignore_case' => true,
                ],
                'cs_CZ' => [
                    'type' => 'hunspell',
                    'locale' => 'cs_CZ',
                    'dedup' => true,
                    'recursion_level' => 0,
                ],
                'cs_icu_collation' => [
                    'type' => 'icu_collation',
                    'language' => 'cs',
                ],
                'sk_SK' => [
                    'type' => 'hunspell',
                    'locale' => 'sk_SK',
                    'dedup' => true,
                    'recursion_level' => 0,
                ],
                'remove_duplicities' => [
                    'type' => 'unique',
                    'only_on_same_position' => true,
                ],
            ],
            'tokenizer' => [
                'pathIndexTokenizer' => [
                    'type' => 'path_hierarchy',
                    'delimiter' => '/',
                ],
            ],
        ];
    }

    /**
     * @return array{
     *     id: array{type: string},
     *     parentId: array{type: string},
     *     key: array{type: string},
     *     type: array{type: string},
     *     published: array{type: string},
     * }
     */
    private function getBaseConfiguration(): array
    {
        return [
            'id' => ['type' => 'integer'],
            'parentId' => ['type' => 'integer'],
            'key' => ['type' => 'keyword'],
            'type' => ['type' => 'keyword'],
            'published' => ['type' => 'boolean'],
        ];
    }
}
