<?php declare(strict_types=1);

namespace App\Elasticsearch\Service;

use App\Elasticsearch\Contract\ElasticsearchFulltextInterface;
use App\Model\Document\Page;
use Pimcore\Model\Document\Service;
use Pimcore\Translation\Translator;
use simplehtmldom\HtmlDocument;
use Twig\Environment;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;

readonly class ContentService
{
    public function __construct(
        private Translator  $translator,
        private Environment $twig,
    ) {
    }

    /**
     * @throws \RuntimeException
     */
    public function renderContent(ElasticsearchFulltextInterface $element, string $language): string
    {
        if ($element instanceof Page) {
            return $this->renderPage($element, $element->elasticsearchGetFulltextContentParameters());
        }

        try {
            $content = $this->renderView(
                $element->elasticsearchGetFulltextContentTemplateName(),
                $language,
                $element->elasticsearchGetFulltextContentParameters(),
            );
        } catch (\Throwable $e) {
            throw new \RuntimeException(
                sprintf('Cannot render Twig view for ID %d', $element->getId()),
                previous: $e,
            );
        }

        return $content;
    }

    /**
     * @param array<string, mixed> $parameters
     */
    public function renderPage(Page $page, array $parameters): string
    {
        $parameters['elastic_document_indexing'] = true;

        return Service::render($page, $parameters);
    }

    /**
     * @param array<string, mixed> $parameters
     *
     * @throws RuntimeError
     * @throws SyntaxError
     * @throws LoaderError
     */
    public function renderView(
        string $templateName,
        string $language,
        array  $parameters,
    ): string {
        $this->translator->setLocale($language);

        $parameters['elastic_document_indexing'] = true;
        $parameters['editmode'] = false;
        $parameters['language'] = $language;

        return $this->transformText($this->twig->render($templateName, $parameters));
    }


    private function transformText(string $text): string
    {
        $encodeText = (string)iconv(
            'UTF-8',
            'UTF-8//IGNORE',
            mb_convert_encoding($text, 'UTF-8', 'UTF-8'),
        );

        $DOM = new HtmlDocument('<div>' . html_entity_decode($encodeText) . '</div>');

        foreach ($this->getHTMLTagsToRemoveFromText() as $tag) {
            foreach ($DOM->find($tag) as $node) {
                $node->outertext = '';
            }
        }

        $stripText = $DOM->save();
        $breakWordsText = preg_replace('/>/', '> ', $stripText);

        return (string)preg_replace('/[ \t\r\n]+/', ' ', strip_tags($breakWordsText));
    }

    /**
     * @return array<string>
     */
    private function getHTMLTagsToRemoveFromText(): array
    {
        return [
            'audio',
            'area',
            'base',
            'canvas',
            'embed',
            'head',
            'link',
            'map',
            'meta',
            'noscript',
            'object',
            'param',
            'script',
            'source',
            'style',
            'svg',
            'track',
            'video',
        ];
    }
}
