<?php declare(strict_types=1);

namespace App\Elasticsearch\Service;

use App\Elasticsearch\Contract\ElasticsearchDataObjectInterface;
use App\Elasticsearch\Contract\ElasticsearchFulltextInterface;
use App\Elasticsearch\Trait\OutputAwareTrait;
use App\Model\Document\Page;
use App\Tool\Utils;
use Elastic\Elasticsearch\Exception\AuthenticationException;
use Elastic\Elasticsearch\Exception\ClientResponseException;
use Elastic\Elasticsearch\Exception\MissingParameterException;
use Elastic\Elasticsearch\Exception\ServerResponseException;
use Pimcore\Bundle\ApplicationLoggerBundle\ApplicationLogger;
use Pimcore\Model\DataObject;
use Pimcore\Model\DataObject\ClassDefinition;
use Pimcore\Model\Document;

class IndexService
{
    use OutputAwareTrait;

    const ELASTICSEARCH_SKIP_INDEXING = 'elasticsearch_skip_indexing';

    public function __construct(
        private readonly ClientService     $clientService,
        private readonly ContentService    $contentService,
        private readonly ApplicationLogger $applicationLogger
    ) {
        $this->clientService->setOutput($this->output);
    }

    /**
     * @param ?array<string> $dataObjectClassNames
     * @param ?array<string> $fulltextClassNames
     *
     * @throws ClientResponseException
     * @throws MissingParameterException
     * @throws ServerResponseException
     * @throws AuthenticationException
     */
    public function updateIndicesThenReindex(
        ?array $dataObjectClassNames = [],
        ?array $fulltextClassNames = [],
        bool   $copyTheRest = true,
    ): void {
        $this->clientService->setIsUpdateRequest(true);
        $this->clientService->setNewIndexPrefix(date('Ymdhis'));

        $this->output?->info('Creating indices');

        $fulltextIndices = $this->clientService->createFulltextIndices();
        $objectIndices = $this->clientService->createObjectIndices();

        if (!is_null($dataObjectClassNames)) {
            $this->output?->info('Indexing data objects');
            $this->reindexDataObjects($dataObjectClassNames, $copyTheRest);
        }

        if (!is_null($fulltextClassNames)) {
            $this->output?->info('Indexing fulltext');
            $this->reindexFulltext($fulltextClassNames, $copyTheRest);
        }

        $this->output?->info('Switching aliases');
        $client = $this->clientService->getClient();

        foreach ($fulltextIndices + $objectIndices as $suffix => $index) {
            $alias = $this->clientService->getAliasPrefix() . '_' . $suffix;
            $currentIndex = $this->clientService->getIndexName($suffix, true);

            $client->indices()->putAlias(['index' => $index, 'name' => $alias]);
            $client->indices()->delete(['index' => $currentIndex]);
            $client->indices()->refresh(['index' => $index]);
        }

        $this->clientService->setIsUpdateRequest(null);
        $this->clientService->setNewIndexPrefix(null);

        $this->output?->success('Done');
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws ServerResponseException
     * @throws MissingParameterException
     */
    public function indexDataObject(ElasticsearchDataObjectInterface $dataObject): void
    {
        if ($dataObject->getProperty(self::ELASTICSEARCH_SKIP_INDEXING)) {
            return;
        }

        if (is_null($dataObject->getId())) {
            return;
        }

        $client = $this->clientService->getClient();
        $mapping = $dataObject->elasticsearchGetMappingConfiguration();

        foreach ($mapping['meta']['languages'] as $language) {
            $data = $dataObject->elasticsearchGetMappingData($language);
            
            $indexName = $this->clientService->getIndexName(
                $this->clientService->getIndexSuffix($dataObject, $language),
            );
            
            if (empty($indexName)) {
                continue;
            }

            $objectData = $data;
            unset($objectData['elasticsearchGroup']);
            
            $client->index([
                'index' => $indexName,
                'id' => $data['id'],
                'body' => $objectData,
            ]);

            if (!empty($data['elasticsearchGroup'])) {
                $this->eraseDataObject($language, current($data['elasticsearchGroup'])['id']);

                $indexGroupName = $this->clientService->getIndexName(
                    $this->clientService->getIndexSuffix($dataObject, $language, true),
                );

                foreach ($data['elasticsearchGroup'] as $key => $groupData) {
                    $client->index([
                        'index' => $indexGroupName,
                        'id' => $groupData['id'] . '-' . $key,
                        'body' => $groupData,
                    ]);
                }
            } else {
                $this->eraseDataObject($language, $data['id']);
            }

            unset($data);

            if ($this->clientService->isNotUpdateRequest()) {
                $client->indices()->refresh(['index' => $indexName]);
            }
        }
    }

    /**
     * @throws AuthenticationException
     * @throws ServerResponseException
     * @throws ClientResponseException
     */
    public function eraseDataObject(
        ElasticsearchDataObjectInterface $dataObject,
        string                           $eraseLanguage = null,
        string                           $groupId = null,
    ): void {
        $client = $this->clientService->getClient();
        $mapping = $this->clientService->getElementMapping($dataObject);

        foreach ($mapping['meta']['languages'] as $language) {
            if ($eraseLanguage && $eraseLanguage !== $language) {
                continue;
            }

            $indexName = $this->clientService->getIndexName(
                $this->clientService->getIndexSuffix($dataObject, $language),
            );

            if (empty($indexName)) {
                continue;
            }

            $indexGroupName = $this->clientService->getIndexName(
                $this->clientService->getIndexSuffix($dataObject, $language, true),
            );

            try {
                if ($groupId) {
                    $client->deleteByQuery([
                        'index' => $indexGroupName,
                        'body' => ['query' => ['term' => ['id' => $groupId]]],
                    ]);
                } else {
                    $client->delete([
                        'index' => $indexName,
                        'id' => $dataObject->getId(),
                    ]);

                    $client->deleteByQuery([
                        'index' => $indexGroupName,
                        'body' => ['query' => ['term' => ['id' => $dataObject->getId()]]],
                    ]);
                }
            } catch (\Throwable $e) {
                $this->applicationLogger->logException(
                    sprintf(
                        'Cannot erase "%s" index data object ID %d entry',
                        $indexName,
                        $dataObject->getId(),
                    ),
                    $e,
                );
            }

            if ($this->clientService->isUpdateRequest()) {
                $client->indices()->refresh(['index' => $indexName]);
            }
        }
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws ServerResponseException
     * @throws MissingParameterException
     */
    public function indexFulltext(ElasticsearchFulltextInterface $element): void
    {
        if ($element->getProperty(self::ELASTICSEARCH_SKIP_INDEXING)) {
            return;
        }

        foreach ($this->getElementFulltextLanguages($element) as $language) {
            $indexName = $this->clientService->getIndexName(
                $this->clientService->getIndexFulltextSuffix($element, $language),
            );

            if (empty($indexName)) {
                continue;
            }

            $data = $element->elasticsearchGetFulltextMapping($language);

            if ($element->elasticsearchRenderContentToFulltextMapping()) {
                $data['content'] = $this->contentService->renderContent($element, $language);
            }

            $this->clientService->getClient()->index([
                'index' => $indexName,
                'id' => $data['id'],
                'body' => $data,
            ]);

            unset($data);

            if ($this->clientService->isUpdateRequest()) {
                $this->clientService->getClient()->indices()->refresh(['index' => $indexName]);
            }
        }
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws ServerResponseException
     */
    public function eraseFulltext(ElasticsearchFulltextInterface $element): void
    {
        foreach ($this->getElementFulltextLanguages($element) as $language) {
            $indexName = $this->clientService->getIndexName(
                $this->clientService->getIndexFulltextSuffix($element, $language),
            );

            if (empty($indexName)) {
                continue;
            }

            try {
                $this->clientService->getClient()->delete([
                    'index' => $indexName,
                    'id' => $element->getId(),
                ]);
            } catch (\Throwable $e) {
                $this->applicationLogger->logException(
                    sprintf(
                        'Cannot erase "%s" index data object ID %d entry',
                        $indexName,
                        $element->getId(),
                    ),
                    $e,
                );
            }

            if ($this->clientService->isUpdateRequest()) {
                $this->clientService->getClient()->indices()->refresh(['index' => $indexName]);
            }
        }
    }

    /**
     * @param mixed|null $payload
     */
    public function indexSuggestion(string $string, mixed $payload = null): array
    {
        $temp = Utils::webalize($string);

        $suggest = ['input' => explode('-', $temp), 'output' => $string, 'weight' => 1];

        if ($payload) {
            $suggest['payload'] = $payload;
        }

        return $suggest;
    }

    /**
     * @return array<string>
     */
    private function getElementFulltextLanguages(ElasticsearchFulltextInterface $element): mixed
    {
        $languages = [];

        if ($element instanceof Document\Page) {
            $languages[] = $element->getProperty('language') ?: Utils::getDefaultLanguage();
        } else {
            $mapping = $this->clientService->getElementMapping($element);
            $languages = $mapping['meta']['languages'];
        }

        return $languages;
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws MissingParameterException
     * @throws ServerResponseException
     */
    private function reindexDataObjects(array $classNames = [], bool $copyTheRest = true): void
    {
        $classList = (new ClassDefinition\Listing());

        foreach ($classList->getClasses() as $class) {
            $classNameShort = (string)$class->getName();
            $className = '\\App\\Model\\' . $classNameShort;
            $classListName = $className . '\\Listing';

            $classesExist = class_exists($className) && class_exists($classListName);
            $isOfType = in_array(ElasticsearchDataObjectInterface::class, (array)class_implements($className));

            if (!$classesExist && !$isOfType) {
                continue;
            }

            if ($classNames && !in_array($classNameShort, $classNames)) {
                if ($copyTheRest) {
                    $this->clientService->copyIndices($classNameShort, strtolower($classNameShort));
                }

                continue;
            }

            /** @var DataObject\Listing $dataObjectList */
            $dataObjectList = new $classListName();
            $dataObjectList
                ->setUnpublished(true)
                ->setObjectTypes([
                    DataObject\AbstractObject::OBJECT_TYPE_OBJECT,
                    DataObject\AbstractObject::OBJECT_TYPE_VARIANT,
                ]);

            $this->output?->progressStart($dataObjectList->count());

            foreach ($dataObjectList->getObjects() as $dataObject) {
                $this->indexDataObject($dataObject);

                $this->output?->progressAdvance();
            }

            $this->output?->progressFinish();
        }
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws MissingParameterException
     * @throws ServerResponseException
     */
    private function reindexFulltext(array $classNames = [], bool $copyTheRest = true): void
    {
        if (!$classNames || in_array('Page', $classNames)) {
            $documentList = (new Page\Listing())
                ->setUnpublished(true);

            $queue = [Page::getById(1)];

            $this->output?->progressStart($documentList->count());

            while ($queue) {
                $document = array_shift($queue);

                if ($document instanceof ElasticsearchFulltextInterface) {
                    $this->indexFulltext($document);
                }

                $this->output?->progressAdvance();

                $documentList->setCondition(
                    '(type = ? OR type = ?) AND parentId = ?',
                    ['page', 'folder', $document?->getId()],
                );

                $queue = array_merge($queue, $documentList->getDocuments());
            }

            $this->output?->progressFinish();
        } elseif ($copyTheRest) {
            $this->clientService->copyIndices('Document Page', $this->clientService->getDocumentFulltextIndexKey());
        }

        $classList = new ClassDefinition\Listing();

        foreach ($classList->getClasses() as $class) {
            $classNameShort = (string)$class->getName();
            $className = '\\App\\Model\\' . $classNameShort;
            $classListName = $className . '\\Listing';

            $classesExist = @class_exists($className) && @class_exists($classListName);
            $isOfType = in_array(ElasticsearchFulltextInterface::class, (array)class_implements($className));

            if (!$classesExist && !$isOfType) {
                continue;
            }

            if ($classNames && !in_array($classNameShort, $classNames)) {
                if ($copyTheRest) {
                    $this->clientService->copyIndices(
                        $classNameShort,
                        $this->clientService->getDataObjectFulltextIndexKey(),
                    );
                }

                continue;
            }

            /** @var DataObject\Listing $dataObjectList */
            $dataObjectList = new $classListName();
            $dataObjectList
                ->setUnpublished(true)
                ->setObjectTypes([
                    DataObject\AbstractObject::OBJECT_TYPE_OBJECT,
                    DataObject\AbstractObject::OBJECT_TYPE_VARIANT,
                ]);

            $this->output?->progressStart($dataObjectList->count());

            foreach ($dataObjectList->getObjects() as $dataObject) {
                $this->indexFulltext($dataObject);
                
                $this->output?->progressAdvance();
            }

            $this->output?->progressFinish();
        }
    }
}
