<?php declare(strict_types=1);

namespace App\Elasticsearch\Service;

use App\Service\LanguageService;
use Elastic\Elasticsearch\Exception\AuthenticationException;
use Elastic\Elasticsearch\Exception\ClientResponseException;
use Elastic\Elasticsearch\Exception\ServerResponseException;
use Elastic\Elasticsearch\Response\Elasticsearch;
use Symfony\Component\HttpFoundation\RequestStack;

class SearchService
{
    private string $requestLanguage;

    /**
     * @throws \Exception
     */
    public function __construct(
        private readonly ClientService   $clientService,
        private readonly LanguageService $languageService,
        RequestStack                     $requestStack = null,
    ) {
        $this->requestLanguage = $this->languageService->getDefaultLanguage();

        if ($requestStack instanceof RequestStack) {
            $requestLocale = (string)$requestStack->getCurrentRequest()?->getLocale();
            $requestLanguage = substr($requestLocale, 0, 2);

            if ($this->languageService->checkLanguage($requestLanguage)) {
                $this->requestLanguage =$requestLanguage;
            }
        }
    }

    /**
     * @throws AuthenticationException
     * @throws \Throwable
     * @throws ClientResponseException
     * @throws ServerResponseException
     */
    public function fulltextSearch(array $filter = []): array
    {
        $baseOptions = ['index' => [
            $this->clientService->getAliasPrefix() . '_' . $this->clientService->getDocumentFulltextIndexKey() . '_' . $this->requestLanguage,
            $this->clientService->getAliasPrefix() . '_' . $this->clientService->getDataObjectFulltextIndexKey() . '_' .$this->requestLanguage,
        ]];

        if (!empty($filter['query'])) {
            $baseOptions['query']['multi_match']['query'] = $filter['query'];
            $baseOptions['query']['multi_match']['fields'] = ['title^4', 'keywords^3', 'description^2', 'content'];
        }

        $filteredQuery = $this->queryBase($baseOptions, $filter);

        $aggregations = [];
        $aggregations['type'] = $this->fulltextTypeAggregation($filteredQuery);

        if (!empty($filter['types'])) {
            $filteredQuery['body']['query']['bool']['filter']['bool']['must'][]['terms']['type'] = $filter['types'];
        }

        $pagination = $this->query($filteredQuery, $filter, true, true, true);
        $pagination['aggregations'] = $aggregations;

        return $pagination;
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws ServerResponseException
     */
    public function getNews(array $filters = []): array
    {
        $baseOptions = ['index' => $this->clientService->getAliasPrefix() . '_news_' . $this->requestLanguage];

        if (!empty($filters['query'])) {
            $baseOptions['query']['multi_match']['query'] = $filters['query'];
            $baseOptions['query']['multi_match']['fields'] = ['name^1'];
        }

        $filteredQuery = $this->queryBase($baseOptions, $filters);

        return $this->query($filteredQuery, $filters);
    }

    /**
     * @throws \Throwable
     */
    private function fulltextTypeAggregation(array $query, int $size = 100): array
    {
        $aggQuery = [];
        $aggQuery['type']['terms']['size'] = $size;
        $aggQuery['type']['terms']['field'] = 'type';

        $query['body']['aggs'] = $aggQuery;

        /** @var Elasticsearch $result */
        $result = $this->clientService->getClient()->search($query);

        $types = [];

        foreach ($result['aggregations']['type']['buckets'] as $bucket) {
            $types[$bucket['key']] = $bucket['doc_count'];
        }

        return $types;
    }

    private function queryBase(array $options = [], array $filter = []): array
    {
        $filteredQuery['index'] = $options['index'];

        $query = (!empty($options['query'])) ? $options['query'] : null;

        if (is_array($query)) {
            $filteredQuery['body']['query']['bool']['must'] = $query;
        } elseif ('random_score' == $query) {
            $filteredQuery['body']['query']['bool']['must']['function_score']['functions'][0]['random_score'] = new \stdClass();
        } elseif (!empty($filter['ids_score']) && !empty($filter['ids'])) {
            $functionScore = [
                'boost_mode' => 'replace',
                'script_score' => [
                    'script' => "count = ids.size(); return -ids.indexOf(Integer.parseInt(doc['id'].value));",
                    'params' => [
                        'ids' => $filter['ids'],
                    ],
                ],
            ];
            $filteredQuery['body']['query']['bool']['must']['function_score'] = $functionScore;
        } else {
            $filteredQuery['body']['query']['bool']['must']['match_all'] = new \stdClass();
        }

        $filteredQuery['body']['query']['bool']['filter']['bool']['must'][]['term']['published'] = true;

        return $filteredQuery;
    }

    private function constructSimplePagination(Elasticsearch $resultSet, int $limit, int $page, int $totalCount): array
    {
        $resultSet = (array)$resultSet;

        $pagination = [];
        $pagination['current'] = $page;
        $pagination['pageCount'] = ($limit) ? (int) ceil($totalCount / $limit) : 1;

        if ($page > 1) {
            $pagination['previous'] = $page - 1;
        }

        if ($page < $pagination['pageCount']) {
            $pagination['next'] = $page + 1;
        }

        $pagination['totalCount'] = $totalCount;
        $pagination['first'] = 1;
        $pagination['last'] = $pagination['pageCount'];
        $pagination['items'] = empty($resultSet) ? [] : $resultSet['hits']['hits'];

        return $pagination;
    }

    /**
     * @throws AuthenticationException
     * @throws ClientResponseException
     * @throws ServerResponseException
     */
    private function query(
        array $filteredQuery,
        array $filter,
        bool  $foldedFallback = true,
        bool  $paginate = true,
        bool  $highlighting = false,
    ): array {
        $fulltextFields = [];

        if ($foldedFallback && !empty($filter['query'])) {
            if (!empty($filteredQuery['body']['query']['bool']['must']['multi_match']['fields'])) {
                $fulltextFields = $filteredQuery['body']['query']['bool']['must']['multi_match']['fields'];
            }

            $fields = [];

            foreach ($fulltextFields as $field) {
                $parts = explode('^', $field);
                $foldedField = $parts[0].'.folded';
                if (!empty($parts[1])) {
                    $foldedField .= '^'.$parts[1];
                }
                $fields[] = $field;
                $fields[] = $foldedField;
            }

            if (!empty($fields)) {
                $filteredQuery['body']['query']['bool']['must']['multi_match']['fields'] = $fields;
            }
        }

        $countQuery = ['index' => $filteredQuery['index'], 'body' => ['query' => $filteredQuery['body']['query']]];
        $countResult = $this->clientService->getClient()->count($countQuery);
        $totalCount = $countResult['count'];

        if ($totalCount && empty($filter['query']) && !empty($filter['sort']) && empty($filter['ids_score'])) {
            $sortStruct = $filter['sort'];

            if (is_array($sortStruct)) {
                foreach ($sortStruct as $key => $sort) {
                    $filteredQuery['body']['sort'][$key] = $sort;
                }
            } elseif (stristr($sortStruct, ':')) {
                $filteredQuery['body']['sort']['_geo_distance'] = [
                    'location' => str_replace(':', ',', $sortStruct),
                    'order' => 'asc',
                    'unit' => 'km',
                    'distance_type' => 'plane',
                ];
            }
        }

        if ($highlighting && $fulltextFields && !empty($filter['query'])) {
            $highlight = ['fields' => []];

            foreach ($fulltextFields as $field) {
                $parts = explode('^', $field);

                $highlight['fields'][$parts[0]] = [
                    'matched_fields' => [$parts[0]],
                    'type' => 'fvh',
                ];

                if ($foldedFallback) {
                    $highlight['fields'][$parts[0]]['matched_fields'][] = $parts[0].'.folded';
                }
            }

            $filteredQuery['body']['highlight'] = $highlight;
        }

        $limit = !empty($filter['limit']) ? intval($filter['limit']) : 999;

        if ($paginate) {
            $page = !empty($filter['page']) ? intval($filter['page']) : 1;

            if ($page > ceil($totalCount / $limit)) {
                $page = ceil($totalCount / $limit);
            }

            if ($page < 1) {
                $page = 1;
            }

            $filteredQuery['body']['from'] = isset($filter['offset']) ? ($page - 1) * $limit : 0;
            $filteredQuery['body']['size'] = isset($filter['offset']) ? $limit : $page * $limit;
        } else {
            $filteredQuery['body']['size'] = $limit;
        }

        /** @var Elasticsearch $resultSet */
        $resultSet = $this->clientService->getClient()->search($filteredQuery);

        if ($paginate) {
            return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
        }

        return (array)$resultSet;
    }
}
