<?php declare(strict_types=1);

namespace App\Service;

use App\Tool\Utils;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Templating\EngineInterface;
use Twig\Environment;

class AjaxResponseService
{
    protected array $snippets = [];
    protected Request $request;

    public function __construct(protected EngineInterface $templating, protected Environment $twig, RequestStack $requestStack)
    {
        $this->request = $requestStack->getCurrentRequest();

        $this->clearSnippets();
    }

    public function addFlashMessages(): static
    {
        return $this->addElement('#flash-messages', 'snippet/flashMessages.html.twig');
    }

    public function addElement(string $selector, string $body, array $templateVariables = [], bool $rendered = false, bool $append = false): static
    {
        $this->snippets[$selector] = [
            'type' => ($append) ? 'element-append' : 'element',
            'templateVariables' => $templateVariables,
            (($rendered) ? 'body' : 'render') => $body,
        ];

        return $this;
    }

    public function addPopup(string $body, array $templateVariables = [], bool $rendered = false): static
    {
        $this->snippets['popup'] = [
            'type' => 'popup',
            'templateVariables' => $templateVariables,
            (($rendered) ? 'body' : 'render') => $body,
        ];

        return $this;
    }

    public function closePopup(): static
    {
        $this->snippets['popup-close'] = ['type' => 'popup-close'];

        return $this;
    }

    public function removeElement(string $selector, string $body): static
    {
        $this->snippets[$selector] = ['type' => 'remove', 'body' => $body];

        return $this;
    }

    public function hideElement(string $selector, string $body): static
    {
        $this->snippets[$selector] = ['type' => 'hide', 'body' => $body];

        return $this;
    }

    public function showElement(string $selector, string $body): static
    {
        $this->snippets[$selector] = ['type' => 'show', 'body' => $body];

        return $this;
    }

    public function addRedirect(string $url, bool $ajax = false): static
    {
        if (!isset($this->snippets[$url])) {
            $this->snippets[$url] = [
                'type' => ($ajax) ? 'redirect-ajax' : 'redirect',
                'body' => $url,
            ];
        }

        return $this;
    }

    public function addEval(string $script): static
    {
        if (!isset($this->snippets['eval'])) {
            $this->snippets['eval'] = ['type' => 'eval', 'body' => ''];
        }
        $this->snippets['eval']['body'] .= $script;

        return $this;
    }

    public function removeSnippet(string $key): static
    {
        unset($this->snippets[$key]);

        return $this;
    }

    public function clearSnippets(): static
    {
        $this->snippets = [];

        return $this;
    }

    public function getResponse(): RedirectResponse|JsonResponse
    {
        if (!$this->request->isXmlHttpRequest()) {
            return new RedirectResponse(Utils::docUrl('/', $this->twig->getGlobals()['language']));
        } else {
            return new JsonResponse($this->buildResponse());
        }
    }

    private function buildResponse(): array
    {
        $response = ['status' => 'success', 'snippets' => []];
        foreach ($this->snippets as $selector => $snippet) {
            $body = true;
            switch ($snippet['type']) {
                case 'element': case 'element-append': case 'remove': case 'hide':
                case 'show': case 'popup': case 'popup-close':
                    if (isset($snippet['body'])) {
                        $body = $snippet['body'];
                    } elseif (isset($snippet['render'])) {
                        $templateVariables = isset($snippet['templateVariables']) ?
                            array_merge($this->twig->getGlobals(), $snippet['templateVariables']) :
                            $this->twig->getGlobals();
                        $body = $this->templating->render($snippet['render'], $templateVariables);
                    }
                    break;
                case 'redirect': case 'redirect-ajax': case 'eval':
                    $body = $snippet['body'];
                    break;
            }

            if ($selector) {
                $response['snippets'][$selector] = [
                    'type' => $snippet['type'], 'body' => $body,
                ];
            }
        }

        return $response;
    }
}
