<?php declare(strict_types=1);

namespace App\Service;

use App\Tool\Utils;
use Pimcore\Bundle\ApplicationLoggerBundle\ApplicationLogger;
use Pimcore\File;
use Pimcore\Model\DataObject\NewsletterForm;
use Pimcore\Translation\Translator;
use Psr\Log\LogLevel;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Throwable;

class NewsletterService
{
    const ROOT_FOLDER_ID = 6;

    protected Request $request;

    public function __construct(
        RequestStack $requestStack,
        protected Translator $translator,
        protected MailService $mailService,
        protected ApplicationLogger $applicationLogger
    ) {
        $this->request = $requestStack->getCurrentRequest();
    }

    public function subscribe(array $data): bool
    {
        try {
            $newsletter = NewsletterForm::getByEmail($data['email'], 1);

            if ($newsletter && $newsletter->getNewsletterConfirmed()) {
                return true;
            } else {
                $data = array_merge($data, [
                    'subscribeCode' => uniqid(),
                    'unsubscribeCode' => uniqid(),
                ]);
                if (!$newsletter) {
                    $newsletter = NewsletterForm::create($data);
                    $newsletter->setCreationDate(time());
                    $newsletter->setKey(File::getValidFilename($data['email']));
                    $newsletter->setParentId(self::ROOT_FOLDER_ID);
                    $newsletter->setPublished(true);
                    $newsletter->setFirstname('');
                    $newsletter->setLastname('');
                    $newsletter->setNewsletterConfirmed(false);
                    $newsletter->setNewsletterActive(false);
                    $newsletter->save();
                }
            }

            $confirmationUrl = $this->request->getSchemeAndHttpHost() . Utils::url('newsletter-confirm', [
                'code' => $newsletter->getSubscribeCode(),
            ]);

            $this->mailService->sendMail('/notifikace/newsletter/prihlaseni', $newsletter->getLang(), $newsletter->getEmail(), null, [
                'confirmationUrl' => $confirmationUrl,
            ]);

        } catch (Throwable $t) {
            $this->applicationLogger->logException(
                sprintf('Newsletter subscribe %s', $data['email']),
                $t,
                LogLevel::ERROR,
                null,
                __METHOD__ . ':' . __LINE__,
            );

            return false;
        }

        return true;
    }

    public function confirm(string $code): bool
    {
        if (!$newsletter = NewsletterForm::getBySubscribeCode($code, 1)) {
            return false;
        }

        try {
            $newsletter->setNewsletterConfirmed(true);
            $newsletter->setNewsletterActive(true);
            $newsletter->save();
            $unsubscribeUrl = $this->request->getSchemeAndHttpHost() . Utils::url('newsletter-unsubscribe', [
                'code' => $newsletter->getUnsubscribeCode(),
            ]);

            $this->mailService->sendMail('/notifikace/newsletter/potvrzeni-odberu', $newsletter->getLang(), $newsletter->getEmail(), null, [
                'unsubscribeUrl' => $unsubscribeUrl,
            ]);
        } catch (Throwable $t) {
            $this->applicationLogger->logException(
                sprintf('Newsletter confirm %s', $newsletter->getEmail()),
                $t,
                LogLevel::ERROR,
                $newsletter,
                __METHOD__ . ':' . __LINE__,
            );

            return false;
        }

        return true;
    }

    public function unsubscribe(string $code): bool
    {
        if (!$newsletter = NewsletterForm::getByUnsubscribeCode($code, 1)) {
            return false;
        }

        try {
            $email = $newsletter->getEmail();
            $newsletter->delete();

            return $email;
        } catch (Throwable $t) {
            $this->applicationLogger->logException(
                sprintf('Newsletter unsubscribe %s', $newsletter->getEmail()),
                $t,
                LogLevel::ERROR,
                $newsletter,
                __METHOD__ . ':' . __LINE__,
            );

            return false;
        }
    }
}
