<?php declare(strict_types=1);

namespace App\Twig\Extension;

use Exception;
use Pimcore\Model\Document;
use Pimcore\Model\Document\Link;
use Pimcore\Model\Document\Page;
use Pimcore\Navigation\Container;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class NavigationExtension extends AbstractExtension
{
    public function getFunctions(): array
    {
        return [
            new TwigFunction('extend_breadcrumbs', [$this, 'extendBreadcrumbs']),
            new TwigFunction('get_main_navigation', [$this, 'getMainNavigation']),
            new TwigFunction('get_social_navigation', [$this, 'getSocialNavigation']),
        ];
    }

    /**
     * @throws Exception
     */
    public function extendBreadcrumbs($navigation, $addToBreadcrumbs): ?Container
    {
        if (!$navigation instanceof Container || empty($addToBreadcrumbs->pages)) {
            return $navigation;
        }

        $activePages = $navigation->findAllBy('active', true);
        $deepestActive = end($activePages);

        foreach ($addToBreadcrumbs->pages as $key => $page) {
            $newPage = \Pimcore\Navigation\Page\Document::factory([
                'label' => $page['label'],
                'title' => $page['label'],
                'uri' => $page['url'],
                'active' => true,
            ]);
            $deepestActive->addPage($newPage);
            $deepestActive = $newPage;
        }

        return $navigation;
    }

    public function getMainNavigation(Document $rootDocument): array
    {
        $mainNav = [];
        $mainNavDocsListing = $rootDocument->getChildren();

        if (!$mainNavDocsListing->getCount()) {
            return $mainNav;
        }

        foreach ($mainNavDocsListing->getDocuments() as $mainNavDoc) {
            if (!$this->isInMainNavigation($mainNavDoc)) {
                continue;
            }

            $mainNav[] = $mainNavDoc;
        }

        return $mainNav;
    }

    private function isInMainNavigation(Document $document): bool
    {
        return !$document->getProperty('hideInMainNavigation') &&
            $document->isPublished() &&
            ($document instanceof Page || $document instanceof Link);
    }

    public function getSocialNavigation(Document $document): array
    {
        if (!$document) {
            return [];
        }

        $socialNav = [
            ['title' => 'Facebook', 'icon' => 'facebook', 'url' => $document->getProperty('facebookUrl')],
            ['title' => 'LinkedIn', 'icon' => 'linkedin', 'url' => $document->getProperty('linkedInUrl')],
            ['title' => 'Instagram', 'icon' => 'instagram', 'url' => $document->getProperty('instagramUrl')],
            ['title' => 'Youtube', 'icon' => 'youtube', 'url' => $document->getProperty('youtubeUrl')],
            ['title' => 'X', 'icon' => 'x', 'url' => $document->getProperty('xUrl')],
        ];

        $socialNav = array_filter($socialNav, function ($item) { return !empty($item['url']); });

        return $socialNav;
    }
}
