<?php

namespace AppBundle\Controller;

use AppBundle\Form\NewsletterFormType;
use AppBundle\Tool\Utils;
use Pimcore\Controller\FrontendController;
use Pimcore\Model\Document;
use Pimcore\Model\Site;
use Pimcore\Cache;
use Symfony\Component\HttpKernel\Event\FilterControllerEvent;
use Symfony\Component\Form\FormInterface;

class BaseController extends FrontendController
{
	use Helper\ControllerTrait;

	/** @var FormInterface */
	protected $newsletterForm;

	/**
	 * This method is called before any action.
	 *
	 * @param FilterControllerEvent $event
	 */
	public function onKernelController(FilterControllerEvent $event)
	{
		//enable view auto-rendering (no need to return response from actions)
		$this->setViewAutoRender($event->getRequest(), true, 'twig');

		//avoid cache misses in some cases
		\Pimcore\Cache::setForceImmediateWrite(true);

		//set request
		$this->request = $event->getRequest();

		//detect language
		$this->defaultLanguage = Utils::getDefaultLanguage();
		$this->language = substr($event->getRequest()->getLocale(), 0, 2);
		if (!$this->language) {
			$this->language = Utils::getDefaultLanguage();
		}
		$this->view->language = $this->language;

		//set translator
		$this->translator = $this->get('translator');

		//detect current url
		$this->currentUrl = strtok($event->getRequest()->getRequestUri(), '?');
		$this->view->currentUrl = $this->currentUrl;

		//generate unique cache keys for various cachable content chunks based on current URL
		$cacheKeySuffix = $this->language.'_'.$this->currentUrl;
		if (Site::isSiteRequest()) {
			$cacheKeySuffix = Site::getCurrentSite()->id.'_'.$cacheKeySuffix;
		}
		$this->cacheKeySuffix = Utils::toCacheKey($cacheKeySuffix);
		$this->view->cacheKey = $this->cacheKeySuffix;
		$this->view->breadcrumbsCacheKey = '_breadcrumbs_'.$this->cacheKeySuffix;
		$this->view->metaCacheKey = '_meta_'.$this->cacheKeySuffix;
		$this->view->languageSwitchCacheKey = '_language_switch_'.$this->cacheKeySuffix;

		//assign breadcrumbs extension to template variable by reference (for legacy reasons)
		$this->view->addToBreadcrumbs = new \stdClass();
		$this->view->addToBreadcrumbs->pages = &$this->addToBreadcrumbs;

		//assign language switch extension to template variable by reference (for legacy reasons)
		$this->view->languageSwitch = new \stdClass();
		$this->view->languageSwitch->pages = &$this->languageSwitch;

		//set some document editables defaults
		$this->view->width = 0;

		// Other variables.
		$this->view->isFrontendRequestByAdmin = \Pimcore\Tool::isFrontendRequestByAdmin();

		$this->initHelpers();

		//global forms
		if ($this->document instanceof Document\Page) {
			$this->newsletterForm = $this->createForm(NewsletterFormType::class);
			$this->view->newsletterForm = $this->newsletterForm->createView();
		}

		// Flash messages.
		// getSession() returns a SessionInterface, that's why the IDE complains
		// about a non-existent "getFlashBag()" method. But it works.
		if($this->request->getSession()) {
			$sessionFlashMessages = $this->request->getSession()->getFlashBag()->all();
			$currentRequestFlashMessages = Cache\Runtime::isRegistered('currentRequestFlashMessages') ? Cache\Runtime::get('currentRequestFlashMessages') : [];
			$this->view->flashMessages = array_merge($sessionFlashMessages, $currentRequestFlashMessages);
		}

		// Main navigation.
		$mainNav = [];
		$rootDoc = Utils::getTranslatedDocumentByPath('/'.Utils::getDefaultLanguage(), $this->language)
				?: Utils::getTranslatedDocumentByPath('/', $this->language);
		if ($rootDoc && ($mainNavDocs = $rootDoc->getChildren())) {
			foreach ($mainNavDocs as $mainNavDoc) {
				if($mainNavDoc->getProperty('hideInMainNav') || !$mainNavDoc->isPublished()) {
					continue;
				}
				if ($mainNavDoc instanceof Document\Page || $mainNavDoc instanceof Document\Link) {
					$mainNav[] = $mainNavDoc;
				}
			}
		}
		$this->view->mainNav = $mainNav;
		$this->view->rootDocument = $rootDoc;
	}
}
