<?php

namespace AppBundle\Tool;

use Pimcore\Model\Asset;
use Pimcore\Model\Document;
use Pimcore\Model\DataObject;
use Pimcore\Model\Site;
use Pimcore\Model\Staticroute;

class Utils
{
	//hardcoded pimcore IDs
	const DOCUMENT_ROOT = 2;
	const DOCUMENT_ACCOUNT = 12;

	const SALT = 'hopcipcibirybcibirybhopaa';

	/** @var bool */
	private static $inheritance;
	/** @var bool */
	private static $fallback;
	/** @var array */
	private static $languageSwitch = null;
	/** @var string */
	public static $defaultLanguage = null;

	/**
	 * returns the default website language (the first from system settings)
	 * default language should match the language property of master document tree.
	 *
	 * @return string default language ISO 639-1
	 */
	public static function getDefaultLanguage()
	{
		if (null === self::$defaultLanguage) {
			$config = \Pimcore\Config::getSystemConfiguration();
			self::$defaultLanguage = current(explode(',', $config['general']['validLanguages']));
		}

		return self::$defaultLanguage;
	}

	/**
	 * defines the valid frontend languages as a subset of valid admin languages
	 * used when we need to add another language in the admin, but don't want to break the site while filling in all the localized data...
	 *
	 * @return array valid languages (ISO 639-1)
	 */
	public static function getValidFrontendLanguages()
	{
		$config = \Pimcore\Config::getWebsiteConfig();
		$allowed = explode(',', $config->get('frontendLanguages', self::getDefaultLanguage()));
		$combined = [];

		foreach (\Pimcore\Tool::getValidLanguages() as $language) {
			if (in_array($language, $allowed)) {
				$combined[] = $language;
			}
		}

		return $combined;
	}

	public static function getDocumentTranslation($document, $language)
	{
		if ($language == $document->getProperty('language')) {
			return $document;
		}
		$service = new \Pimcore\Model\Document\Service();
		$translations = $service->getTranslations($document);
		if ($translations && $translations[$language]) {
			return \Pimcore\Model\Document\Page::getById($translations[$language]);
		}
		return null;
	}

	public static function getTranslatedDocumentByPath($documentPath, $language)
	{
		$document = \Pimcore\Model\Document\Page::getByPath($documentPath);
		if ($document) {
			if ($language == $document->getProperty('language')) {
				return $document;
			}
			return self::getDocumentTranslation($document, $language);
		}
		return null;
	}

	/**
	 * @param Document\Tag\Image|DataObject\Data\Hotspotimage|Asset\Image $image
	 * @param string                                                      $thumbnail     image thumbnail pipeline name
	 * @param Asset\Image|string|int                                      $fallbackImage
	 *
	 * @return string relative URL
	 */
	public static function thumbnail($image, $thumbnail, $fallbackImage = null)
	{
		if ($image instanceof Document\Tag\Image && $image->getImage()) {
			return (string) $image->getThumbnail($thumbnail);
		} elseif ($image instanceof DataObject\Data\Hotspotimage && $image->getImage()) {
			return (string) $image->getThumbnail($thumbnail);
		} elseif ($image instanceof Asset\Image) {
			return (string) $image->getThumbnail($thumbnail);
		} elseif ($fallbackImage instanceof Asset\Image) {
			return (string) $fallbackImage->getThumbnail($thumbnail);
		} elseif ($fallbackImage) {
			$image = Asset\Image::getById($fallbackImage);
			if (!$image) {
				$image = Asset\Image::getByPath($fallbackImage);
			}
			if ($image) {
				return (string) $image->getThumbnail($thumbnail);
			}
		} else {
			return '';
		}
	}

	/**
	 * @param Document\Tag\Image|DataObject\Data\Hotspotimage|Asset\Image $image
	 * @param string                                                      $thumbnail         image thumbnail pipeline name
	 * @param Asset\Image|string|int                                      $fallbackImage
	 * @param array                                                       $additionalOptions additional options (see pimcore's document image editable documentation)
	 *
	 * @return string HTML image element like the document image editable
	 */
	public static function image($image, $thumbnail, $fallbackImage = null, $additionalOptions = [])
	{
		$html = null;
		$imageTag = new Document\Tag\Image();
		if ($image instanceof Document\Tag\Image && $image->getImage()) {
			$imageTag = $image;
		} elseif ($image instanceof Asset\Image) {
			$imageTag->setImage($image);
		} elseif ($image instanceof DataObject\Data\Hotspotimage && $image->getImage()) {
			$html = $image->getThumbnail($thumbnail)->getHTML($additionalOptions);
		} elseif ($fallbackImage instanceof Asset\Image) {
			$imageTag->setImage($fallbackImage);
		} elseif ($fallbackImage) {
			$image = Asset\Image::getById($fallbackImage);
			if (!$image) {
				$image = Asset\Image::getByPath($fallbackImage);
			}
			if ($image) {
				$imageTag->setImage($image);
			}
		}

		if (!$html) {
			$additionalOptions['thumbnail'] = $thumbnail;
			$imageTag->setOptions($additionalOptions);
			$html = $imageTag->frontend();
		}

		return $html;
	}

	/**
	 * @param Document\Tag\Link|DataObject\Data\Link $link
	 * @param array                                  $additionalAttributes
	 * @param string                                 $text
	 *
	 * @return string ahref html
	 */
	public static function link($link, $additionalAttributes = [], $text = null)
	{
		if ($link instanceof Document\Tag\Link) {
			$link->setOptions($additionalAttributes);
		} elseif ($link instanceof DataObject\Data\Link) {
			$link->setValues($additionalAttributes);
		}
		$tag = (string) $link;
		if ($text) {
			$tag = preg_replace('#(<a.*?>).*?(</a>)#', '$1' . $text . '$2', $tag);
		}

		return $tag;
	}

	/**
	 * @param Staticroute|string $staticRoute
	 * @param array              $params
	 *
	 * @return string relative or absolute (if from other Site) URL
	 */
	public static function url($staticRoute, $params = [])
	{
		if (!$staticRoute instanceof Staticroute) {
			$staticRoute = Staticroute::getByName($staticRoute, (Site::isSiteRequest()) ? Site::getCurrentSite()->getId() : 0);
		}

		if (!$staticRoute) {
			return '/';
		}

		return $staticRoute->assemble($params, true);
	}

	/**
	 * @param string        $realFullPath
	 * @param string        $language
	 * @param Document\Page $rootDoc
	 *
	 * @return string relative or absolute (if from other Site) URL
	 */
	public static function docUrl($realFullPath, $language, $rootDoc = null)
	{
		$languageSwitch = self::getLanguageSwitch($language, $realFullPath, $rootDoc);

		return (isset($languageSwitch[$language])) ? $languageSwitch[$language] : '/';
	}

	/**
	 * @param string        $language
	 * @param string        $section    realFullPath
	 * @param Document\Page $rootDoc
	 * @param bool          $hpFallback
	 *
	 * @return array language => URL pairs
	 */
	public static function getLanguageSwitch($language, $section, $rootDoc = null, $hpFallback = false)
	{
		if (!$rootDoc) {
			$rootDoc = Document\Page::getById(1);
		}
		$siteId = (Site::isSiteRequest())
			? Site::getCurrentSite()->getRootId()
			: $rootDoc->getId();
		$cacheKey = 'language_switcher_' . $rootDoc->getId() . '_' . $siteId;
		$switcher = (self::$languageSwitch[$rootDoc->getId()])
			? self::$languageSwitch[$rootDoc->getId()]
			: \Pimcore\Cache::load($cacheKey);
		if (false === $switcher) {
			$switcher = [];
			$originLanguage = $rootDoc->getProperty('language');
			$queue = [$rootDoc];
			$service = new Document\Service();
			while (!empty($queue)) {
				$doc = array_shift($queue);
				foreach ($doc->getChildren(true) as $child) {
					$queue[] = $child;
				}
				if (!$doc->getPublished()) {
					continue;
				}

				$docRealFullPath = $doc->getRealFullPath();
				$translations = $service->getTranslations($doc);
				if (!isset($translations[$originLanguage])) {
					$translations[$originLanguage] = $doc->getId();
				}

				foreach ($translations as $language => $translationDocId) {
					$translationDoc = Document::getById($translationDocId);
					if ($translationDoc && $translationDoc->isPublished()) {
						$switcher[$docRealFullPath][$language] = $translationDoc->getFullPath();
					}
				}
			}

			// global cache, for frontend requests
			if (!\Pimcore\Tool::isFrontendRequestByAdmin()) {
				\Pimcore\Cache::save($switcher, $cacheKey, ['output']);
			}
		}

		// per-request cache
		self::$languageSwitch[$rootDoc->getId()] = $switcher;

		$sectionSwitch = (is_array($switcher) && isset($switcher[$section])) ? $switcher[$section] : [];

		// add HPs for non-existing page localizations
		if ($hpFallback && is_array($switcher)) {
			$langParamPosition = stripos($section, '/' . $language . '/');
			if (false !== $langParamPosition) {
				$hpKey = substr($section, 0, $langParamPosition) . '/' . $language;
				if (isset($switcher[$hpKey])) {
					$sectionSwitch += $switcher[$hpKey];
				}
			}
		}

		return $sectionSwitch;
	}

	/**
	 * @return bool
	 */
	public static function elasticSearchEnabled()
	{
		return class_exists('\\Elasticsearch\\Client');
	}

	/**
	 * turn object inheritance and fallback values on/off using the Force (usefull outside of admin).
	 *
	 * @param bool $inherit
	 */
	public static function forceInheritanceAndFallbackValues($inherit = true)
	{
		self::$inheritance = DataObject\AbstractObject::getGetInheritedValues();
		self::$fallback = DataObject\Localizedfield::getGetFallbackValues();
		DataObject\AbstractObject::setGetInheritedValues($inherit);
		DataObject\Localizedfield::setGetFallbackValues($inherit);
	}

	/**
	 * restore object inheritance and fallback values setup after calling self::forceInheritanceAndFallbackValues.
	 */
	public static function restoreInheritanceAndFallbackValues()
	{
		DataObject\AbstractObject::setGetInheritedValues(self::$inheritance);
		DataObject\Localizedfield::setGetFallbackValues(self::$fallback);
	}

	/**
	 * load assets (recursively if a Asset/Folder was provided).
	 *
	 * @param Asset\Folder|Asset[] $rootFolderOrAssetArray
	 * @param array                $allowedTypes           image, document, video...
	 *
	 * @return Asset[]
	 */
	public static function loadAssets($rootFolderOrAssetArray = [], $allowedTypes = ['image'])
	{
		$queue = [];
		if ($rootFolderOrAssetArray instanceof Asset\Folder) {
			$queue[] = $rootFolderOrAssetArray;
		} elseif (is_array($rootFolderOrAssetArray)) {
			$queue = $rootFolderOrAssetArray;
		} else {
			$root = Asset\Folder::getById($rootFolderOrAssetArray);
			if ($root) {
				$queue[] = $root;
			}
		}

		$assets = [];
		while ($asset = array_shift($queue)) {
			if (!$asset instanceof Asset) {
				continue;
			}

			if ($asset instanceof Asset\Folder) {
				foreach ($asset->getChildren() as $child) {
					$queue[] = $child;
				}
			} elseif (in_array($asset->getType(), $allowedTypes)) {
				$assets[] = $asset;
			}
		}

		return $assets;
	}

	/**
	 * @param mixed  $listObjectOrClassName
	 * @param int    $limit
	 * @param int    $offsetId
	 * @param string $orderKey
	 * @param string $order
	 * @param string $additionalConditionsAsString
	 *
	 * @return array
	 */
	public static function loadMoreItems($listObjectOrClassName, $limit, $offsetId = 0, $orderKey = 'o_id', $order = 'asc', $additionalConditionsAsString = null)
	{
		$result = ['items' => [], 'offsetId' => null];

		//load the offset item if needed
		$itemClass = str_replace('\\Listing', '', (is_object($listObjectOrClassName)) ? get_class($listObjectOrClassName) : $listObjectOrClassName);
		if ($offsetId) {
			$offsetItem = $itemClass::getById($offsetId);
			if (!$offsetItem) {
				return $result;
			}
		}

		//just in case
		if ('o_id' == $orderKey && 'Object' != substr($itemClass, 0, 6)) {
			$orderKey = 'id';
		}

		//create the list
		$list = (is_object($listObjectOrClassName)) ? $listObjectOrClassName : new $listObjectOrClassName();
		if (!$list) {
			return $result;
		}

		//create the condition
		$conditions = [];
		if ($additionalConditionsAsString) {
			$conditions[] = $additionalConditionsAsString;
		}
		if ($list->getCondition()) {
			$conditions[] = $list->getCondition();
		}
		if ($offsetId) {
			$offsetValue = $offsetItem->{'get' . ucfirst($orderKey)}();
			if ($offsetValue instanceof \Pimcore\Date) {
				$offsetValue = $offsetValue->getTimestamp();
			}
			$conditions[] = sprintf('`%s` %s %s', $orderKey, ('asc' == $order) ? '>=' : '<=', $list->quote($offsetValue));
		}

		//load the list
		$list->setOrderKey($orderKey);
		$list->setOrder($order);
		$list->setLimit($limit + 1); // +1 because we want to check if there are more items
		$list->setCondition(implode(' AND ', $conditions));
		$list->load();

		//remove last item and set next offset ID
		$items = $list->getItems(0, 0);
		if (count($items) > $limit) {
			$tmp = array_pop($items);
			$result['offsetId'] = $tmp->getId();
		}
		$result['items'] = $items;

		return $result;
	}

	//Text stuff

	/**
	 * @param string $path
	 *
	 * @return string
	 */
	public static function webalize($path)
	{
		return trim(preg_replace('#-+#', '-', str_replace('.', '-', \Pimcore\File::getValidFilename($path))), '-');
	}

	/**
	 * @param string $text
	 *
	 * @return string
	 */
	public static function toCacheKey($text)
	{
		return str_replace(['-', '~'], '_', self::webalize($text));
	}

	/**
	 * @param string $text
	 *
	 * @return string
	 */
	public static function toObjectKey($text)
	{
		return preg_replace('#-+#', '-', str_replace(['_', '~'], '-', self::webalize($text)));
	}

	/**
	 * @param string $string
	 * @param int    $length
	 * @param string $suffix
	 *
	 * @return string
	 */
	public static function cutStringRespectingWhitespace($string, $length, $suffix = '...')
	{
		if ($length < strlen($string)) {
			$text = substr($string, 0, $length);
			if (false !== ($length = strrpos($text, ' '))) {
				$text = substr($text, 0, $length);
			}
			$string = $text . $suffix;
		}

		return $string;
	}

	/**
	 * @param string $string
	 * @param string $encoding
	 *
	 * @return string
	 */
	public static function mb_ucfirst($string, $encoding = 'utf-8')
	{
		$strlen = mb_strlen($string, $encoding);
		$firstChar = mb_substr($string, 0, 1, $encoding);
		$restOfString = mb_substr($string, 1, $strlen - 1, $encoding);

		return mb_strtoupper($firstChar, $encoding) . $restOfString;
	}

	/**
	 * @param string $string
	 * @param string $encoding
	 *
	 * @return string
	 */
	public static function mb_lcfirst($string, $encoding = 'utf-8')
	{
		$strlen = mb_strlen($string, $encoding);
		$firstChar = mb_substr($string, 0, 1, $encoding);
		$restOfString = mb_substr($string, 1, $strlen - 1, $encoding);

		return mb_strtolower($firstChar, $encoding) . $restOfString;
	}

	/**
	 * @param int $month
	 *
	 * @return string
	 */
	public static function czechMonth($month)
	{
		static $arr = [1 => 'leden', 'únor', 'březen', 'duben', 'květen', 'červen', 'červenec', 'srpen', 'září', 'říjen', 'listopad', 'prosinec'];

		return $arr[$month];
	}

	/**
	 * @param int $day
	 *
	 * @return string
	 */
	public static function czechDay($day)
	{
		$days = ['neděle', 'pondělí', 'úterý', 'středa', 'čtvrtek', 'pátek', 'sobota'];

		return $days[$day];
	}

	/**
	 * @param int $size
	 *
	 * @return string
	 */
	public static function formatSizeUnits($size)
	{
		if ($size >= 1073741824) {
			$bytes = number_format($size / 1073741824, 2, ',', ' ') . ' GB';
		} elseif ($size >= 1048576) {
			$bytes = number_format($size / 1048576, 2, ',', ' ') . ' MB';
		} elseif ($size >= 1024) {
			$bytes = number_format($size / 1024, 2, ',', ' ') . ' KB';
		} elseif ($size > 1) {
			$bytes = $size . ' B';
		} elseif (1 == $size) {
			$bytes = $size . ' B';
		} else {
			$bytes = '0 B';
		}

		return $bytes;
	}

	/**
	 * @param string $text
	 *
	 * @return string
	 */
	public static function encrypt($text)
	{
		return trim(base64_encode(mcrypt_encrypt(MCRYPT_RIJNDAEL_256, self::SALT, $text, MCRYPT_MODE_ECB, mcrypt_create_iv(mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_ECB), MCRYPT_RAND))));
	}

	/**
	 * @param string $text
	 *
	 * @return string
	 */
	public static function decrypt($text)
	{
		return trim(mcrypt_decrypt(MCRYPT_RIJNDAEL_256, self::SALT, base64_decode($text), MCRYPT_MODE_ECB, mcrypt_create_iv(mcrypt_get_iv_size(MCRYPT_RIJNDAEL_256, MCRYPT_MODE_ECB), MCRYPT_RAND)));
	}

	/**
	 * @param mixed $data
	 *
	 * @return string
	 */
	public static function serialize($data)
	{
		return base64_encode(serialize($data));
	}

	/**
	 * @param string $data
	 *
	 * @return mixed
	 */
	public static function unserialize($data)
	{
		return unserialize(base64_decode($data));
	}

	/**
	 * @param string $s
	 *
	 * @return string
	 */
	public static function detectCzechEncoding($s)
	{
		if (preg_match('#[\x80-\x{1FF}\x{2000}-\x{3FFF}]#u', $s)) {
			return 'UTF-8';
		}

		if (preg_match('#[\x7F-\x9F\xBC]#', $s)) {
			return 'WINDOWS-1250';
		}

		return 'ISO-8859-2';
	}
}
