<?php

namespace AppBundle\Twig\Extension;

use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;
use Pimcore\Templating\Renderer\TagRenderer;
use AppBundle\Tool\Utils;

class DocExtension extends AbstractExtension
{
	/**
	 * @var TagRenderer
	 */
	protected $tagRenderer;

	/**
	 * @param TagRenderer $tagRenderer
	 */
	public function __construct(TagRenderer $tagRenderer)
	{
		$this->tagRenderer = $tagRenderer;
	}

	/**
	 * @return array|TwigFunction[]
	 */
	public function getFunctions()
	{
		return [
			new TwigFunction('doc_url', [$this, 'docUrl'], [
				'needs_context' => true,
			]),
			new TwigFunction('doc_url_by_id', [$this, 'docUrlById'], [
				'needs_context' => true,
			]),
			new TwigFunction('doc_is_active', [$this, 'docIsActive'], [
				'needs_context' => true,
			]),
			new TwigFunction('doc_language_switch', [$this, 'docLanguageSwitch']),
			new TwigFunction('doc_editable', [$this, 'docEditable'], [
				'needs_context' => true,
				'is_safe' => ['html'],
			]),
			new TwigFunction('doc_pagination', [$this, 'docPagination']),
		];
	}

	/**
	 * Returns a document URL by document path, based on current request language.
	 *
	 * @param \Twig\Environment $context
	 * @param string $realFullPath
	 * @return string
	 */
	public function docUrl($context, $realFullPath)
	{
		return Utils::docUrl($realFullPath, $context['language']);
	}

	/**
	 * Returns a document URL by document id, based on current request language.
	 *
	 * @param \Twig\Environment $context
	 * @param string $realFullPath
	 * @return string
	 */
	public function docUrlById($context, $id)
	{
		return Utils::docUrlById($id, $context['language']);
	}

	/**
	 * Is document with $id currently active?
	 *
	 * @param \Twig\Environment $context
	 * @param mixed $id
	 * @return bool
	 */
	public function docIsActive($context, $id)
	{
		if (!$id || !($document = $context['document'])) {
			return false;
		}
		return (
			$document instanceof \Pimcore\Model\Document\Page
			&&
			($document->getId() == $id || $document->getContentMasterDocumentId() == $id)
		);
	}

	/**
	 * Returns document language switcher.
	 *
	 * @param string $language
	 * @param string $section
	 * @param \Pimcore\Model\Document $rootDoc
	 * @param boolean $hpFallback
	 * @return array
	 */
	public function docLanguageSwitch($language, $section, $rootDoc = null, $hpFallback = false)
	{
		return Utils::getLanguageSwitch($language, $section, $rootDoc, $hpFallback);
	}

	/**
	 * Returns a Pimcore tag with set defaults.
	 *
	 * @param \Twig\Environment $context
	 * @param string $name
	 * @param string $type
	 * @param array $options
	 * @return \Pimcore\Model\Document\Tag
	 */
	public function docEditable($context, $name, $type, $options = [])
	{
		$defaultConfigurations = [
			'input' => [
				'htmlspecialchars' => true,
			],
			'numeric' => [
				'minValue' => 0,
			],
			'textarea' => [
				'height' => 100,
				'nl2br' => true,
				'htmlspecialchars' => true,
			],
			'wysiwyg' => [
				'height' => 100,
				'inline' => true,
				'enterMode' => 1,
				'customConfig' => '/bundles/app/js/pimcore/cke_config_document_loader.js',
			],
			'image' => [
				'hidetext' => true,
				'reload' => false,
				'thumbnail' => 'AdminFallback',
				'uploadPath' => '/dokumenty',
			],
			'multihref' => [
				'uploadPath' => '/dokumenty',
				'thumbnail' => null,
			],
			'relations' => [
				'uploadPath' => '/dokumenty',
				'thumbnail' => null,
			],
			'href' => [
			],
			'relation' => [
			],
			'checkbox' => [
				'reload' => false,
			],
			'date' => [
				'format' => 'd.m.Y',
				'startDay' => 1,
			],
			'select' => [
				'reload' => false,
				'store' => [
					['dummy', 'dummy'],
				],
			],
			'table' => [
				'defaults' => [
					'cols' => 1,
					'rows' => 1,
					'data' => [
						['dummy'],
					]
				],
			],
			'multiselect' => [
				'store' => [
					['dummy', 'dummy'],
				],
			],
		];
		if ($type === 'image' && $context['editmode']) {
			$defaultConfigurations['image']['width'] = 0;
		}
		if (array_key_exists($type, $defaultConfigurations)) {
			$defaultOptions = $defaultConfigurations[$type];
			$options = (function () use ($defaultOptions, $options) {
				foreach (array_keys($defaultOptions) as $option) {
					// Replace defaults with passed options.
					if (isset($options[$option])) {
						if ($options[$option]) {
							$defaultOptions[$option] = $options[$option];
						} else {
							unset($defaultOptions[$option]);
						}
					}
				}
				// Add passed options missing in defaults.
				foreach ($options as $option => $value) {
					if (!isset($defaultOptions[$option])) {
						$defaultOptions[$option] = $value;
					}
				}
				return $defaultOptions;
			})();
			return $this->tagRenderer->render($context['document'], $type, $name, $options, $context['editmode']);
		}
	}

	/**
	 * Builds and returns an array of pagination variables.
	 *
	 * @param object $pagination
	 * @return object
	 */
	public function docPagination(object $pagination)
	{
		$urlPrevious = $urlFirst = $urlNext = $urlLast = 'javascript:void(0)';
		if (isset($pagination->previous) && !isset($pagination->ajax)) {
			$urlFirst = $pagination->urlprefix . $pagination->first;
			$urlPrevious = $pagination->urlprefix . $pagination->previous;
		}
		if (isset($pagination->next) && !isset($pagination->ajax)) {
			$urlNext = $pagination->urlprefix . $pagination->next;
			$urlLast = $pagination->urlprefix . $pagination->last;
		}
		$listBoundaryLength = 3;
		$listStart = max([1, $pagination->current - $listBoundaryLength]);
		$listEnd = min([$pagination->last, $listStart + $listBoundaryLength * 2]);
		$listStart = max([1, $listEnd - $listBoundaryLength * 2]);
		// Extend the pagination with new properties.
		$pagination->isAjax = isset($pagination->ajax);
		$pagination->urlPrevious = $urlPrevious;
		$pagination->urlFirst = $urlFirst;
		$pagination->urlNext = $urlNext;
		$pagination->urlLast = $urlLast;
		$pagination->listBoundaryLength = $listBoundaryLength;
		$pagination->listStart = $listStart;
		$pagination->listEnd = $listEnd;
		return $pagination;
	}
}
