<?php

namespace AppBundle\Controller\Helper;

use AppBundle\Tool\Utils;
use Pimcore\Templating\Model\ViewModel;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\Templating\EngineInterface;

class AjaxResponseBuilder
{
	/** @var array */
	protected $snippets = [];
	/** @var Request */
	protected $request;
	/** @var EngineInterface */
	protected $templating;
	/** @var ViewModel */
	protected $view;

	/**
	 * @param Request         $request
	 * @param EngineInterface $templating
	 * @param ViewModel       $view
	 */
	public function __construct(Request $request, EngineInterface $templating, ViewModel $view)
	{
		$this->request = $request;
		$this->templating = $templating;
		$this->view = $view;

		$this->clearSnippets();
	}

	/**
	 * @return $this
	 */
	public function addFlashMessages()
	{
		return $this->addElement('#flash-messages', 'Snippet/flashMessages.html.php');
	}

	/**
	 * @param string $selector
	 * @param string $body
	 * @param bool   $rendered
	 * @param bool   $append
	 *
	 * @return $this
	 */
	public function addElement($selector, $body, $rendered = false, $append = false)
	{
		$this->snippets[$selector] = [
			'type' => ($append) ? 'element-append' : 'element',
			(($rendered) ? 'body' : 'render') => $body,
		];

		return $this;
	}

	/**
	 * @param string $body
	 * @param bool   $rendered
	 *
	 * @return $this
	 */
	public function addPopup($body, $rendered = false)
	{
		$this->snippets['popup'] = [
			'type' => 'popup',
			(($rendered) ? 'body' : 'render') => $body,
		];

		return $this;
	}

	/**
	 * @return $this
	 */
	public function closePopup()
	{
		$this->snippets['popup-close'] = ['type' => 'popup-close'];

		return $this;
	}

	/**
	 * @param string $selector
	 * @param string $body
	 *
	 * @return $this
	 */
	public function removeElement($selector, $body)
	{
		$this->snippets[$selector] = ['type' => 'remove', 'body' => $body];

		return $this;
	}

	/**
	 * @param string $selector
	 * @param string $body
	 *
	 * @return $this
	 */
	public function hideElement($selector, $body)
	{
		$this->snippets[$selector] = ['type' => 'hide', 'body' => $body];

		return $this;
	}

	/**
	 * @param string $selector
	 * @param string $body
	 *
	 * @return $this
	 */
	public function showElement($selector, $body)
	{
		$this->snippets[$selector] = ['type' => 'show', 'body' => $body];

		return $this;
	}

	/**
	 * @param string $url
	 * @param bool   $ajax
	 *
	 * @return $this
	 */
	public function addRedirect($url, $ajax = false)
	{
		if (!isset($this->snippets[$url])) {
			$this->snippets[$url] = [
				'type' => ($ajax) ? 'redirect-ajax' : 'redirect',
				'body' => $url,
			];
		}

		return $this;
	}

	/**
	 * @param string $script
	 *
	 * @return $this
	 */
	public function addEval($script)
	{
		if (!isset($this->snippets['eval'])) {
			$this->snippets['eval'] = ['type' => 'eval', 'body' => ''];
		}
		$this->snippets['eval']['body'] .= $script;

		return $this;
	}

	/**
	 * @param string $key
	 *
	 * @return $this
	 */
	public function removeSnippet($key)
	{
		unset($this->snippets[$key]);

		return $this;
	}

	/**
	 * @return $this
	 */
	public function clearSnippets()
	{
		$this->snippets = [];

		return $this;
	}

	/**
	 * @return RedirectResponse|JsonResponse
	 */
	public function getResponse()
	{
		if (!$this->request->isXmlHttpRequest()) {
			return new RedirectResponse(Utils::docUrl('/', $this->view->get('language')));
		} else {
			return new JsonResponse($this->buildResponse());
		}
	}

	/**
	 * @return array
	 */
	private function buildResponse()
	{
		$response = ['status' => 'success', 'snippets' => []];

		foreach ($this->snippets as $selector => $snippet) {
			$body = true;
			switch ($snippet['type']) {
				case 'element': case 'element-append': case 'remove': case 'hide':
				case 'show': case 'popup': case 'popup-close':
					if (isset($snippet['body'])) {
						$body = $snippet['body'];
					} elseif (isset($snippet['render'])) {
						$body = $this->templating->render($snippet['render'], $this->view->getAllParameters());
					}
					break;
				case 'redirect': case 'redirect-ajax': case 'eval':
					$body = $snippet['body'];
					break;
			}

			if ($selector) {
				$response['snippets'][$selector] = [
					'type' => $snippet['type'], 'body' => $body,
				];
			}
		}

		return $response;
	}
}
