<?php

namespace AppBundle\Model;

use AppBundle\Tool;
use AppBundle\Tool\Utils;
use Pimcore\Log;
use Pimcore\Templating;
use Pimcore\Translation\Translator;
use Sunra\PhpSimple\HtmlDomParser;

trait ElasticTrait
{
	/** @var string */
	private $shortClassName = null;
	/** @var array */
	private $elasticSearchmapping = null;

	/**
	 * @return string
	 */
	protected function getShortClassName()
	{
		if (!$this->shortClassName) {
			$reflection = new \ReflectionClass($this);
			$this->shortClassName = $reflection->getShortName();
		}

		return $this->shortClassName;
	}

	/**
	 * get custom object type mapping.
	 *
	 * @return array
	 */
	private function getObjectMapping()
	{
		if (!$this->elasticSearchmapping) {
			$this->elasticSearchmapping = Tool\ElasticSearch::getObjectMapping(
				$this->getShortClassName()
			);
		}

		return $this->elasticSearchmapping;
	}

	/**
	 * @param string $lang
	 * @param bool   $forFulltext
	 *
	 * @return string
	 */
	private function getElasticType($lang, $forFulltext = false)
	{
		if ($forFulltext) {
			if ($this instanceof Document\Page) {
				return Tool\ElasticSearch::DOCUMENT_FULLTEXT_TYPE_KEY.'_'.$lang;
			} else {
				return Tool\ElasticSearch::OBJECT_FULLTEXT_TYPE_KEY.'_'.$lang;
			}
		} else {
			return strtolower($this->getShortClassName()).'_'.$lang;
		}
	}

	/**
	 * @return array base data indexed in each type
	 */
	public function getElasticBaseData()
	{
		return [
			'id' => (int) parent::getId(),
			'parentId' => (int) parent::getParentId(),
			'published' => (bool) parent::getPublished(),
			'key' => parent::getKey(),
			'type' => $this->getShortClassName(),
		];
	}

	/**
	 * updates object in main index and custom type.
	 */
	public function elasticSearchUpdate()
	{
		if (parent::getProperty('elastic_search_exclude')) {
			return;
		}

		$mainIndexName = Tool\ElasticSearch::getMainIndexName();
		$elasticClient = Tool\ElasticSearch::getClient();
		$mapping = $this->getObjectMapping();

		if (parent::getId()) {
			foreach ($mapping['meta']['languages'] as $lang) {
				$data = $this->elasticSearchLoad($lang);
				if (isset($data['elasticGroup'])) {
					$this->elasticSearchDelete($lang, current($data['elasticGroup'])['id']);
					foreach ($data['elasticGroup'] as $key => $groupData) {
						$elasticClient->index([
							'index' => $mainIndexName,
							'type' => $this->getElasticType($lang),
							'id' => $groupData['id'].'-'.$key,
							'body' => $groupData,
						]);
					}
				} else {
					$elasticClient->index([
						'index' => $mainIndexName,
						'type' => $this->getElasticType($lang),
						'id' => $data['id'],
						'body' => $data,
					]);
				}
				unset($data);

				if (!Tool\ElasticSearch::isUpdateRequest()) {
					$elasticClient->indices()->refresh(['index' => $mainIndexName]);
				}
			}
		}
	}

	/**
	 * removes object from main index and custom type.
	 *
	 * @param string $language
	 * @param string $groupId
	 */
	public function elasticSearchDelete($language = null, $groupId = null)
	{
		$mainIndexName = Tool\ElasticSearch::getMainIndexName();
		$elasticClient = Tool\ElasticSearch::getClient();
		$mapping = $this->getObjectMapping();

		foreach ($mapping['meta']['languages'] as $lang) {
			if ($language && $language != $lang) {
				continue;
			}
			try {
				if ($groupId) {
					$elasticClient->deleteByQuery([
						'index' => $mainIndexName,
						'type' => $this->getElasticType($lang),
						'body' => ['query' => ['term' => ['id' => $groupId]]],
					]);
				} else {
					$elasticClient->delete([
						'index' => $mainIndexName,
						'type' => $this->getElasticType($lang),
						'id' => parent::getId(),
					]);
				}
			} catch (\Exception $e) {
				Log\Simple::log('elasticsearch',
					'Could not delete entry for object with ID: '
					.parent::getId().' and type: '.$this->getElasticType($lang)
					.'. Exception: '.$e->getMessage()."\n".$e->getTraceAsString()
				);
			}
		}
		if (!Tool\ElasticSearch::isUpdateRequest()) {
			$elasticClient->indices()->refresh(['index' => $mainIndexName]);
		}
	}

	/**
	 * updates document or object in main index and fulltext type.
	 */
	public function elasticSearchUpdateFulltext()
	{
		if (parent::getProperty('elastic_search_exclude')) {
			return;
		}

		$indexName = Tool\ElasticSearch::getMainIndexName();
		$elasticClient = Tool\ElasticSearch::getClient();

		$languages = null;
		if ($this instanceof Document\Page) {
			$language = $this->getProperty('language');
			if (!$language) {
				$language = Utils::getDefaultLanguage();
			}
			$languages[] = $language;
		} else {
			$mapping = $this->getObjectMapping();
			$languages = $mapping['meta']['languages'];
		}

		foreach ($languages as $lang) {
			$data = $this->elasticSearchLoadFulltext($lang);
			$elasticClient->index([
				'index' => $indexName,
				'type' => $this->getElasticType($lang, true),
				'id' => $data['id'],
				'body' => $data,
			]);
			unset($data);

			if (!Tool\ElasticSearch::isUpdateRequest()) {
				$elasticClient->indices()->refresh(['index' => $indexName]);
			}
		}
	}

	/**
	 * removes document or object from main index and fulltext type.
	 */
	public function elasticSearchDeleteFulltext()
	{
		$mainIndexName = Tool\ElasticSearch::getMainIndexName();
		$elasticClient = Tool\ElasticSearch::getClient();

		$languages = null;
		if ($this instanceof Document\Page) {
			$language = $this->getProperty('language');
			if (!$language) {
				$language = Utils::getDefaultLanguage();
			}
			$languages[] = $language;
		} else {
			$mapping = $this->getObjectMapping();
			$languages = $mapping['meta']['languages'];
		}

		foreach ($languages as $lang) {
			try {
				$elasticClient->delete([
					'index' => $mainIndexName,
					'type' => $this->getElasticType($lang, true),
					'id' => parent::getId(),
				]);
			} catch (\Exception $e) {
				Log\Simple::log('elasticsearch',
					'Could not delete entry for document or object with ID: '
					.parent::getId().' and type: '.$this->getElasticType($lang, true)
					.'. Exception: '.$e->getMessage()."\n".$e->getTraceAsString()
				);
			}
		}
		if (!Tool\ElasticSearch::isUpdateRequest()) {
			$elasticClient->indices()->refresh(['index' => $mainIndexName]);
		}
	}

	/**
	 * decomposes a string into parts for better results of prefix suggestions.
	 *
	 * @param string $string
	 * @param mixed  $payload
	 *
	 * @return array
	 */
	public function createElasticSuggestion($string, $payload = null)
	{
		$tmp = Utils::webalize($string);

		$suggest = ['input' => explode('-', $tmp), 'output' => $string, 'weight' => 1];
		if ($payload) {
			$suggest['payload'] = $payload;
		}

		return $suggest;
	}

	/**
	 * @param string $text
	 *
	 * @return string
	 */
	public function normalizeElasticText($text)
	{
		$normalizedText = iconv('UTF-8', 'UTF-8//IGNORE', mb_convert_encoding($text, 'UTF-8', 'UTF-8'));
		$dom = HtmlDomParser::str_get_html(
			'<div>'.html_entity_decode($normalizedText).'</div>'
		);
		if ($dom) {
			$tagList = [
				'audio', 'area', 'base', 'canvas', 'embed', 'head', 'link', 'map', 'meta',
				'noscript', 'object', 'param', 'script', 'source', 'style', 'svg', 'track', 'video',
			];
			foreach ($tagList as $tag) {
				foreach ($dom->find($tag) as $node) {
					$node->outertext = '';
				}
			}

			$normalizedText = $dom->save();
		}

		return preg_replace('/[ \t\r\n]+/', ' ', strip_tags($normalizedText));
	}

	/**
	 * TODO we could maybe render in a sub-request, similar to Document\Service::render()
	 *      it would be slower, but passing params would not be needed.
	 *
	 * @param string $template
	 * @param string $language
	 * @param array  $params
	 *
	 * @return string
	 */
	public function renderView($template, $language, $params)
	{
		/* @var $templatingEngine Templating\PhpEngine */
		$templatingEngine = \Pimcore::getContainer()->get('templating');

		/* @var $translator Translator */
		$translator = \Pimcore::getContainer()->get('translator');
		$translator->setLocale($language);

		// disable layout
		$params[Templating\PhpEngine::PARAM_NO_PARENT] = true;

		$content = $this->normalizeElasticText($templatingEngine->render($template, $params));

		return $content;
	}
}
