<?php

namespace AppBundle\Service;

use AppBundle\Tool;
use AppBundle\Tool\Utils;
use Symfony\Component\HttpFoundation\RequestStack;

class ElasticSearch
{
	/**
	 * @var ElasticSearch[]
	 */
	private static $instances;
	/**
	 * @var \Elasticsearch\Client
	 */
	protected $elasticClient = null;
	/** @var string */
	protected $index = null;
	/** @var string */
	protected $language = null;

	/**
	 * @param string $language
	 *
	 * @return ElasticSearch
	 */
	public static function getInstance($language)
	{
		$key = $language;
		if (empty(self::$instances[$key])) {
			self::$instances[$key] = new self(null, $language);
		}

		return self::$instances[$key];
	}

	/**
	 * @param RequestStack $requestStack
	 * @param string       $language
	 *
	 * @throws \Exception
	 */
	public function __construct(RequestStack $requestStack = null, $language = null)
	{
		// try to load language from $requestStack (injected), or $language (passed directly)
		if (null !== $requestStack) {
			$this->language = substr($requestStack->getCurrentRequest()->getLocale(), 0, 2);
		} elseif (null !== $language) {
			$this->language = $language;
		}
		// fallback to default language
		if (!$this->language) {
			$this->language = Utils::getDefaultLanguage();
		}

		$this->elasticClient = Tool\ElasticSearch::getClient();
		$this->index = Tool\ElasticSearch::getMainIndexName(false);

		if (!$this->elasticClient->indices()->existsAlias(['name' => $this->index])) {
			throw new \Exception('Elastic search main index not found.');
		}
	}

	/**
	 * @param array $filter
	 *
	 * @return \stdClass
	 */
	public function fulltextSearch($filter = [])
	{
		// query base
		$baseOptions = ['type' => [
			Tool\ElasticSearch::DOCUMENT_FULLTEXT_TYPE_KEY.'_'.$this->language,
			Tool\ElasticSearch::OBJECT_FULLTEXT_TYPE_KEY.'_'.$this->language,
		]];
		if (!empty($filter['query'])) {
			$baseOptions['query']['multi_match']['query'] = $filter['query'];
			$baseOptions['query']['multi_match']['fields'] = ['title^4', 'keywords^3', 'description^2', 'content'];
		}
		$filteredQuery = $this->queryBase($baseOptions, $filter);

		$aggregations = [];
		$aggregations['type'] = $this->fulltextTypeAggregation($filteredQuery);

		if (!empty($filter['type'])) {
			$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['type'] = $filter['type'];
		}

		$pagination = $this->query($filteredQuery, $filter, true, true, true);
		$pagination->aggregations = $aggregations;

		return $pagination;
	}

	/**
	 * @param array $query
	 *
	 * @return array
	 */
	private function fulltextTypeAggregation($query)
	{
		$aggQuery = [];
		$aggQuery['type']['terms']['size'] = 100;
		$aggQuery['type']['terms']['field'] = 'type';

		$query['body']['aggs'] = $aggQuery;
		$result = $this->elasticClient->search($query);

		$types = [];
		foreach ($result['aggregations']['type']['buckets'] as $bucket) {
			$types[$bucket['key']] = $bucket['doc_count'];
		}

		return $types;
	}

	/**
	 * @param array $options
	 * @param array $filter
	 *
	 * @return array
	 */
	private function queryBase($options = [], $filter = [])
	{
		$filteredQuery['index'] = $this->index;
		$filteredQuery['type'] = $options['type'];

		// set query for filtered query
		$query = (!empty($options['query'])) ? $options['query'] : null;
		if (is_array($query)) {
			$filteredQuery['body']['query']['filtered']['query'] = $query;
		} elseif ('random_score' == $query) {
			$filteredQuery['body']['query']['filtered']['query']['function_score']['functions'][0]['random_score'] = new \stdClass();
		} elseif (!empty($filter['ids_score']) && !empty($filter['ids'])) {
			$functionScore = [
				'boost_mode' => 'replace',
				'script_score' => [
					'script' => "count = ids.size(); return -ids.indexOf(Integer.parseInt(doc['id'].value));",
					'params' => [
						'ids' => $filter['ids'],
					],
				],
			];
			$filteredQuery['body']['query']['filtered']['query']['function_score'] = $functionScore;
		} else {
			$filteredQuery['body']['query']['filtered']['query']['match_all'] = [];
		}

		// add filters
		$filteredQuery['body']['query']['filtered']['filter']['bool']['must'][]['term']['published'] = true;

		return $filteredQuery;
	}

	/**
	 * @param array $resultSet
	 * @param int   $limit
	 * @param int   $page
	 * @param int   $totalCount
	 *
	 * @return \stdClass
	 */
	private function constructSimplePagination($resultSet, $limit, $page, $totalCount)
	{
		$pagination = new \stdClass();
		$pagination->current = $page;
		$pagination->pageCount = ($limit) ? (int) ceil($totalCount / $limit) : 1;
		if ($page > 1) {
			$pagination->previous = $page - 1;
		}
		if ($page < $pagination->pageCount) {
			$pagination->next = $page + 1;
		}
		$pagination->totalCount = $totalCount;
		$pagination->first = 1;
		$pagination->last = $pagination->pageCount;
		$pagination->items = empty($resultSet) ? [] : $resultSet['hits']['hits'];

		return $pagination;
	}

	/**
	 * @param array $filteredQuery
	 * @param array $filter
	 * @param bool  $foldedFallback
	 * @param bool  $paginate
	 * @param bool  $highlighting
	 *
	 * @return array|\stdClass
	 */
	private function query($filteredQuery, $filter, $foldedFallback = true, $paginate = true, $highlighting = false)
	{
		//add folded fallback fields
		$fulltextFields = [];
		if ($foldedFallback && !empty($filter['query'])) {
			$fulltextFields = $filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'];
			$fields = [];
			foreach ($fulltextFields as $field) {
				$parts = explode('^', $field);
				$foldedField = $parts[0].'.folded';
				if (!empty($parts[1])) {
					$foldedField .= '^'.$parts[1];
				}
				$fields[] = $field;
				$fields[] = $foldedField;
			}
			$filteredQuery['body']['query']['filtered']['query']['multi_match']['fields'] = $fields;
		}

		//get total count
		$countQuery = ['index' => $filteredQuery['index'], 'type' => $filteredQuery['type'], 'body' => ['query' => $filteredQuery['body']['query']]];
		$countResult = $this->elasticClient->count($countQuery);
		$totalCount = $countResult['count'];

		//sort, only when no searchQuery is set and some results matched and some sort is defined...
		if ($totalCount && empty($filter['query']) && !empty($filter['sort'])
		&& empty($filter['ids_score'])) {
			$sortStruct = $filter['sort'];
			if (is_array($sortStruct)) {
				foreach ($sortStruct as $key => $sort) {
					$filteredQuery['body']['sort'][$key] = $sort;
				}
			} elseif (stristr($sortStruct, ':')) {
				//$gps = explode(':', $sortStruct);
				$filteredQuery['body']['sort']['_geo_distance'] = [
					'location' => str_replace(':', ',', $sortStruct),
					'order' => 'asc',
					'unit' => 'km',
					'distance_type' => 'plane',
				];
			} else {
				//TODO define sorting types
			}
		}

		//add highlighting
		if ($highlighting && $fulltextFields && !empty($filter['query'])) {
			$highlight = ['fields' => []];
			foreach ($fulltextFields as $field) {
				$parts = explode('^', $field);
				$highlight['fields'][$parts[0]] = [
					'matched_fields' => [$parts[0]],
					'type' => 'fvh',
				];
				if ($foldedFallback) {
					$highlight['fields'][$parts[0]]['matched_fields'][] = $parts[0].'.folded';
				}
			}
			$filteredQuery['body']['highlight'] = $highlight;
		}

		//pagination
		$limit = !empty($filter['limit']) ? intval($filter['limit']) : 999;
		if ($paginate) {
			$page = !empty($filter['page']) ? intval($filter['page']) : 1;
			if ((int) $page > ceil($totalCount / $limit)) {
				$page = ceil($totalCount / $limit);
			}
			if ($page < 1) {
				$page = 1;
			}
			$filteredQuery['body']['from'] = isset($filter['offset']) ? ($page - 1) * $limit : 0;
			$filteredQuery['body']['size'] = isset($filter['offset']) ? $limit : $page * $limit;
		} else {
			$filteredQuery['body']['size'] = $limit;
		}
		//query
		$resultSet = $this->elasticClient->search($filteredQuery);
		if ($paginate) {
			return $this->constructSimplePagination($resultSet, $limit, $page, $totalCount);
		} else {
			return $resultSet;
		}
	}
}
