<?php

namespace AppBundle\Service;

use AppBundle\Tool\Utils;
use Pimcore\File;
use Pimcore\Logger;
use Pimcore\Model\DataObject\NewsletterForm;
use Pimcore\Translation\Translator;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

class NewsletterManager
{
	const ROOT_FOLDER_ID = 6;

	/**
	 * @var Translator
	 */
	public $translator;
	/**
	 * @var MailManager
	 */
	protected $mailManager;
	/**
	 * @var Request
	 */
	protected $request;

	/**
	 * @param Translator   $translator
	 * @param MailManager  $mailManager
	 * @param RequestStack $requestStack
	 */
	public function __construct(Translator $translator, MailManager $mailManager, RequestStack $requestStack)
	{
		$this->translator = $translator;
		$this->mailManager = $mailManager;
		$this->request = $requestStack->getCurrentRequest();
	}

	/**
	 * @param array $data
	 *
	 * @return bool
	 */
	public function subscribe($data)
	{
		$data['unsubscribeCode'] = uniqid();

		//check if exists
		if (count(NewsletterForm::getByEmail($data['email']))) {
			return true; //no error msg needed
		}

		//create
		$newsletter = NewsletterForm::create($data);
		$newsletter->setCreationDate(time());
		$newsletter->setKey(File::getValidFilename($data['email']));
		$newsletter->setParentId(self::ROOT_FOLDER_ID);
		$newsletter->setPublished(true);
		//crm stuff (not used)
		$newsletter->setFirstname('');
		$newsletter->setLastname('');
		$newsletter->setGender('');
		$newsletter->setNewsletterActive(true);
		$newsletter->setNewsletterConfirmed(true);
		try {
			$newsletter->save();

			$this->mailManager->sendMail('/notifikace/newsletter', $data['lang'], $data['email'], null, [
				'unsubscribeUrl' => $this->request->getSchemeAndHttpHost().
					Utils::url('newsletter-unsubscribe', ['language' => $data['lang'], 'code' => $data['unsubscribeCode']]),
			]);
		} catch (\Exception $e) {
			Logger::error('NEWSLETTER SUBSCRIBE - '.$e->getMessage()."\n".$e->getTraceAsString());

			return false;
		}

		return true;
	}

	/**
	 * @param string $code
	 *
	 * @return bool
	 */
	public function unsubscribe($code)
	{
		$object = NewsletterForm::getByUnsubscribeCode($code)->current();
		if ($object) {
			try {
				$email = $object->getEmail();
				$object->delete();

				return $email;
			} catch (\Exception $e) {
				Logger::error('NEWSLETTER UNSUBSCRIBE - '.$e->getMessage()."\n".$e->getTraceAsString());

				return false;
			}
		} else {
			return false;
		}
	}
}
