<?php

namespace App\Controller;

use App\Form\ContactFormType;
use App\Model\News;
use App\Service\MailManager;
use Knp\Component\Pager\PaginatorInterface;
use Pimcore\File;
use Pimcore\Logger;
use Pimcore\Model\DataObject;
use Pimcore\Model\DataObject\ContactForm;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Form\FormInterface;

class CmsController extends BaseController
{
	/*						DOCUMENT ROUTED ACTIONS							*/

	/**
	 * @return Response
	 */
	public function templateSimpleAction()
	{
		return $this->render('cms/templateSimple.html.twig');
	}

	/**
	 * @return Response
	 */
	public function simpleMailAction()
	{
		return $this->render('cms/simpleMail.html.twig');
	}

	/**
	 * @return Response
	 */
	public function sitemapAction()
	{
		return $this->render('cms/sitemap.html.twig');
	}

	/**
	 * @param Request     $request
	 * @param MailManager $mailManager
	 *
	 * @return Response
	 */
	public function contactUsAction(Request $request, MailManager $mailManager)
	{
		$form = $this->createForm(ContactFormType::class);
		$form->handleRequest($request);

		if ($form->isSubmitted()) {
			if ($this->handleContactForm($form, $mailManager, ContactFormType::FIELD_HONEYPOT)) {
				return $this->gotoUrl($this->docUrl('/kontakt'));
			}
		}

		return $this->render('cms/contactUs.html.twig', [
			'contactForm' => $form->createView(),
		]);
	}

	/**
	 * @param Request $request
	 * @param PaginatorInterface $paginator
	 *
	 * @return Response
	 */
	public function newsAction(Request $request, PaginatorInterface $paginator)
	{
		$page = $request->get('page', 1);

		$list = new News\Listing();
		$list->setOrder('desc');
		$list->setOrderKey('createdOn');
		$list->setCondition("createdOn < ? AND name IS NOT NULL AND name <> ''", [time()]);

		$paginator = $paginator->paginate(
			$list,
			$page,
			6
		);

		$paginationData = $paginator->getPaginationData();
		$paginationData['urlprefix'] = $this->docUrl($this->document->getFullPath()) . '?page=';

		return $this->render('cms/news.html.twig', [
			'news' => $paginator,
			'paginationData' => $paginationData
		]);
	}

	/* 						STATIC-ROUTE ROUTED ACTIONS						 */

	/**
	 * @param string $id
	 *
	 * @return Response
	 */
	public function newsDetailAction($id)
	{
		//output cached stuff
		if (!$this->isOutputCached($this->cacheKeySuffix)) {
			$news = News::getById($id);

			if (!$news || !$news->isPublished()) {
				throw $this->createNotFoundException($this->translate('system_page_not_found'));
			}

			//get sibblings (next and previous news)
			$list = new News\Listing();
			$list->setOrder('desc');
			$list->setOrderKey('createdOn');
			$list->setCondition("createdOn < ? AND name IS NOT NULL AND name <> ''", [time()]);
			$list->load();
			list($previousNews, $nextNews) = $this->getSiblings($list, $news, 'createdOn');

			//extend breadcrumbs
			$this->addToBreadcrumbs[] = [
				'label' => $news->getName(),
				'url' => $news->getUrl(),
			];

			//explicit language switch
			$this->languageSwitch = $this->buildLanguageSwitchFromObject($news);

			//explicit seo stuff
			$this->headTitle->set($news->getName());
			if ($news->getSeoDescription()) {
				$this->headMeta->setName('description', $news->getSeoDescription());
			}

			return $this->render('cms/newsDetail.html.twig', [
				'news' => $news,
				'previousNews' => $previousNews,
				'nextNews' => $nextNews
			]);
		}
	}

	/*				VIEW-LESS ACTIONS (ALWAYS REDIRECT)					*/

	/*							FORM HANDLERS								*/

	/**
	 * @param FormInterface $form
	 * @param MailManager   $mailManager
	 * @param string $honeypot
	 *
	 * @return bool
	 */
	private function handleContactForm(FormInterface $form, MailManager $mailManager, string $honeypot)
	{
		if ($form->isValid()) {
			$data = $form->getData();

			// test honeypot field to not be empty
			if($honeypot && !empty($data[$honeypot])) {
				$this->addErrorMsg($this->translate('msg_form_invalid'), true);
				return false;
			}

			$data['lang'] = $this->language;

			$contactForm = ContactForm::create($data);
			$contactForm->setCreationDate(time());
			$key = File::getValidFilename($data['fullName'].'_'.date('Y-m-d-H:i:s', time()));
			$contactForm->setKey($key);
			$contactForm->setParentId(DataObject\Folder::getByPath('/kontakt/kontaktni-formulare')->getId());
			$contactForm->setPublished(true);

			try {
				$contactForm->save();

				//send email to admin
				$mailManager->sendMail('/notifikace/pro-administratora-novy-kontaktni-formular', $this->language, null, null, [
					'deeplink' => $this->getRequest()->getSchemeAndHttpHost().'/admin/login/deeplink?object_'.$contactForm->getId().'_object',
					'form' => $contactForm,
				]);

				$this->addSuccessMsg($this->translate('msg_contact_form_success'));

				return true;
			} catch (\Exception $e) {
				Logger::error('CONTACT FORM - '.$e->getMessage()."\n".$e->getTraceAsString());
				$this->addErrorMsg($this->translate('msg_unknown_error'));
			}
		} else {
			$this->addErrorMsg($this->translate('msg_form_invalid'));
		}

		return false;
	}
}
