<?php

namespace App\Twig\Extension;

use App\Model\Document\Page;
use App\Tool\Utils;
use Pimcore\Model\DataObject;
use Pimcore\Model\DataObject\Fieldcollection;
use Symfony\Component\HttpFoundation\RequestStack;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

class OpenGraphExtension  extends AbstractExtension
{
	protected RequestStack $requestStack;

	protected ?string $host;
	protected ?string $currentUrl;

	public function __construct(RequestStack $requestStack)
    {
		$this->requestStack = $requestStack;

		$request = $requestStack?->getCurrentRequest();
		$this->host = $request?->getSchemeAndHttpHost();
		$this->currentUri = strtok($request?->getRequestUri(), '?');
    }

	/**
	 * @return array|TwigFunction[]
	 */
	public function getFunctions()
	{
		return [
			new TwigFunction('get_page_og_meta_tags', [$this, 'getPageOGMeta']),
			new TwigFunction('get_data_object_og_meta_tags', [$this, 'getDataObjectOGMeta']),
		];
	}

	public function getPageOGMeta(Page $page, string $language): array
	{
		$ogTags = [];
		$ogTags['twitter:card'] = 'summary_large_image';

		// title
		$fallbackTitle = $page->getProperty('navigation_title') ?: $page->getProperty('titlePostfix');
		$title = $page->getTitle() ?: $page->getProperty('OGTitlePlaceholder') ?: $fallbackTitle;

		if ($title) {
			$ogTags['og:title'] = $title;
			$ogTags['twitter:title'] = $title;
		}

		// description
		$description = $page->getDescription() ?: $page->getProperty('OGDescriptionPlaceholder') ?: $fallbackTitle;
		if ($description) {
			$description = Utils::cutStringRespectingWhitespace(strip_tags($description), 300);
			$ogTags['og:description'] = $description;
			$ogTags['twitter:description'] = $description;
		}

		// image
		$image = $page->getEditable('article-image')?->getImage();
		if (!$image) {
			$image = $page->getProperty('OGImagePlaceholder');
		}
		if ($image) {
			$thumbnailPath = Utils::thumbnail($image, 'OpenGraphImage');
			$twitterThumbnailPath = Utils::thumbnail($image, 'TwitterImage');
			$ogTags['og:image'] = $this->host . $thumbnailPath;
			$ogTags['twitter:image'] = $this->host . $twitterThumbnailPath;
			$ogTags['og:image:width'] = 1200;
			$ogTags['og:image:height'] = 630;
		}

		// url
		$ogTags['og:url'] = $this->host . $this->currentUri;

		return $ogTags;
	}

	public function getDataObjectOGMeta(DataObject $object, string $language): array
	{
		$ogTags = [];
		$ogTags['twitter:card'] = 'summary_large_image';

		// image
		$image = null;
		if (method_exists($object, 'getPreviewImage')) {
			$image = $object->getPreviewImage();
		}
		elseif (method_exists($object, 'getTeaserImage')) {
			$image = $object->getTeaserImage();
		}
		if (!$image && method_exists($object, 'getContent') && ($content = $object->getContent())) {
			if ($content instanceof \Pimcore\Model\DataObject\Fieldcollection) {
				foreach ($content->getItems() as $block) {
					$type = $block->getType();
					if ($type == 'ImageList') {
						$image = $block->getImage();
						break;
					}
					elseif ($type == 'GallerySlider' && !empty($block->getImages())) {
						$image = current($block->getImages());
						break;
					}
				}
			}
		}

		// image below replace placeholder OG image from document property
		if($image) {
			$thumbnailPath = Utils::thumbnail($image, 'OpenGraphImage');
			$twitterThumbnailPath = Utils::thumbnail($image, 'TwitterImage');
			$ogTags['og:image'] = $this->host . $thumbnailPath;
			$ogTags['twitter:image'] = $this->host . $twitterThumbnailPath;
			$ogTags['og:image:width'] = 1200;
			$ogTags['og:image:height'] = 630;
		}

		// url
		if (method_exists($object, 'getUrl')) {
			$ogTags['og:url'] = $this->host . $object->getUrl($language);
		}

		// title
		if (method_exists($object, 'getName')) {
			$ogTags['og:title'] = $object->getName($language);
			$ogTags['twitter:title'] = $object->getName($language);
		}

		// description
		$description = null;
		$descriptionElements = [
			'getPerex',
			'getPreviewDescription',
			'getDescription',
			'getContent'
		];

		foreach ($descriptionElements as $element) {
			if ($description || !method_exists($object, $element)) continue;
			$content = $object->{$element}($language);

			if (!$content) continue;

			if ($content instanceof Fieldcollection) {
				foreach ($content->getItems() as $block) {
					$type = $block->getType();
					if ($type == 'TextBlock') {
						$description = strip_tags($block->getText($language));
						break;
					}
				}
			}
			else {
				$description = strip_tags($content);
				break;
			}
		}

		if ($description) {
			$description = Utils::cutStringRespectingWhitespace(strip_tags($description), 300);
			$ogTags['og:description'] = $description;
			$ogTags['twitter:description'] = $description;
		}

		return $ogTags;
	}
}
